//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization;

import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.cleversafe.serialization.asn1.ASN1;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.cleversafe.serialization.asn1.custom.Registrator;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class MessagePerformanceComparisonTest
{
   private static Logger _logger = Logger.getLogger(MessagePerformanceComparisonTest.class);

   private static ProtocolMessageFactory pmFactory = null;

   private static SliceName[] sliceNames = null;
   private static DataSlice[] dataSlices = null;
   private static long txid = 1020221182;
   private static int numIterations = 10000;
   private static int numDataSlices = 32;
   private static int numSliceNames = 32;
   private static int SLICE_BLOCK_SIZE = 353;

   private static int GRID_WIDTH = 16;
   private static int GRID_TRESHOLD = 12;

   private static int BLOCK_SIZE = 4096;

   private static String SET_SLICE_NAMES_METHOD = "setSliceNames";
   private static String SET_DATA_SLICES_METHOD = "setDataSlices";
   private static String SET_TRANSACTION_ID_METHOD = "setTransactionID";

   private Class<?>[] frequentProtocolMessages =
         new Class<?>[]{
               MultipleReadRequest.class, MultipleReadResponse.class, MultipleWriteRequest.class,
               MultipleWriteResponse.class, BeginTransactionRequest.class,
               BeginTransactionResponse.class, CommitTransactionRequest.class,
               CommitTransactionResponse.class
         };

   @BeforeClass
   public static void setup()
   {
      numIterations = Integer.getInteger("test.iteration", numIterations);
      numSliceNames = numDataSlices = Integer.getInteger("test.numSlices", numSliceNames);

      pmFactory = new ASN1GridProtocolMessageFactory();
      sliceNames = ProtocolMessageFactoryTestBase.getRandomSliceNames(numSliceNames);
      dataSlices =
            ProtocolMessageFactoryTestBase.getRandomDataSliceArray(numDataSlices, SLICE_BLOCK_SIZE);
   }

   @AfterClass
   public static void tearDown()
   {
      pmFactory = null;
   }

   private void testProtocolMessageSerializationPerformance(
         ProtocolMessage pm,
         int numIterations,
         boolean configure) throws ProtocolLayerException
   {
      String className = pm.getClass().getName();
      int indNameNoPack = className.lastIndexOf('.');
      String classNameNoPack = className.substring(indNameNoPack + 1);
      testProtocolMessageSerializationPerformance(new ProtocolMessage[]{
         pm
      }, classNameNoPack, numIterations, configure);
   }

   DataInputStream serialize(ProtocolMessage message) throws ProtocolSerializationException
   {
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);
      pmFactory.serialize(out, message);
      return new DataInputStream(new ByteArrayInputStream(bytes.toByteArray()));
   }

   private double testProtocolMessageSerializationPerformance(
         ProtocolMessage[] pms,
         String description,
         int numIterations,
         boolean configure) throws ProtocolLayerException
   {
      if (configure)
      {
         for (int i = 0; i < pms.length; i++)
         {
            configureProtocolMessage(pms[i]);
         }
      }

      long startTime = System.currentTimeMillis();
      for (int i = 0; i < numIterations; ++i)
      {
         for (int j = 0; j < pms.length; j++)
         {
            ProtocolMessage pm2 =
                  pmFactory.deserialize(pms[j].getOperation().getOperationCode(), serialize(pms[j]));
         }
      }
      double totalTime = System.currentTimeMillis() - startTime;
      float averageTime = ((float) totalTime) / ((float) numIterations);
      System.out.println(description + ": \t\t" + averageTime + "ms.");
      return averageTime;
   }

   private void configureProtocolMessage(ProtocolMessage pm)
   {
      try
      {
         Method setSliceBuffers =
               pm.getClass().getMethod(SET_DATA_SLICES_METHOD, DataSlice[].class);
         _logger.info("Attempting to set slice buffers for " + pm.getClass());
         setSliceBuffers.invoke(pm, (Object) dataSlices);
      }
      catch (NoSuchMethodException ignore)
      {
      }
      catch (IllegalAccessException e)
      {
         _logger.warn("Failed to set slice buffers!", e);
      }
      catch (InvocationTargetException e)
      {
         _logger.warn("Failed to set slice buffers!", e);
      }

      try
      {
         Method setSourceNames = pm.getClass().getMethod(SET_SLICE_NAMES_METHOD, SliceName[].class);
         _logger.info("Attempting to set source names for " + pm.getClass());
         setSourceNames.invoke(pm, (Object) sliceNames);
      }
      catch (NoSuchMethodException ignore)
      {
      }
      catch (IllegalAccessException e)
      {
         _logger.warn("Failed to set source names!", e);
      }
      catch (InvocationTargetException e)
      {
         _logger.warn("Failed to set source names!", e);
      }

      try
      {
         Method setTransactionID = pm.getClass().getMethod(SET_TRANSACTION_ID_METHOD, long.class);
         _logger.info("Attempting to set transaction id for " + pm.getClass());
         setTransactionID.invoke(pm, txid);
      }
      catch (NoSuchMethodException ignore)
      {
      }
      catch (IllegalAccessException e)
      {
         _logger.warn("Failed to set transaction id!", e);
      }
      catch (InvocationTargetException e)
      {
         _logger.warn("Failed to set transaction id!", e);
      }

   }

   private void testProtocolMessageAllSingle(boolean configure)
   {
      for (Class<?> protocolMessage : frequentProtocolMessages)
      {
         try
         {
            testProtocolMessageSerializationPerformance(
                  (ProtocolMessage) protocolMessage.newInstance(), numIterations, configure);
         }
         catch (Exception e)
         {
            fail(e.getMessage());
         }
      }
   }

   private void testProtocolMessageGroup(boolean configure, String operation, Class<?>... inGroup)
   {
      ProtocolMessage[] objectGroup = new ProtocolMessage[inGroup.length];
      try
      {
         for (int i = 0; i < inGroup.length; i++)
         {
            objectGroup[i] = (ProtocolMessage) inGroup[i].newInstance();
         }
         double average =
               testProtocolMessageSerializationPerformance(objectGroup, operation, numIterations,
                     configure);
         // Calculate throughput for a block
         // (1MB/8)/32*4K - # of chunks for 1Mbit
         int bytesInMbit = 1024 * 1024 / 8;
         double sendsPerMbit = bytesInMbit / (BLOCK_SIZE * numDataSlices);
         double conversionsForMbit = sendsPerMbit * GRID_WIDTH;
         double timePerMbit = conversionsForMbit * (average / 1000);
         System.out.println(String.format("Throughput: %.0f Mbps", 1 / timePerMbit));
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   @Test
   public void testProtocolMessageSerializationPerformance()
   {
      /*
       * System.out.println("Empty protocol messages with custom serialization: ");
       * testProtocolMessageAllSingle(false); System.out.println();
       */
      System.out.println("Protocol messages with data and custom serialization: ");
      testProtocolMessageAllSingle(true);
      System.out.println();

      System.out.println("Protocol read group with custom serialization: ");
      testProtocolMessageGroup(true, "Read roundtrip", MultipleReadRequest.class,
            MultipleReadResponse.class);
      System.out.println();

      System.out.println("Protocol write group with custom serialization: ");
      testProtocolMessageGroup(true, "Write roundtrip", BeginTransactionRequest.class,
            BeginTransactionResponse.class, MultipleWriteRequest.class,
            MultipleWriteResponse.class, CommitTransactionRequest.class,
            CommitTransactionResponse.class);
      System.out.println();

      // Unregister custom serialization so reflection is used automatically
      for (Class<?> protocolMessage : frequentProtocolMessages)
      {
         ASN1.DERConverter.unregisterDefaultEncodeDecode(protocolMessage);
      }
      /*
       * System.out.println("Empty protocol messages with reflection: ");
       * testProtocolMessageAllSingle(false); System.out.println();
       */
      System.out.println("Protocol messages with data and  with reflection: ");
      testProtocolMessageAllSingle(true);
      System.out.println();

      System.out.println("Protocol read group with reflection: ");
      testProtocolMessageGroup(true, "Read roundtrip", MultipleReadRequest.class,
            MultipleReadResponse.class);
      System.out.println();

      System.out.println("Protocol write group with reflection: ");
      testProtocolMessageGroup(true, "Write roundtrip", BeginTransactionRequest.class,
            BeginTransactionResponse.class, MultipleWriteRequest.class,
            MultipleWriteResponse.class, CommitTransactionRequest.class,
            CommitTransactionResponse.class);

      // Re-register when done
      Registrator.registorCustomObjects();
   }
}
