//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: spalthepu
//
// Date: Sep 26, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.security.SecureRandom;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Logger;
import org.cleversafe.PerformanceResults;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultPermission;
import org.junit.BeforeClass;
import org.junit.Test;

public class BlockFileSliceStorePerformanceTest
{
   private static Logger _logger = Logger.getLogger(BlockFileSliceStorePerformanceTest.class);

 //  private static final String TEST_PATH =  "test-data" ;
 //  private static final String TEST_OUTPUT_PATH = TEST_PATH + "/output/";
 //  private static final String TEST_INPUT_PATH = TEST_PATH + "/input/";

   public static String SERVER_CONFIGURATION_PROPERTY =
      "org.cleversafe.storage.ss.xml.configuration";
 
   // I/O Request Size in KB
   private static int requestSize;
   // Number of concurrent requests
   private static int numConcurrentRequests;
   
   public  byte[] data;
   public static SliceName sliceNames[] = null;

   // how many operations to be performed. This is mutually exclusive with testDuration.
   // Only one of them is used.
   private static int testIterations;
   private static final int txnID = 1;
   
   private static UUID accountUUID;
//         private SliceServerApplication sliceServer = null;

   private static ClientSession session = null;
   public static SliceStore sliceStore = null;
   public static UUID vaultIdentifier;
   
   @BeforeClass
   public static void initializeTests()
   {
         

      // Only one of testIterations or testDuration is used.
      testIterations = Integer.getInteger("test.operations", 1000); //
      numConcurrentRequests = Integer.getInteger("test.concurrency", 10); //
      requestSize = Integer.getInteger("test.requestsize", 4); // 4K
      
      _logger.info("Deleting existing store.");
      File f =  new File("slice-data");
      deleteDirectory(f);
      if (f.exists()) {
         fail("Failed to delete the dir " + "slice-data");
      }
      // Setup basic logging
      // generate slice names in advance.
      sliceNames = new SliceName[testIterations];
      for (int sliceId = 0; sliceId < testIterations; sliceId++){
         sliceNames[sliceId] = new SliceName(Integer.toString(sliceId), 0);   
      }
      
      BasicConfigurator.configure();
   }
   
   @Test
   public void writePerformance()
   {
      data = new byte[requestSize * 1024];
      randomizeBuffer(data);
      class WorkItem implements Runnable
      {
         int sliceNum;
         public WorkItem(int slice)
         {
            sliceNum = slice;
         }
         public void run()
         {
            BlockFileSliceStorePerformanceTest test = BlockFileSliceStorePerformanceTest.this;;
            try {
               // Write to the Store.
               sliceStore.write( new DataSlice(sliceNames[sliceNum], txnID, test.data));
            } catch (Exception ex) {
                ex.printStackTrace();
                fail("SliceServer returned an inproper Response type");
            }
         }
      }
   
      
      ExecutorService executor = new BoundedThreadPoolExecutor("Write Store Test", numConcurrentRequests);

      try {
         // Create a session with a valid SliceStore
         session = createSession();
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("Unable to start an operational slice store");
      }
      long beginTest, endTest;
      beginTest = System.currentTimeMillis();
      
      SliceStoreTransaction trans = null; 
      try {
         trans = sliceStore.createTransaction(txnID); // Create transaction.
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("createTransaction failed.");
      }

      try {
         sliceStore.beginTransaction(trans); // begin Txn
            
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("beginTransaction failed.");
      }
   
      // Start Operations.
      int operations;
      
      for (operations = 1; operations < testIterations; operations++) {
         // execute in an available thread.
         executor.execute( new WorkItem(operations) );
      }
      
      executor.shutdown();
      _logger.info("Waiting for threads to stop.");
      boolean finished = false;
      do
      {
         try
         {
            finished = executor.awaitTermination( 10, TimeUnit.SECONDS );
         } catch (InterruptedException e)
         {
            e.printStackTrace();
         }
      } while ( !finished );
   
      // Commit Transaction.
      try {
         // Commit Transaction
         sliceStore.commitTransaction(trans);
      } catch (Exception ex) {
         ex.printStackTrace();
      }
  
      endTest = System.currentTimeMillis();
      
      PerformanceResults results =  new PerformanceResults("WriteToStore", testIterations, requestSize, beginTest, endTest, numConcurrentRequests);
      results.print(true);
   }
   
   @Test
   public void readPerformance()
   {
      
      class WorkItem implements Runnable
      {
         int sliceNum;
         public WorkItem(int slice)
         {
            sliceNum = slice;
        }
         public void run()
         {
            try {
               //Read from the Store. Ignore the return value (DataSlice)
              sliceStore.read( sliceNames[sliceNum]); 
            } catch (Exception ex) {
                ex.printStackTrace();
                fail("SliceStore Read failed.");
            }
         }
      }
   
      
      ExecutorService executor = new BoundedThreadPoolExecutor("Read Test", numConcurrentRequests);

      // Start Operations.
      int operations;
      long beginTest, endTest;
      beginTest = System.currentTimeMillis();
     
      SliceStoreTransaction trans = null; 
      try {
         trans = sliceStore.createTransaction(txnID+1); // Create transaction.
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("createTransaction failed.");
      }

      try {
         sliceStore.beginTransaction(trans); // begin Txn
            
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("beginTransaction failed.");
      }
   
   
      for (operations = 1; operations < testIterations; operations++) {
         // execute in an available thread.
         executor.execute( new WorkItem(operations) );
      }
      
      executor.shutdown();
      _logger.info("Waiting for threads to stop.");
      boolean finished = false;
      do
      {
         try
         {
            finished = executor.awaitTermination( 10, TimeUnit.SECONDS );
         } catch (InterruptedException e)
         {
            e.printStackTrace();
         }
      } while ( !finished );
   
      // Commit Transaction.
      try {
         // Commit Transaction
         sliceStore.commitTransaction(trans);
      } catch (Exception ex) {
         ex.printStackTrace();
      }
      endTest = System.currentTimeMillis();
      
      PerformanceResults results =  new PerformanceResults("ReadFromStore", testIterations, requestSize, beginTest, endTest, numConcurrentRequests);
      results.print(false);
   }
  
   
   @Test
   public void listingPerformance()
   {
      
      // Start Operations.
      long beginTest, endTest;
      beginTest = System.currentTimeMillis();
     
      SliceStoreTransaction trans = null; 
      try {
         trans = sliceStore.createTransaction(txnID+1); // Create transaction.
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("createTransaction failed.");
      }

      try {
         sliceStore.beginTransaction(trans); // begin Txn
            
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("beginTransaction failed.");
      }
   
      try {
         sliceStore.listBegin();
      } catch (Exception e) {
         e.printStackTrace();
         fail("excpetion");
      }
      int operations=0;
      int slices =0;
       
      try {
         while(sliceStore.listInProgress())
         {
            List<SliceInfo> sinf = sliceStore.listContinue();
            slices += sinf.size();
            operations++;
         }
      } catch (Exception e) {
         e.printStackTrace();
         fail("excpetion");
      }
      
      // Commit Transaction.
      try {
         // Commit Transaction
         sliceStore.commitTransaction(trans);
      } catch (Exception ex) {
         ex.printStackTrace();
      }
      endTest = System.currentTimeMillis();
      
      _logger.info("Total Slices " + slices + " operations " + operations);
      PerformanceResults results =  new PerformanceResults("Listing", testIterations, requestSize, beginTest, endTest, numConcurrentRequests);
      results.print(false);
   }

   @Test
   public void removePerformance()
   {
      
      class WorkItem implements Runnable
      {
         int sliceNum;
         public WorkItem(int slice)
         {
            sliceNum = slice;
         }
         public void run()
         {
            try {
               //Read from the Store. Ignore the return value (DataSlice)
              sliceStore.remove( sliceNames[sliceNum]); 
            } catch (Exception ex) {
                ex.printStackTrace();
                fail("SliceStore Read failed.");
            }
         }
      }
   
      
      ExecutorService executor = new BoundedThreadPoolExecutor("Remove Test", numConcurrentRequests);

      // Start Operations.
      int operations;
      long beginTest, endTest;
      beginTest = System.currentTimeMillis();
     
      SliceStoreTransaction trans = null; 
      try {
         trans = sliceStore.createTransaction(txnID+1); // Create transaction.
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("createTransaction failed.");
      }

      try {
         sliceStore.beginTransaction(trans); // begin Txn
            
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("beginTransaction failed.");
      }
   
   
      for (operations = 1; operations < testIterations; operations++) {
         // execute in an available thread.
         executor.execute( new WorkItem(operations) );
      }
      
      executor.shutdown();
      _logger.info("Waiting for threads to stop.");
      boolean finished = false;
      do
      {
         try
         {
            finished = executor.awaitTermination( 10, TimeUnit.SECONDS );
         } catch (InterruptedException e)
         {
            e.printStackTrace();
         }
      } while ( !finished );
   
      // Commit Transaction.
      try {
         // Commit Transaction
         sliceStore.commitTransaction(trans);
      } catch (Exception ex) {
         ex.printStackTrace();
      }
      endTest = System.currentTimeMillis();
      
      PerformanceResults results =  new PerformanceResults("RemoveFromStore", testIterations, requestSize, beginTest, endTest, numConcurrentRequests);
      results.print(false);
   }
   
  
   /**
    * Attempts to create a SliceStore in the session and checks that it is
    * operational
    * 
    */
   private ClientSession createSession() throws Exception {
      
      System.setProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY,
            "conf/config.xml");
      System.setProperty(SERVER_CONFIGURATION_PROPERTY, "conf/slice-server.xml");

      // createConnection(true); // Authenticate

      VaultACL vaultACL = createACL();

      // Setup the session appropriately
      ClientSession session = new ClientSession();
      
      session.put(ClientSession.GRID_ACCOUNT_UUID, accountUUID);
      session.put(ClientSession.AUTHENTICATED, true);
      session.put(ClientSession.APPLICATION_TYPE,
            ApplicationType.TYPE_AUTHENTICATED);

      // Create the Store
      File sliceDataDirectory = new File("slice-data/data");
      sliceDataDirectory.mkdirs();
      
     // SliceStore store = new BDBSliceStore(sliceDataDirectory);
      sliceStore = new BlockFileSliceStore(vaultIdentifier, sliceDataDirectory.getPath());
    
      sliceStore.createStore("block", 81920, SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED, vaultACL, null);

      sliceStore.startSession();
      
      // Save store in the session
      session.put(ClientSession.SLICE_STORE, sliceStore);
      
      // Create permission object with full permissions
      UUID vaultIdentifier = UUID.randomUUID();
      VaultPermission permission = VaultPermission.getFullPermissions(vaultIdentifier);
      
      // Store vault permission and vault identifier in session
      session.put(ClientSession.VAULT_PERMISSION, permission);
      session.put(ClientSession.VAULT_UUID, vaultIdentifier);
      
      return session;
   }

   private VaultACL createACL() throws Exception
   {
      vaultIdentifier = UUID.nameUUIDFromBytes("RemoteSliceStore".getBytes());

      final String aclFileName =
            "test-data/org/cleversafe/vault/" + vaultIdentifier.toString() + ".der";

      FileInputStream in = new FileInputStream(aclFileName);
      return new FileVaultACL(in);
   }

   /**
    * Creates a connection to a SliceServer, optionally authenticates
    * 
    * @param authenticate
    *            Flag specifying if the connection should be authenticated
    * @return A connection to a slice server
    */
   /*
   private Connector createConnection(boolean authenticate) {
      // Create slice server connection
      sliceServer = null;
      try {
         sliceServer = new SliceServerApplication(
               new SliceServerConfiguration());
      } catch (Exception ex) {
         ex.printStackTrace();
         fail("Unable to construct sliceServer");
      }

      Connector conn = new ServerApplicationConnection(
            sliceServer);

      return conn;
   }
 */
    
   
   private static void randomizeBuffer(byte data[]) {
      SecureRandom generator = new SecureRandom();
      generator.nextBytes(data);
   }

   private static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }
}
   



