//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.util.Enumeration;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERBoolean;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;

/**
 * Stores vault permission information for a single access control entry. Allows serialization
 * and deserialization.
 * <p>
 * Permission implications are not enforced in this class. Use {@link VaultPermission} to obtain
 * new vault permission entry instances.
 * 
 * <pre>
 * VaultPermissionEntry   ::= SEQUENCE  {
 *      read         [0] BOOLEAN DEFAULT FALSE,
 *      write        [1] BOOLEAN DEFAULT FALSE,
 *      list         [2] BOOLEAN DEFAULT FALSE,
 *      rebuild      [3] BOOLEAN DEFAULT FALSE,
 *      deleteVault  [4] BOOLEAN DEFAULT FALSE,
 *      modifyACL    [5] BOOLEAN DEFAULT FALSE,
 *      ...  }
 * </pre>
 *
 */
public class VaultPermissionEntry extends ASN1Encodable
{
   
   private boolean read;
   private boolean write;
   private boolean list;
   private boolean rebuild;
   private boolean deleteVault;
   private boolean modifyACL;
   
   
   public static VaultPermissionEntry getFullPermission()
   {
      VaultPermissionEntry entry = new VaultPermissionEntry();
      entry.read = true;
      entry.write = true;
      entry.list = true;
      entry.rebuild = true;
      entry.deleteVault = true;
      entry.modifyACL = true;
      return entry;
   }
   
   public VaultPermissionEntry()
   {
      this.read = false;
      this.write = false;
      this.list = false;
      this.rebuild = false;
      this.deleteVault = false;
      this.modifyACL = false;
   }
   
   public VaultPermissionEntry( VaultPermissionEntry entry )
   {
      this.read = entry.read;
      this.write = entry.write;
      this.list = entry.list;
      this.rebuild = entry.rebuild;
      this.deleteVault = entry.deleteVault;
      this.modifyACL = entry.modifyACL;
   }
   
   protected VaultPermissionEntry( ASN1Sequence seq )
   {
      Enumeration<?> e = seq.getObjects();
      
      while ( e.hasMoreElements() )
      {
         Object obj = e.nextElement();
         if ( obj instanceof DERTaggedObject )
         {
            Integer tagNo = ((DERTaggedObject)obj).getTagNo();
            boolean value = DERBoolean.getInstance( (DERTaggedObject)obj, true ).isTrue();
            
            switch( tagNo )
            {
               // if tags not listed here are encountered, we ignore them
               case 0: this.read = value;
               case 1: this.write = value;
               case 2: this.list = value;
               case 3: this.rebuild = value;
               case 4: this.deleteVault = value;
               case 5: this.modifyACL = value;
            }
         }
         else
         {
            throw new IllegalArgumentException("expected sequence of tagged objects");
         }
      }
   }

   public boolean canDeleteVault()
   {
      return deleteVault;
   }

   public void setDeleteVault(boolean delete)
   {
      this.deleteVault = delete;
   }

   public boolean canList()
   {
      return list;
   }

   public void setList(boolean list)
   {
      this.list = list;
   }

   public boolean canModifyACL()
   {
      return modifyACL;
   }

   public void setModifyACL(boolean modifyACL)
   {
      this.modifyACL = modifyACL;
   }

   public boolean canRead()
   {
      return read;
   }

   public void setRead(boolean read)
   {
      this.read = read;
   }

   public boolean canRebuild()
   {
      return rebuild;
   }

   public void setRebuild(boolean rebuild)
   {
      this.rebuild = rebuild;
   }

   public boolean canWrite()
   {
      return write;
   }

   public void setWrite(boolean write)
   {
      this.write = write;
   }

   @Override
   public int hashCode()
   {
      final int PRIME = 31;
      int result = super.hashCode();
      result = PRIME * result + (deleteVault ? 1231 : 1237);
      result = PRIME * result + (list ? 1231 : 1237);
      result = PRIME * result + (modifyACL ? 1231 : 1237);
      result = PRIME * result + (read ? 1231 : 1237);
      result = PRIME * result + (rebuild ? 1231 : 1237);
      result = PRIME * result + (write ? 1231 : 1237);
      return result;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;
      if (!super.equals(obj))
         return false;
      if (getClass() != obj.getClass())
         return false;
      final VaultPermissionEntry other = (VaultPermissionEntry) obj;
      if (deleteVault != other.deleteVault)
         return false;
      if (list != other.list)
         return false;
      if (modifyACL != other.modifyACL)
         return false;
      if (read != other.read)
         return false;
      if (rebuild != other.rebuild)
         return false;
      if (write != other.write)
         return false;
      return true;
   }
   
   public static VaultPermissionEntry getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   public static VaultPermissionEntry getInstance( Object obj )
   {
      if ( obj instanceof VaultPermissionEntry )
      {
         return (VaultPermissionEntry)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new VaultPermissionEntry( (ASN1Sequence)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }

   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();
      
      v.add( new DERTaggedObject(0, new DERBoolean(this.read)) );
      v.add( new DERTaggedObject(1, new DERBoolean(this.write)) );
      v.add( new DERTaggedObject(2, new DERBoolean(this.list)) );
      v.add( new DERTaggedObject(3, new DERBoolean(this.rebuild)) );
      v.add( new DERTaggedObject(4, new DERBoolean(this.deleteVault)) );
      v.add( new DERTaggedObject(5, new DERBoolean(this.modifyACL)) );
      
      return new DERSequence(v);
   }
}


