//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Enumeration;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERUTF8String;

/**
 * Contains an encoded key and enough information to decode it.
 * 
 * <pre>
 * KeyInfo   ::= SEQUENCE  {
 *      algorithm   UTF8String,
 *      type        KeyType,
 *      usage       KeyUsage,
 *      key         OCTET STRING  }
 * </pre>
 */
public class PlainKeyInfo extends KeyInfo
{
   private Key key;
   
   /**
    * Constructor for use by subclasses. If used all values must be set manually before the
    * object is in a valid state.
    */
//   protected PlainKeyInfo()
//   {
//      super();
//   }
   
   // Copy constructor
   protected PlainKeyInfo( PlainKeyInfo info )
   {
      super(info);
      this.key = info.key;    // Key is immutable
   }
   
   
   /* Constructor to use from key generator. */
   protected PlainKeyInfo(Key key, KeyUsage usage)
   {
      super( key.getAlgorithm(), queryType(key), usage );
      this.key = key;
   }


   /*
    * Constructor to use from static factory methods
    */
   protected PlainKeyInfo( ASN1Sequence seq )
   {
      if ( seq.size() != 4 )
      {
         throw new IllegalArgumentException("Bad sequence size: " + seq.size());
      }
      
      Enumeration<?> e = seq.getObjects();
      
      this.setAlgorithm( DERUTF8String.getInstance(e.nextElement()).getString() );
      this.setType( DERInteger.getInstance(e.nextElement()).getValue().intValue() );
      this.setUsage( KeyUsage.getInstance(e.nextElement()) );
      try
      {
         this.setKey( decodeKey(
               DEROctetString.getInstance(e.nextElement()).getOctets(),
               this.getAlgorithm(),
               this.getType() ));
      }
      catch (NoSuchAlgorithmException e1)
      {
         throw new IllegalArgumentException("decoding key failed: no such algorithm", e1);
      }
      catch (InvalidKeySpecException e1)
      {
         throw new IllegalArgumentException("decoding key failed: invalid key spec", e1);
      }
   }
   
   
   /*
    * Decodes the given encoded key. Relies on all other member variables to be properly set.
    */
   protected static Key decodeKey( byte[] encodedKey, String algorithm, int type )
         throws NoSuchAlgorithmException, InvalidKeySpecException
   {
      Key key;
      
      if ( type == Cipher.PUBLIC_KEY )
      {
         KeyFactory fact = KeyFactory.getInstance( algorithm );
         X509EncodedKeySpec pubKeySpec = new X509EncodedKeySpec( encodedKey );
         key = fact.generatePublic( pubKeySpec );
         assert( algorithm.equals(key.getAlgorithm()) );
      }
      else if ( type == Cipher.PRIVATE_KEY )
      {
         KeyFactory fact = KeyFactory.getInstance( algorithm );
         PKCS8EncodedKeySpec priKeySpec = new PKCS8EncodedKeySpec( encodedKey );
         key = fact.generatePrivate( priKeySpec );
         assert( algorithm.equals(key.getAlgorithm()) );
      }
      else if ( type == Cipher.SECRET_KEY )
      {
         key = new SecretKeySpec( encodedKey, algorithm );
         assert( algorithm.equals(key.getAlgorithm()) );
      }
      else
      {
         throw new IllegalArgumentException("invalid key type passed into KeyInfo constructor");
      }
      
      return key;
   }
   
   
   /**
    * Encodes the stored key in the desired platform-independent manner.
    */
   protected static byte[] encodeKey( Key key, int type )
   {
      try
      {
         if ( type == Cipher.PUBLIC_KEY )
         {
            KeyFactory fact = KeyFactory.getInstance( key.getAlgorithm() );
            return fact.getKeySpec(key, X509EncodedKeySpec.class).getEncoded();
         }
         else if ( type == Cipher.PRIVATE_KEY )
         {
            KeyFactory fact = KeyFactory.getInstance( key.getAlgorithm() );
            return fact.getKeySpec(key, PKCS8EncodedKeySpec.class).getEncoded();
         }
         else if ( type == Cipher.SECRET_KEY )
         {
            // Secret keys generally "encode" by dumping their key material as a byte array.
            // The following line assumes that is always the case.
            return ((SecretKey)key).getEncoded();
         }
      }
      catch (NoSuchAlgorithmException e)
      {
         // If the algorithm is invalid this exception should have been thrown in the constructor
         throw new RuntimeException("previously found algorithm now invalid", e);
      }
      catch (InvalidKeySpecException e)
      {
         throw new RuntimeException("failed to create valid key spec during encoding", e);
      }
      
      // An IllegalArguementException should have been thrown in the constructor.
      throw new IllegalStateException(
            "invalid state: unexpected key info type value: " + type);
   }


   /**
    * Generates a new secret (symmetric) key and returns it in a KeyInfo object.
    * 
    * @param algorithm The cryptographic algorithm the key will be used with.
    * @param size The size in bits of the key to generate. 
    * @param usage What stack operations the key will be used for.
    * @return A new key wrapped in a KeyInfo object.
    * @throws NoSuchAlgorithmException If a key generator for the specified algorithm is not
    *             available from the default cryptographic provider.
    */
   public static PlainKeyInfo generateKey( String algorithm, int size, KeyUsage usage )
         throws NoSuchAlgorithmException
   {
      KeyGenerator keygen = KeyGenerator.getInstance( algorithm );
      keygen.init(size);
      Key key = keygen.generateKey();
      
      assert( key instanceof SecretKey );
      assert( key.getAlgorithm().equals(algorithm) );
      
      return new PlainKeyInfo( key, usage );
   }
   
   /**
    * Generates a new key pair and returns it in two KeyInfo objects.
    * 
    * @param algorithm The cryptographic algorithm the key will be used with.
    * @param size The size in bits of the key to generate. 
    * @param usage What stack operations the key will be used for.
    * @return A pair of KeyInfo objects.
    * @throws NoSuchAlgorithmException 
    * @throws NoSuchAlgorithmException If the algorithm is not available from the provider.
    */
   public static KeyInfoPair generateKeyPair( String algorithm, int size, KeyUsage usage )
         throws NoSuchAlgorithmException
   {
      KeyPairGenerator keygen = KeyPairGenerator.getInstance( algorithm );
      keygen.initialize( size );
      
      KeyPair pair = keygen.generateKeyPair();
      
      assert( pair.getPrivate() instanceof PrivateKey );
      assert( pair.getPublic() instanceof PublicKey );
      
      KeyInfoPair info = new KeyInfoPair();
      info.setPrivate( new PlainKeyInfo(pair.getPrivate(), usage) );
      info.setPublic( new PlainKeyInfo(pair.getPublic(), usage) );
      
      return info;
   }
   

   /**
    * The key object.
    */
   @Override
   public Key getKey()
   {
      return key;
   }
   
   @Override
   protected void setKey( Key key )
   {
      this.key = key;
      this.setAlgorithm( key.getAlgorithm() );
      this.setType( key );
   }
   
   
   public static PlainKeyInfo getInstance( ASN1TaggedObject obj, boolean explicit )
   {
      return getInstance( ASN1Sequence.getInstance(obj, explicit) );
   }
   
   public static PlainKeyInfo getInstance( Object obj )
   {
      if ( obj instanceof PlainKeyInfo )
      {
         return (PlainKeyInfo)obj;
      }
      else if ( obj instanceof ASN1Sequence )
      {
         return new PlainKeyInfo( (ASN1Sequence)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   


   @Override
   public DERObject toASN1Object()
   {
      ASN1EncodableVector v = new ASN1EncodableVector();
      
      v.add( new DERUTF8String(this.getAlgorithm()) );
      v.add( new DERInteger(this.getType()) );
      v.add( this.getUsage().getDERObject() );
      v.add( new DEROctetString(encodeKey(this.getKey(), this.getType())) );
      
      return new DERSequence(v);
   }
   
}


