//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;
import java.security.Key;
import java.security.PrivateKey;
import java.security.PublicKey;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;

import org.bouncycastle.asn1.ASN1Encodable;

/**
 * Base class for encoded keys. Contains enough information to encode and decode keys.
 */
abstract public class KeyInfo extends ASN1Encodable
{
   private String algorithm;
   private int type;
   private KeyUsage usage;
   
   public static class KeyInfoPair
   {
      private KeyInfo pri;
      private KeyInfo pub;
      
      public KeyInfo getPrivate()
      {
         return pri;
      }
      public void setPrivate(KeyInfo pri)
      {
         this.pri = pri;
      }
      public KeyInfo getPublic()
      {
         return pub;
      }
      public void setPublic(KeyInfo pub)
      {
         this.pub = pub;
      }
   }
   
   protected static int queryType( Key key )
   {
      if ( key instanceof SecretKey )
      {
         return Cipher.SECRET_KEY;
      }
      else if ( key instanceof PrivateKey )
      {
         return Cipher.PRIVATE_KEY;
      }
      else if ( key instanceof PublicKey )
      {
         return Cipher.PUBLIC_KEY;
      }
      else
      {
         throw new IllegalArgumentException("invalid key class passed to KeyInfo constructor");
      }
   }
   
   
   protected KeyInfo() {}
   
   protected KeyInfo( KeyInfo info )
   {
      this.algorithm = info.algorithm;
      this.type = info.type;
      this.usage = info.usage;
   }
   
   protected KeyInfo(String algorithm, int type, KeyUsage usage )
   {
      this.algorithm = algorithm;
      this.type = type;
      this.usage = usage;
   }

   /**
    * The key object.
    * @throws SecurityException If access to the key is not authorized (e.g., if the key
    *                           has not yet been decrypted).
    */
   abstract public Key getKey();
   
   abstract protected void setKey( Key key );
   
   /**
    * The key algorithm. This is the same as <code>getKey().getAlgorithm()</code>.
    * @return
    */
   public String getAlgorithm()
   {
      return algorithm;
   }

   /**
    * The key type. Can be either {@link Cipher#SECRET_KEY}, {@link Cipher#PRIVATE_KEY}, or
    * {@link Cipher#PUBLIC_KEY}.
    */
   public int getType()
   {
      return type;
   }
   
   /**
    * The intended usage for this key on the codec stacks.
    */
   public KeyUsage getUsage()
   {
      return usage;
   }

   protected void setAlgorithm(String algorithm)
   {
      this.algorithm = algorithm;
   }

   protected void setType(int type)
   {
      this.type = type;
   }
   
   protected void setType( Key key )
   {
      this.type = queryType( key );
   }

   protected void setUsage(KeyUsage usage)
   {
      this.usage = usage;
   }
   
   
   
   
}


