//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: wleggette
//
// Date: Aug 29, 2007
//---------------------

package org.cleversafe.vault.storage;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.cleversafe.codec.KeyedCodec;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.evaluator.Parameter;
import org.cleversafe.config.evaluator.Reference;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.VaultDocument;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.storage.asn1.KeyUsage;

/**
 * Generates a list of {@link VaultKeyInfo} specifications from an XML vault descriptor file. This
 * class is temporary. An eventual vault configuration UI should create the vault key specifications
 * directly, separately from vault descriptor XML file creation.
 */
public class VaultKeyInfoGenerator
{
   private static Logger _logger = Logger.getLogger(VaultKeyInfoGenerator.class);

   private static Map<String, KeyUsage> _kindUsageMapCompat;

   private List<VaultKeyInfo> keyInfoList;

   // FIXME: Change this back to 256 in the future, requires having the
   // Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy Files
   private static int DEFAULT_SYMMETRIC_KEY_SIZE = 128;
   private static int DEFAULT_ASYMMETRIC_KEY_SIZE = 1024;

   static
   {
      // TODO: This kind to key usage map is hard coded. It is probably not worth
      // creating a configuration framework to determine these since this whole
      // class will be eliminated shortly.
      _kindUsageMapCompat = new HashMap<String, KeyUsage>();
      _kindUsageMapCompat.put("DATASOURCE/Codec.Encryption/cipher-codec",
            KeyUsage.DATASOURCE_ENCRYPTION);
      _kindUsageMapCompat.put("DATASOURCE/Codec.Integrity/signature",
            KeyUsage.DATASOURCE_VERIFICATION);
      _kindUsageMapCompat.put("SLICE/Codec.Integrity/signature", KeyUsage.SLICE_VERIFICATION);

   }

   public static List<VaultKeyInfo> generate(InputStream xmlInputStream)
         throws VaultDescriptorException, XmlException, IOException, ConfigurationException
   {
      VaultKeyInfoGenerator gen = new VaultKeyInfoGenerator();

      VaultDocument vd = VaultDocument.Factory.parse(xmlInputStream);
      if (!vd.validate())
      {
         throw new VaultDescriptorException("Vault descriptor validation failed");
      }
      VaultDocument.Vault vault = vd.getVault();

      List<VaultKeyInfo> compatKeyInfoList = gen.parseVaultDescriptorForCompatibility(vault);

      Tuple2<List<VaultKeyInfo>, List<Integer>> tuple = gen.parseVaultDescriptor(vault);
      List<VaultKeyInfo> keyInfoList = tuple.getFirst();
      List<Integer> compatList = tuple.getSecond();

      keyInfoList = gen.mergeKeyInfoLists(compatKeyInfoList, keyInfoList, compatList);
      _logger.trace("Completed vault key info generation: " + keyInfoList);
      return keyInfoList;
   }

   public VaultKeyInfoGenerator()
   {
   }

   private List<VaultKeyInfo> mergeKeyInfoLists(
         final List<VaultKeyInfo> compatKeyInfoList,
         final List<VaultKeyInfo> keyInfoList,
         final List<Integer> compatList) throws VaultDescriptorException
   {
      List<VaultKeyInfo> list = new ArrayList<VaultKeyInfo>();

      // Go through "may be compatibility list" and check entries
      Iterator<VaultKeyInfo> cvkit = compatKeyInfoList.iterator();
      Set<Integer> indicies = new HashSet<Integer>();
      for (VaultKeyInfo info : keyInfoList)
      {
         if (compatList.contains(info.getStorageIndex()))
         {
            final VaultKeyInfo compat = cvkit.next();
            _logger.trace("Replacing key entry " + info + " with compatibility entry " + compat);
            info = compareKeyInfoEntry(info.getStorageIndex(), compat, info);
         }

         // If the key index is a duplicate, a failure has occured
         if (!indicies.add(info.getStorageIndex()))
            throw new VaultDescriptorException(
                  "Deprecated vault descriptor codec entry caused key index collision: " + info);

         list.add(info);
      }

      return list;
   }

   /**
    * Compares a vault key info object obtained using the compatibility reader with an entry
    * obtained using the new reader to decide which entry to use.
    * 
    * @param compat
    *           An entry obtained from the compatibility reader.
    * @param info
    *           An entry obtained using the new reader.
    * @return The final entry that should be used.
    */
   private VaultKeyInfo compareKeyInfoEntry(
         int index,
         final VaultKeyInfo compat,
         final VaultKeyInfo info) throws VaultDescriptorException
   {
      _logger.trace("Comparing compatibility entry " + compat + " with " + info);
      if (compat.getUsage() != info.getUsage())
      {
         // This should not happen normally, but is probably not a runtime exception
         throw new VaultDescriptorException("Vault Descriptor interpreter fault for codec entry "
               + index);
      }

      if (info.getAlgorithm() != null)
      {
         // If new reader detects algorithm then it is set in the bindings
         _logger.warn("In vault descriptor, deprecated codec entry " + index
               + " overrides default " + "algorithm " + info.getAlgorithm() + " with "
               + compat.getAlgorithm());
      }

      if (compat.getSize() != info.getSize())
      {
         _logger.warn("In vault descriptor, deprecated codec entry " + index
               + " overrides default " + "key size " + info.getSize() + " with " + compat.getSize());
      }

      if (compat.getStorageIndex() != info.getStorageIndex())
      {
         _logger.warn("In vault descriptor, deprecated codec entry " + index + " has improper key "
               + "index " + compat.getStorageIndex() + "; expected " + info.getStorageIndex());
      }

      return compat;
   }

   private String parseAlgorithm(Reference codec, Parameter param, String old)
         throws VaultDescriptorException
   {
      if (param.getName().equals("algorithm") || param.getName().equals("transformation"))
      {
         if (!param.isSetLiteral())
         {
            throw new VaultDescriptorException("Codec " + codec.getKind()
                  + " algorithm or transformation parameter does not have a literal set");
         }

         return param.getLiteral().getValue();
      }
      else
      {
         return old;
      }
   }

   private int parseExplicitStorageIndex(Reference codec) throws VaultDescriptorException
   {
      int index = -1;
      for (Parameter param : codec.getParamArray())
      {
         index = parseExplicitStorageIndex(codec, param, index);
      }
      return index;
   }

   /*
    * Find the storage index explicitly indicated in the descriptor codec parameters. Note that this
    * form of index indication will be deprecated soon.
    */
   private int parseExplicitStorageIndex(Reference codec, Parameter param, int old)
         throws VaultDescriptorException
   {
      if (param.getName().equals("private-key") || param.getName().equals("key"))
      {
         if (!param.isSetMethod() || param.getMethod().getArgArray().length < 1
               || !param.getMethod().getArgArray(0).isSetLiteral())
         {
            throw new VaultDescriptorException("Codec " + codec.getKind()
                  + " does not have a proper key lookup method");
         }
         try
         {
            return Integer.parseInt(param.getMethod().getArgArray(0).getLiteral().getValue());
         }
         catch (NumberFormatException e)
         {
            throw new VaultDescriptorException("Codec " + codec.getKind()
                  + " storage index improper", e);
         }
      }
      else
      {
         return old;
      }
   }

   private List<VaultKeyInfo> parseVaultDescriptorForCompatibility(VaultDocument.Vault vault)
         throws VaultDescriptorException
   {
      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();

      for (Reference codec : vault.getCodecArray())
      {
         final String searchString = "DATASOURCE/" + codec.getKind() + "/" + codec.getReferral();
         _logger.trace("Examining codec in compatibility mode: " + searchString);

         if (_kindUsageMapCompat.containsKey(searchString))
         {
            _logger.trace("Codec does require encryption keys");
            KeyUsage usage = _kindUsageMapCompat.get(searchString);

            // These must be set to valid values at the end of parameter parsing
            String algorithm = null;
            int storageIndex = -1;

            for (Parameter param : codec.getParamArray())
            {
               algorithm = parseAlgorithm(codec, param, algorithm);
               storageIndex = parseExplicitStorageIndex(codec, param, storageIndex);
            }

            if (algorithm == null)
            {
               throw new VaultDescriptorException("Codec " + codec.getKind()
                     + " does not contain algorithm or transform parameter");
            }
            if (storageIndex < 0)
            {
               throw new VaultDescriptorException("Codec " + codec.getKind()
                     + " does not contain proper storage index reference");
            }

            // now create a VaultKeyInfo object.

            VaultKeyInfo keyInfo = new VaultKeyInfo();
            keyInfo.setAlgorithm(algorithm);

            if (usage == KeyUsage.DATASOURCE_ENCRYPTION)
            {
               keyInfo.setSize(DEFAULT_SYMMETRIC_KEY_SIZE);
            }
            else
            {
               keyInfo.setSize(DEFAULT_ASYMMETRIC_KEY_SIZE);
            }

            keyInfo.setStorageIndex(storageIndex);
            keyInfo.setUsage(usage);

            _logger.trace("Adding key usage data: " + keyInfo);
            keyInfoList.add(keyInfo);
         }
         // else ignore, not a codec that needs a key
      }

      for (Reference codec : vault.getSliceCodecArray())
      {
         String kindUsageMapKey = "SLICE/" + codec.getKind() + "/" + codec.getReferral();
         _logger.trace("Examining codec in compatibility mode: " + kindUsageMapKey);
         if (_kindUsageMapCompat.containsKey(kindUsageMapKey))
         {
            _logger.trace("Codec does require encryption keys");
            KeyUsage usage = _kindUsageMapCompat.get(kindUsageMapKey);

            // These must be set to valid values at the end of parameter parsing
            String algorithm = null;
            int storageIndex = -1;

            for (Parameter param : codec.getParamArray())
            {
               algorithm = parseAlgorithm(codec, param, algorithm);
               storageIndex = parseExplicitStorageIndex(codec, param, storageIndex);
            }

            if (algorithm == null)
            {
               throw new VaultDescriptorException("Slice codec " + codec.getKind()
                     + " does not contain algorithm or transform parameter");
            }
            if (storageIndex < 0)
            {
               throw new VaultDescriptorException("Slice codec " + codec.getKind()
                     + " does not contain proper storage index reference");
            }

            // now create a VaultKeyInfo object.

            VaultKeyInfo keyInfo = new VaultKeyInfo();
            keyInfo.setAlgorithm(algorithm);
            keyInfo.setSize(DEFAULT_ASYMMETRIC_KEY_SIZE);
            keyInfo.setStorageIndex(storageIndex);
            keyInfo.setUsage(usage);

            _logger.trace("Adding key usage data: " + keyInfo);
            keyInfoList.add(keyInfo);
         }
      }

      // check for duplicate storage indices
      Set<Integer> used = new HashSet<Integer>();
      for (VaultKeyInfo keyInfo : keyInfoList)
      {
         if (!used.add(keyInfo.getStorageIndex()))
         {
            throw new VaultDescriptorException("Duplicate key storage index: "
                  + keyInfo.getStorageIndex());
         }

      }

      return keyInfoList;
   }

   /**
    * Checks if a given entry may be a compatibility entry
    */
   private boolean mayBeCompat(Reference reference, boolean datasourceCodec)
   {
      final String kindUsageMapKey =
            (datasourceCodec ? "DATASOURCE/" : "SLICE/") + reference.getKind() + "/"
                  + reference.getReferral();
      if (_kindUsageMapCompat.containsKey(kindUsageMapKey))
      {
         _logger.trace("Codec may be compatibility entry: " + kindUsageMapKey);
         return true;
      }
      else
      {
         return false;
      }
   }

   private int searchReferenceArray(
         final Reference[] references,
         int startIndex,
         boolean datasourceCodecs,
         List<VaultKeyInfo> keyInfoList,
         List<Integer> compatList) throws ConfigurationException, VaultDescriptorException
   {
      int index = startIndex;
      _logger.trace("Starting scan at codec index " + index);
      for (Reference reference : references)
      {
         _logger.trace("Examining reference: " + reference.toString());
         Object obj =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getImplementation(
                     reference.getKind().split("\\."), reference.getReferral());
         _logger.trace("Loaded codec object: " + obj.toString());
         if (obj instanceof KeyedCodec)
         {
            _logger.trace("Codec does require encryption keys");

            if (mayBeCompat(reference, datasourceCodecs))
            {
               compatList.add(index);
            }

            KeyedCodec codec = (KeyedCodec) obj;
            VaultKeyInfo info = new VaultKeyInfo();
            info.setAlgorithm(codec.getAlgorithm());
            info.setSize(codec.getKeySize());
            info.setStorageIndex(index);
            switch (codec.getKeyUsage())
            {
               case ENCRYPTION :
                  if (!datasourceCodecs)
                     throw new VaultDescriptorException(
                           "encryption codecs cannot currently be used as slice codecs");
                  info.setUsage(KeyUsage.DATASOURCE_ENCRYPTION);
                  break;
               case INTEGRITY :
                  info.setUsage(datasourceCodecs
                        ? KeyUsage.DATASOURCE_VERIFICATION
                        : KeyUsage.SLICE_VERIFICATION);
                  break;
            }
            _logger.trace("Adding key usage data: " + info);
            keyInfoList.add(info);
         }
         index++; // increment the codec counter for all codecs, not just keyed codecs
         _logger.trace("next codec is at index " + index);
      }
      return index;
   }

   private Tuple2<List<VaultKeyInfo>, List<Integer>> parseVaultDescriptor(VaultDocument.Vault vault)
         throws VaultDescriptorException, ConfigurationException
   {

      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();
      List<Integer> compatList = new ArrayList<Integer>(); // entries which may be compat entries

      int index = 0; // we increment index count for datasource and slice codecs

      // Search through datasource codecs for "keyed codecs"
      _logger.debug("Examining datasource codecs");
      index = searchReferenceArray(vault.getCodecArray(), index, true, keyInfoList, compatList);

      // Search through slice codecs for "keyed codecs"
      _logger.debug("Examining slice codecs");
      index =
            searchReferenceArray(vault.getSliceCodecArray(), index, false, keyInfoList, compatList);

      return new Tuple2<List<VaultKeyInfo>, List<Integer>>(keyInfoList, compatList);
   }

}
