//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 4, 2007
//---------------------

package org.cleversafe.vault.managers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.URI;
import java.security.KeyPair;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.VaultManager;
import org.cleversafe.vault.XMLVaultDescriptor;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultKeyGenerationException;
import org.cleversafe.vault.exceptions.VaultLoadException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.managers.state.VaultState;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.cleversafe.vault.storage.VaultKeyInfoGenerator;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.cfg.AnnotationConfiguration;

// TODO: Describe class or interface
public class LocalFileVaultManager implements VaultManager
{
   private static Logger _logger = Logger.getLogger(LocalFileVaultManager.class);

   public final static String LOCAL_FILE_DATABASE = "directory.db";
   public final static String LOCAL_FILE_VAULTS = "vaults";

   public final static String DESCRIPTOR_EXTENSION = ".xml";
   public final static String ACL_EXTENSION = ".acl";

   private URI directoryLocation;

   public LocalFileVaultManager()
   {
   }

   public LocalFileVaultManager(URI directoryLocation)
   {
      this.directoryLocation = directoryLocation;
   }

   /**
    * Used by other local file implementations to populate the vault manager with a pre-created
    * session factory which adheres to the expected VaultStatus schema.
    */
   public synchronized VaultManager load(URI directory, SessionFactory sessionFactory)
   {
      this.directoryLocation = directory;
      _sessionFactoryMap.put(directory, sessionFactory);
      return this;
   }

   public synchronized VaultManager load(URI directory) throws VaultIOException
   {
      this.directoryLocation = directory;
      LocalFileVaultManager.loadDirectory(directory);
      return this;
   }

   public synchronized void close() throws VaultIOException
   {
      shutdown(this.directoryLocation);
   }

   protected void checkInitialization()
   {
      if (this.directoryLocation == null)
      {
         throw new InitializationException("Grid access manager not properly initialized");
      }
   }

   // //////////////////////////////////////////////////////////////////////////////////////////////

   private static Map<URI, SessionFactory> _sessionFactoryMap;
   private static Properties _configuration;

   static
   {
      _sessionFactoryMap = new HashMap<URI, SessionFactory>();

      _configuration = new Properties();
      //      Properties c = _configuration;
      //
      //      c.setProperty("hibernate.connection.driver_class", "org.apache.derby.jdbc.EmbeddedDriver");
      //      c.setProperty("hibernate.dialect", "org.hibernate.dialect.DerbyDialect");
      //      c.setProperty("hibernate.hbm2dll.auto", "update");
      //
      //      c.setProperty("hibernate.c3p0.min_size", "5");
      //      c.setProperty("hibernate.c3p0.max_size", "20");
      //      c.setProperty("hibernate.c3p0.timeout", "300");
      //      c.setProperty("hibernate.c3p0.max_statements", "50");
      //      c.setProperty("hibernate.c3p0.idle_test_period", "3000");
      //
      //      c.setProperty("show_sql", "true");
      //      c.setProperty("format_sql", "true");
   }

   protected static void loadDirectory(URI directory) throws VaultIOException
   {
      synchronized (_sessionFactoryMap)
      {
         if (!directory.getScheme().equals("file"))
         {
            throw new RuntimeException(
                  "Local file account manager requires 'file' directory schema");
         }

         File dir = new File(directory.getPath());

         _logger.debug("Loaded directory from database: " + dir);

         _configuration.setProperty("hibernate.connection.url", "jdbc:derby:" + dir
               + File.separator + LOCAL_FILE_DATABASE + ";create=True");

         _logger.trace("Inserting hibernate.connection.url="
               + _configuration.getProperty("hibernate.connection.url"));

         try
         {
            SessionFactory factory =
                  new AnnotationConfiguration().addAnnotatedClass(
                        org.cleversafe.vault.managers.state.VaultState.class).addProperties(
                        _configuration).configure().buildSessionFactory();
            _sessionFactoryMap.put(directory, factory);
         }
         catch (Throwable ex)
         {
            throw new VaultIOException("Error configuring session factory for directory database",
                  ex);
         }
      }
   }

   protected static SessionFactory getSessionFactory(URI directory)
   {
      synchronized (_sessionFactoryMap)
      {
         return _sessionFactoryMap.get(directory);
      }
   }

   private static void shutdown(URI directory)
   {
      synchronized (_sessionFactoryMap)
      {
         SessionFactory hibernateSessionFactory = _sessionFactoryMap.remove(directory);
         if (hibernateSessionFactory != null)
            hibernateSessionFactory.close();
      }
   }

   ////////////////////////////////////////////////////////////////////////////////////////////////

   private static class VaultIdentifierIterator implements Iterator<UUID>
   {
      private final Iterator<VaultState> vaults;

      public VaultIdentifierIterator(Iterator<VaultState> vaults)
      {
         this.vaults = vaults;
      }

      public boolean hasNext()
      {
         return this.vaults.hasNext();
      }

      public UUID next()
      {
         return this.vaults.next().getIdentifier();
      }

      public void remove()
      {
         this.vaults.remove();
      }

   }

   @SuppressWarnings("unchecked")
   public synchronized List<UUID> getVaults() throws VaultIOException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         List<VaultState> vaults = session.createQuery("from VaultState").list();
         List<UUID> ids = new ArrayList<UUID>();
         for (VaultState v : vaults)
         {
            ids.add(v.getIdentifier());
         }
         tx.commit();
         return ids;
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   @SuppressWarnings("unchecked")
   public synchronized Iterator<UUID> iterator()
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         return new VaultIdentifierIterator(session.createQuery("from VaultState").iterate());
      }
      finally
      {
         tx.commit();
         session.close();
      }
   }

   ////////////////////////////////////////////////////////////////////////////////////////////////

   public synchronized void checkVault(
         UUID vaultIdentifier,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException, VaultSecurityException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         getVaultFromDatabase(vaultIdentifier, session);

         // get account credentials
         Credentials credentials = manager.getCredentials(accountIdentifier);
         KeyPair keys = manager.getAccountKeyPair(accountIdentifier);

         // Update vault information on the slice stores
         StoreExecuter executer = new StoreExecuter()
         {
            public void execute(
                  SliceStore store,
                  String vaultType,
                  long maxSliceSize,
                  long sliceStoreSize,
                  VaultACL accessControlList,
                  byte[] vaultDescriptorBytes) throws SliceStoreExistsException,
                  SliceStoreIOException, SliceStoreNotFoundException
            {
               store.updateStore(vaultType, maxSliceSize, sliceStoreSize, accessControlList,
                     vaultDescriptorBytes);
            }
         };
         File descriptor = _getDescriptorFile(this.directoryLocation, vaultIdentifier);
         executeOnStores(vaultIdentifier, descriptor, credentials, keys, executer, false);

         tx.commit();
      }
      catch (CredentialsException e)
      {
         throw new VaultSecurityException("could not load account credentials", e);
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   public synchronized UUID createVault(
         String vaultName,
         String vaultDescriptor,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException
   {
      checkInitialization();
      if (vaultDescriptor == null)
      {
         throw new RuntimeException("invalid vault descriptor given (null)");
      }

      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      UUID vaultIdentifier = null;
      // TODO: Change try blocks to make cathces more localized and clean
      // Exception handling is pretty poor in this code
      // It is not localized and thus very hard to follow
      // However, I have no intent to patch it now. 
      // Adding additional try block to close a session
      try
      {
         try
         {
            // Check vault name uniqueness and generate a new unique vault UUID
            if (session.createQuery("from VaultState v where v.name is :name").setParameter("name",
                  vaultName).list().size() != 0)
            {
               throw new VaultIOException("duplicate vault name already exists: " + vaultName);
            }

            // Theoretically this is not guaranteed to produce unique because
            // a simultaneous create may do the same. But it is about as good as 
            // that world would ceased to exist
            while (true)
            {
               vaultIdentifier = UUID.randomUUID();
               // check for duplicate UUID
               if (session.createQuery(
                     "from VaultState v where v.identifier is '" + vaultIdentifier + "'").list().size() == 0)
                  break;
            }
         }
         catch (HibernateException ex)
         {
            throw new VaultIOException("Error quiering for vault " + vaultIdentifier + ": " + ex);
         }
         
         _logger.debug("Creating new vault '" + vaultName + "' as: " + vaultIdentifier);

         try
         {
            // get account credentials
            Credentials credentials = manager.getCredentials(accountIdentifier);
            KeyPair keys = manager.getAccountKeyPair(accountIdentifier);

            // Save the vault descriptor
            File descriptor = _getDescriptorFile(this.directoryLocation, vaultIdentifier);
            _logger.trace("Writing vault descriptor to file: " + descriptor.getAbsolutePath());
            // FIXME: Have a out.close() in finally
            try
            {
               descriptor.getParentFile().mkdirs();
               FileWriter out = new FileWriter(descriptor);
               out.write(vaultDescriptor);
               out.flush();
               out.close();
               _logger.debug("Vault descriptor written to file: " + descriptor.getAbsolutePath());

               // create the vault ACL
               createVaultACL(vaultIdentifier, descriptor, credentials, keys.getPublic());

               // Setup the vault on the slice stores
               StoreExecuter executer = new StoreExecuter()
               {
                  public void execute(
                        SliceStore store,
                        String vaultType,
                        long maxSliceSize,
                        long sliceStoreSize,
                        VaultACL accessControlList,
                        byte[] vaultDescriptorBytes) throws SliceStoreExistsException,
                        SliceStoreIOException
                  {
                     store.createStore(vaultType, maxSliceSize, sliceStoreSize, accessControlList,
                           vaultDescriptorBytes);
                  }
               };
               executeOnStores(vaultIdentifier, descriptor, credentials, keys, executer, false);
            }
            catch (IOException e)
            {
               _logger.error("could not write or read vault descriptor file");
               throw new VaultIOException("could not write or read vault descriptor file", e);
            }

            VaultState vault = new VaultState();
            vault.setIdentifier(vaultIdentifier);
            vault.setName(vaultName);

            session.save(vault);
            tx.commit();

            return vaultIdentifier;

         }
         catch (CredentialsException e)
         {
            throw new VaultSecurityException("could not load account credentials", e);
         }
         catch (VaultException e)
         {
            // vault creation failed, clean-up any ACL or descriptor file
            // Rollback creation
            _logger.error("vault creation failed, rolling back...", e);
            try
            {
               __deleteVaultImpl(vaultIdentifier, accountIdentifier, manager);
               _logger.info("vault creation rollback was successful");
            }
            catch (Exception ignore)
            {
               _logger.info("error while rolling back vault creation. may have to delete slice stores manually.");
            }
            throw e;
         }
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   // Private method to connect to slice stores and delete them and to delete local vault files
   private synchronized void __deleteVaultImpl(
         UUID vaultIdentifier,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException
   {
      try
      {
         // get account credentials
         Credentials credentials = manager.getCredentials(accountIdentifier);
         KeyPair keys = manager.getAccountKeyPair(accountIdentifier);

         // Delete vault from the slice stores
         StoreExecuter executer = new StoreExecuter()
         {
            public void execute(
                  SliceStore store,
                  String vaultType,
                  long maxSliceSize,
                  long sliceStoreSize,
                  VaultACL accessControlList,
                  byte[] vaultDescriptorBytes) throws SliceStoreExistsException,
                  SliceStoreIOException, SliceStoreNotFoundException
            {
               store.deleteStore();
            }
         };
         File descriptor = _getDescriptorFile(this.directoryLocation, vaultIdentifier);
         executeOnStores(vaultIdentifier, descriptor, credentials, keys, executer, true);

         descriptor.delete();
         _getACLFile(this.directoryLocation, vaultIdentifier).delete();
      }
      catch (CredentialsException e)
      {
         throw new VaultSecurityException("could not load account credentials", e);
      }
   }

   public synchronized void deleteVault(
         UUID vaultIdentifier,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         // Delete vault
         __deleteVaultImpl(vaultIdentifier, accountIdentifier, manager);

         // Remove from database
         VaultState vault = getVaultFromDatabase(vaultIdentifier, session);
         session.delete(vault);
         tx.commit();
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   public synchronized VaultACL getVaultACL(UUID vaultIdentifier) throws VaultIOException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         Query q =
               session.createQuery("from VaultState v where v.identifier is :identifier").setParameter(
                     "identifier", vaultIdentifier);
         if (q.uniqueResult() == null)
            throw new VaultIOException("no such vault exists: " + vaultIdentifier);

         InputStream ain =
               new FileInputStream(_getACLFile(this.directoryLocation, vaultIdentifier));
         VaultACLFactory aclFactory = new VaultACLFactory();
         VaultACL acl = aclFactory.getInstance(ain);
         ain.close();

         tx.commit();
         return acl;
      }
      catch (IOException e)
      {
         throw new VaultIOException("could not load vault ACL", e);
      }
      catch (VaultConfigurationException e)
      {
         throw new VaultIOException(
               "could not load vault ACL because of improper library configuration", e);
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }

   }

   public synchronized UUID getVaultIdentifier(String vaultName) throws VaultIOException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         Query q =
               session.createQuery("from VaultState v where v.name is :name").setParameter("name",
                     vaultName);
         if (q.uniqueResult() == null)
            throw new VaultIOException("no such vault exists: " + vaultName);
         return ((VaultState) q.uniqueResult()).getIdentifier();
      }
      finally
      {
         tx.commit();
         session.close();
      }
   }

   public synchronized String getVaultName(UUID vaultIdentifier) throws VaultIOException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         Query q =
               session.createQuery("from VaultState v where v.identifier is :identifier").setParameter(
                     "identifier", vaultIdentifier.toString());
         if (q.uniqueResult() == null)
            throw new VaultIOException("no such vault exists: " + vaultIdentifier);
         return ((VaultState) q.uniqueResult()).getName();
      }
      finally
      {
         tx.commit();
         session.close();
      }
   }

   public synchronized String getVaultDescriptor(UUID vaultIdentifier) throws VaultIOException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         // check that vault exists in database
         getVaultFromDatabase(vaultIdentifier, session);

         RandomAccessFile file =
               new RandomAccessFile(_getDescriptorFile(this.directoryLocation, vaultIdentifier),
                     "r");
         byte[] bytes = new byte[(int) file.length()];
         file.read(bytes);
         String descriptor = new String(bytes);
         file.close();

         tx.commit();

         return descriptor;
      }
      catch (IOException e)
      {
         throw new VaultIOException("could not write or read vault descriptor file", e);
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }

   }

   public synchronized org.cleversafe.vault.Vault loadVault(
         UUID vaultIdentifier,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException, VaultSecurityException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         // check that vault exists in database
         getVaultFromDatabase(vaultIdentifier, session);

         // get account credentials
         Credentials credentials = manager.getCredentials(accountIdentifier);
         KeyPair keys = manager.getAccountKeyPair(accountIdentifier);

         InputStream in =
               new FileInputStream(_getDescriptorFile(this.directoryLocation, vaultIdentifier));
         Tuple2<org.cleversafe.vault.Vault, VaultACL> tuple =
               getVaultInstance(vaultIdentifier, in, credentials, keys);
         org.cleversafe.vault.Vault vault = tuple.getFirst();
         in.close();

         tx.commit();
         return vault;
      }
      catch (IOException e)
      {
         throw new VaultIOException("could not write or read vault descriptor file", e);
      }
      catch (CredentialsException e)
      {
         throw new VaultSecurityException("could not load account credentials", e);
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   public synchronized void renameVault(
         UUID vaultIdentifier,
         String newVaultName,
         UUID accountIdentifier,
         CredentialsManager manager) throws VaultException, VaultSecurityException
   {
      checkInitialization();
      Session session = getSessionFactory(this.directoryLocation).openSession();
      Transaction tx = session.beginTransaction();

      try
      {
         VaultState vault = getVaultFromDatabase(vaultIdentifier, session);

         Query q =
               session.createQuery("from VaultState v where v.name is :name").setParameter("name",
                     newVaultName);
         if (q.uniqueResult() != null)
            throw new VaultIOException("vault exists with duplicate name: " + newVaultName);

         vault.setName(newVaultName);
      }
      finally
      {
         tx.commit();
         session.close();
      }

   }

   /////////////////////////////////////////////////////////////////////////////////////////////////

   private static File _getACLFile(URI directory, UUID vaultIdentifier)
   {
      return new File(directory.getPath() + File.separator + LOCAL_FILE_VAULTS + File.separator
            + vaultIdentifier + ACL_EXTENSION);
   }

   private static File _getDescriptorFile(URI directory, UUID vaultIdentifier)
   {
      return new File(directory.getPath() + File.separator + LOCAL_FILE_VAULTS + File.separator
            + vaultIdentifier + DESCRIPTOR_EXTENSION);
   }

   protected VaultState getVaultFromDatabase(UUID vaultIdentifier, Session session)
         throws VaultIOException
   {
      Query q =
            session.createQuery("from VaultState v where v.identifier is :identifier").setParameter(
                  "identifier", vaultIdentifier.toString());
      if (q.uniqueResult() == null)
         throw new VaultIOException("no such vault exists: " + vaultIdentifier);
      return (VaultState) q.uniqueResult();
   }

   protected Tuple2<Vault, VaultACL> getVaultInstance(
         UUID vaultIdentifier,
         InputStream vaultDescriptor,
         Credentials credentials,
         KeyPair keypair) throws IOException, VaultIOException, VaultConfigurationException,
         VaultSecurityException, VaultDescriptorException, VaultACLException
   {
      // Load the vault ACL
      InputStream ain = new FileInputStream(_getACLFile(this.directoryLocation, vaultIdentifier));
      VaultACLFactory aclFactory = new VaultACLFactory();
      VaultACL acl = aclFactory.getInstance(ain);
      VaultACL.Entry entry = acl.getEntry(credentials.getAccountIdentifier(), keypair.getPrivate());
      ain.close();

      // Load vault descriptor
      ExecutionContext ctx = new ExecutionContext();
      ctx.add(VaultDescriptor.ACL_CTX_STRING, acl);
      ctx.add(VaultDescriptor.ACL_ENTRY_CTX_STRING, entry);
      ctx.add(VaultDescriptor.CREDENTIALS_CTX_STRING, credentials);

      InputStream din =
            new FileInputStream(_getDescriptorFile(this.directoryLocation, vaultIdentifier));
      VaultDescriptor vd;
      try
      {
         vd = new XMLVaultDescriptor(din, ctx);
      }
      catch (XmlException e)
      {
         throw new VaultDescriptorException("could not read vault descriptor");
      }
      din.close();

      Vault vault = vd.createVaultObject();
      vault.setVaultIdentifier(vaultIdentifier);

      return new Tuple2<org.cleversafe.vault.Vault, VaultACL>(vault, acl);
   }

   /////////////////////////////////////////////////////////////////////////////////////////////////
   // Setup methods

   private interface StoreExecuter
   {
      // Parameters should be ignored for methods that do not need them.
      void execute(
            SliceStore store,
            String vaultType,
            long maxSliceSize,
            long sliceStoreSize,
            VaultACL accessControlList,
            byte[] vaultDescriptorBytes) throws SliceStoreExistsException, SliceStoreIOException,
            SliceStoreNotFoundException;
   }

   /*
    * Executes a given "store executer" method on each store of a loaded vault with the given vault
    * identifier. Invoked by public vault operation methods. Callers must be synchronized.
    */
   private void executeOnStores(
         UUID vaultIdentifier,
         File vaultDescriptorFile,
         Credentials credentials,
         KeyPair keypair,
         StoreExecuter executer,
         boolean ignoreNotFound) throws VaultIOException, VaultConfigurationException,
         VaultSecurityException, VaultDescriptorException, VaultLoadException, VaultACLException
   {
      try
      {
         // FIXME: read vault descriptor in as a buffered input stream and reset instead of
         // re-opening file later.
         InputStream vaultDescriptor = new FileInputStream(vaultDescriptorFile);
         Tuple2<org.cleversafe.vault.Vault, VaultACL> tuple =
               getVaultInstance(vaultIdentifier, vaultDescriptor, credentials, keypair);
         vaultDescriptor.close();

         org.cleversafe.vault.Vault vault = tuple.getFirst();
         VaultACL acl = tuple.getSecond();

         // Note: the following code is specifically required by local file vault manager because
         // the vault directory elements must be propagated to all slice servers. This is a primary
         // disadvantage of using a local file directory. If the vaultDescriptor parameter is
         // null we assume the execute() methods does not need it (and pass null).
         vaultDescriptor = new FileInputStream(vaultDescriptorFile);
         byte[] vaultDescriptorByteArray = new byte[vaultDescriptor.available()];
         vaultDescriptor.read(vaultDescriptorByteArray);
         vaultDescriptor.close();

         // Connect to each slice server and execute desired method
         List<SliceStore> stores = vault.getSliceStores();
         for (int i = 0; i < stores.size(); i++)
         {
            try
            {
               // vault descriptor array is null if input stream was null.
               executer.execute(stores.get(i), vault.getType(), vault.getMaxSliceSize(),
                     vault.getSliceStoreSize(i), acl, vaultDescriptorByteArray);
            }
            catch (SliceStoreNotFoundException e)
            {
               if (ignoreNotFound)
                  _logger.warn("Server missing slice store for vault " + vaultIdentifier + ": "
                        + stores.get(i).toString());
               else
                  throw new VaultLoadException(e.getMessage(), e);
            }
            catch (SliceStoreIOException e)
            {
               if (ignoreNotFound)
                  _logger.warn("IO error with slice store for vault " + vaultIdentifier + ": "
                        + stores.get(i).toString());
               else
                  throw e;
            }
         }

      }
      catch (IOException e)
      {
         throw new VaultIOException("error reading vault configuration from file", e);
      }
      catch (SliceStoreIOException e)
      {
         throw new VaultIOException(e.getMessage(), e);
      }
      catch (SliceStoreExistsException e)
      {
         throw new VaultIOException(e.getMessage(), e);
      }
   }

   public synchronized void createVaultACL(
         UUID vaultIdentifier,
         File vaultDescriptorFile,
         Credentials credentials,
         PublicKey publicKey) throws VaultDescriptorException, VaultIOException,
         VaultKeyGenerationException, VaultConfigurationException
   {
      checkInitialization();
      // Resolve the vault ACL output path
      File output = _getACLFile(this.directoryLocation, vaultIdentifier);

      _logger.trace("Resolved ACL output file path: " + output.getAbsolutePath());

      // Create key information
      List<VaultKeyInfo> keyInfoList;
      try
      {
         InputStream in = new FileInputStream(vaultDescriptorFile);
         keyInfoList = VaultKeyInfoGenerator.generate(in);
         in.close();
      }
      catch (XmlException e)
      {
         throw new VaultDescriptorException("Unable to parse vault descriptor", e);
      }
      catch (IOException e)
      {
         throw new VaultIOException("Unable to read vault descriptor from directory", e);
      }
      catch (ConfigurationException e)
      {
         throw new VaultConfigurationException("Unable to load all libraries for vault", e);
      }
      catch (VaultDescriptorException e)
      {
         throw new VaultDescriptorException("Error in vault descriptor; " + e.getMessage(), e);

      }

      // Create and write ACL
      try
      {
         OutputStream out = new FileOutputStream(output);
         VaultACLFactory aclFactory = new VaultACLFactory();

         aclFactory.create(out, this.directoryLocation.getScheme(), // ACL TYPE argument (is "file")
               vaultIdentifier, credentials, publicKey, keyInfoList);
      }
      catch (IOException e)
      {
         throw new VaultIOException("unable to write ACL to file", e);
      }

   }

}
