//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Jun 15, 2007
//---------------------

package org.cleversafe.vault;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.cleversafe.codec.AbstractCodec;
import org.cleversafe.codec.Codec;
import org.cleversafe.codec.KeyedCodec;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.evaluator.Literal;
import org.cleversafe.config.evaluator.NamedEvaluator;
import org.cleversafe.config.evaluator.Parameter;
import org.cleversafe.config.evaluator.Reference;
import org.cleversafe.config.evaluator.ReferenceEvaluator;
import org.cleversafe.config.evaluator.SupportedTypes;
import org.cleversafe.config.evaluator.XMLEvaluator;
import org.cleversafe.config.evaluator.XMLValidator;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.Version;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;

/**
 * An XML implementation of {@link VaultDescriptor}. Loads a vault descriptor from an XML file.
 * 
 * @author Manish Motwani
 */
public class XMLVaultDescriptor implements VaultDescriptor
{
   private static final Logger _logger = Logger.getLogger(XMLVaultDescriptor.class);

   private ExecutionContext executionContext = null;

   public static final String VAULT_KIND = "Vault";

   private static final String VAULT_PREFIX = "vault";
   private static final String VAULT_NS = "http://cleversafe.org/vault";

   private static final String EVAL_PREFIX = "eval";
   private static final String EVAL_NS = "http://cleversafe.org/config/evaluator";

   private VaultDocument.Vault vaultDescriptor = null;

   private int datasourceCodecs = -1;

   public XMLVaultDescriptor()
   {
      this.vaultDescriptor = VaultDocument.Vault.Factory.newInstance();
   }

   /**
    * Creates a {@link VaultDescriptor} object from an XML input stream.
    * 
    * @param xmlInputStream
    *           XML input stream.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws VaultIOException
    */
   public XMLVaultDescriptor(final InputStream xmlInputStream) throws IOException, XmlException,
         VaultDescriptorException, VaultIOException
   {
      this(xmlInputStream, null);
   }

   /**
    * Creates a {@link VaultDescriptor} object from an XML file name.
    * 
    * @param xmlFileName
    *           XML file name.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws FileNotFoundException
    * @throws VaultIOException
    */
   public XMLVaultDescriptor(final String xmlFileName) throws IOException, XmlException,
         VaultDescriptorException, FileNotFoundException, VaultIOException
   {
      this(xmlFileName, null);
   }

   /**
    * Creates a {@link VaultDescriptor} object from an XML input stream and an
    * {@link ExecutionContext}.
    * 
    * @param xmlInputStream
    *           XML input stream.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws VaultIOException
    */
   public XMLVaultDescriptor(final InputStream xmlInputStream, final ExecutionContext ctx)
         throws IOException, XmlException, VaultDescriptorException, VaultIOException
   {
      this.executionContext = ctx;
      loadVaultFromXML(xmlInputStream);
   }

   /**
    * Creates a {@link VaultDescriptor} object from an XML file name and an {@link ExecutionContext}.
    * 
    * @param xmlFileName
    *           XML file name.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws FileNotFoundException
    * @throws VaultIOException
    */
   public XMLVaultDescriptor(final String xmlFileName, final ExecutionContext ctx)
         throws IOException, XmlException, VaultDescriptorException, FileNotFoundException,
         VaultIOException
   {
      this.executionContext = ctx;

      // Get input stream from file name
      InputStream xmlInputStream = null;
      xmlInputStream = new FileInputStream(xmlFileName);
      assert xmlInputStream != null;
      loadVaultFromXML(xmlInputStream);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getVaultType()
    */
   public String getVaultType()
   {
      return this.vaultDescriptor.getType();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getVaultParameters()
    */
   public List<NamedEvaluator> getVaultParameters() throws VaultDescriptorException
   {
      try
      {
         return XMLEvaluator.parseParameterArray(this.vaultDescriptor.getParamArray(),
               this.executionContext);
      }
      catch (ConfigurationException ce)
      {
         throw new VaultDescriptorException("Vault descriptor configuration error", ce);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getDatasourceCodecs()
    */
   public List<Codec> getDatasourceCodecs() throws VaultDescriptorException, VaultACLException
   {
      try
      {
         List<ReferenceEvaluator> codecReferences =
               XMLEvaluator.parseReferenceArray(this.vaultDescriptor.getCodecArray(),
                     this.executionContext);
         final List<Codec> codecs = new ArrayList<Codec>(codecReferences.size());

         int index = 0;
         for (final ReferenceEvaluator eval : codecReferences)
         {
            final Object codec = eval.evaluate();
            assert codec instanceof Codec;
            if (codec instanceof AbstractCodec)
               ((AbstractCodec) codec).setIndex(index);
            try
            {
               if (codec instanceof KeyedCodec)
                  ((KeyedCodec) codec).setACLEntry((VaultACL.Entry) this.executionContext.get(ACL_ENTRY_CTX_STRING));
            }
            catch (VaultException e)
            {
               final String msg =
                     "Error setting vault ACE in keyed codecs; "
                           + "probably could not load vault keys";
               _logger.debug(msg, e);
               throw new VaultACLException(msg, e);
            }

            codecs.add((Codec) codec);
            index++;
         }
         // Set datasource codec count for use by getSliceCodecs()
         this.datasourceCodecs = index;
         return codecs;
      }
      catch (ConfigurationException ce)
      {
         throw new VaultDescriptorException("Unable to load datasource codecs from descriptor", ce);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getSliceCodecs()
    */
   public List<Codec> getSliceCodecs() throws VaultDescriptorException, VaultACLException
   {
      if (this.datasourceCodecs < 0)
         getDatasourceCodecs(); // process datasource codecs to get accurate count
      return this.getSliceCodecs(this.datasourceCodecs);
   }

   private List<Codec> getSliceCodecs(int index) throws VaultDescriptorException, VaultACLException
   {
      try
      {
         List<ReferenceEvaluator> codecReferences =
               XMLEvaluator.parseReferenceArray(this.vaultDescriptor.getSliceCodecArray(),
                     this.executionContext);
         final List<Codec> codecs = new ArrayList<Codec>(codecReferences.size());

         for (final ReferenceEvaluator eval : codecReferences)
         {
            final Object codec = eval.evaluate();
            assert codec instanceof Codec;
            if (codec instanceof AbstractCodec)
               ((AbstractCodec) codec).setIndex(index);
            try
            {
               if (codec instanceof KeyedCodec)
                  ((KeyedCodec) codec).setACLEntry((VaultACL.Entry) this.executionContext.get(ACL_ENTRY_CTX_STRING));
            }
            catch (VaultException e)
            {
               final String msg =
                     "Error setting vault ACE in keyed codecs; "
                           + "probably could not load vault keys";
               _logger.debug(msg, e);
               throw new VaultACLException(msg, e);
            }
            index++;
            codecs.add((Codec) codec);
         }
         return codecs;
      }
      catch (ConfigurationException ce)
      {
         throw new VaultDescriptorException("Unable to load slice codecs from descriptor", ce);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getInformationDispersalCodec()
    */
   public InformationDispersalCodec getInformationDispersalCodec() throws VaultDescriptorException
   {
      try
      {
         ReferenceEvaluator idaReference =
               XMLEvaluator.parseReference(this.vaultDescriptor.getIda(), this.executionContext);
         final Object ida = idaReference.evaluate();
         assert ida instanceof InformationDispersalCodec;
         return (InformationDispersalCodec) ida;
      }
      catch (ConfigurationException ce)
      {
         throw new VaultDescriptorException("Unable to load IDA codec from descriptor", ce);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getSliceStores()
    */
   public List<SliceStore> getSliceStores() throws VaultDescriptorException
   {
      try
      {
         List<ReferenceEvaluator> sliceStoreReferences =
               XMLEvaluator.parseReferenceArray(this.vaultDescriptor.getSliceStoreArray(),
                     this.executionContext);
         final List<SliceStore> ssList = new ArrayList<SliceStore>(sliceStoreReferences.size());

         for (final ReferenceEvaluator eval : sliceStoreReferences)
         {
            final Object sliceStore = eval.evaluate();
            assert sliceStore instanceof SliceStore;
            ssList.add((SliceStore) sliceStore);
         }
         
         // Get the IDA and make sure that the width matches the number of slice stores
         if (this.getInformationDispersalCodec().getNumSlices() != ssList.size())
         {
            final String errmsg = String.format(
                  "Vault descriptor has %d slice stores; %d required for selected IDA configuration",
                  ssList.size(),
                  this.getInformationDispersalCodec().getNumSlices());
            _logger.error(errmsg);
            throw new VaultDescriptorException(errmsg);
         }
         
         return ssList;
      }
      catch (ConfigurationException ce)
      {
         throw new VaultDescriptorException("Unable to load slice stores from descriptor", ce);
      }
   }

   /**
    * 
    * @param xmlInputStream
    *           XML input stream.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws VaultIOException
    */
   public static VaultDescriptor getInstance(
         final InputStream xmlInputStream,
         final ExecutionContext ctx) throws IOException, XmlException, VaultDescriptorException,
         VaultIOException
   {
      return new XMLVaultDescriptor(xmlInputStream, ctx);
   }

   /**
    * 
    * @param xmlFileName
    *           XML file name.
    * @param ctx
    *           {@link ExecutionContext} object that contains a map of named objects as
    *           string-object pairs.
    * @return
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws FileNotFoundException
    * @throws VaultIOException
    */
   public static VaultDescriptor getInstance(final String xmlFileName, final ExecutionContext ctx)
         throws IOException, XmlException, VaultDescriptorException, FileNotFoundException,
         VaultIOException
   {
      return new XMLVaultDescriptor(xmlFileName, ctx);
   }

   /**
    * 
    * @param xmlInputStream
    *           XML input stream.
    * @return
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws VaultIOException
    */
   public static VaultDescriptor getInstance(final InputStream xmlInputStream) throws IOException,
         XmlException, VaultDescriptorException, VaultIOException
   {
      return new XMLVaultDescriptor(xmlInputStream);
   }

   /**
    * 
    * @param xmlFileName
    *           XML file name.
    * @return
    * @throws IOException
    * @throws XmlException
    * @throws VaultDescriptorException
    * @throws FileNotFoundException
    * @throws VaultIOException
    */
   public static VaultDescriptor getInstance(final String xmlFileName) throws IOException,
         XmlException, VaultDescriptorException, FileNotFoundException, VaultIOException
   {
      return new XMLVaultDescriptor(xmlFileName);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#getExecutionContext()
    */
   public ExecutionContext getExecutionContext()
   {
      return this.executionContext;
   }

   /*
    * There is price for all short comings. Here the price is paid for slice store inclusion in
    * vault definition IT IS FIXED IN >1.0
    * 
    * @see org.cleversafe.vault.VaultDescriptor#createVaultObject()
    */
   public BaseVault createNoSliceStoreVaultObject() throws VaultDescriptorException,
         VaultACLException
   {
      // Create the vault
      try
      {
         XMLValidator.validateKind(new String[]{
            VAULT_KIND
         }, true, this.vaultDescriptor.getType(), this.vaultDescriptor.getParamArray(),
               this.executionContext);

         ReferenceEvaluator vaultEvaluator =
               XMLEvaluator.createReferenceEvaluatorFromKind(new String[]{
                  VAULT_KIND,
               }, true, getVaultType(), null, getVaultParameters());

         Object vaultObject = vaultEvaluator.evaluateWithoutInitialization();
         assert vaultObject instanceof BaseVault;

         BaseVault vault = (BaseVault) vaultObject;

         // Set the common vault attributes
         vault.setDatasourceCodecs(getDatasourceCodecs());
         vault.setSliceCodecs(getSliceCodecs());
         vault.setInformationDispersalCodec(getInformationDispersalCodec());

         vault.initialize();
         return vault;
      }
      catch (Exception ce)
      {
         throw new VaultDescriptorException("Could not create vault object", ce);
      }
   }

   public BaseVault createVaultObject() throws VaultDescriptorException, VaultACLException
   {
      // Optimization vault characteristics
      BaseVault vault = createNoSliceStoreVaultObject();
      vault.setSliceStores(getSliceStores());
      vault.optimizeVault();

      return vault;
   }

   private void loadVaultFromXML(InputStream xmlInputStream) throws VaultDescriptorException,
         XmlException, IOException
   {
      _logger.debug("Loading vault from XML vault descriptor");

      VaultDocument vd = VaultDocument.Factory.parse(xmlInputStream);

      // XMLBeans validation
      if (!vd.validate())
      {
         throw new VaultDescriptorException("Vault descriptor validation failed");
      }

      // Check if the vault descriptor IDA has a version attribute defined      
      if (!vd.getVault().getIda().isSetVersion())
      {
         throw new VaultDescriptorException(
               "The IDA in the vault descriptor must have a version number defined!");
      }

      // Check if the vault descriptor datasource codecs have a version attribute defined
      for (Reference codec : vd.getVault().getCodecArray())
      {
         if (!codec.isSetVersion())
         {
            throw new VaultDescriptorException(
                  "All codecs in vault descriptor must have version number defined!");
         }
      }

      // Check if the vault descriptor slice codecs have a version attribute defined
      for (Reference sliceCodec : vd.getVault().getSliceCodecArray())
      {
         if (!sliceCodec.isSetVersion())
         {
            throw new VaultDescriptorException(
                  "All slice-codecs in vault descriptor must have version number defined!");
         }
      }

      // Load vault descriptor as a local member
      this.vaultDescriptor = vd.getVault();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.vault.VaultDescriptor#writeDescriptor(java.io.File)
    */
   public void writeDescriptor(File loc) throws VaultDescriptorException, VaultIOException
   {
      // Set the namespace prefixes for elements (i.e. <vault:vault... appropriately
      Map<String, String> optionsMap = new HashMap<String, String>();
      optionsMap.put(VAULT_NS, VAULT_PREFIX);
      optionsMap.put(EVAL_NS, EVAL_PREFIX);
      XmlOptions opts = new XmlOptions();
      opts.setSaveSuggestedPrefixes(optionsMap);

      // If this.vaultDescriptor is written directly, the vault tag comes out as <xml-fragment/>
      // instead of <vault:vault/>
      VaultDocument doc = VaultDocument.Factory.newInstance();
      doc.setVault(this.vaultDescriptor);
      if (!doc.validate())
      {
         throw new VaultDescriptorException("Could not validate XML document");
      }

      try
      {
         doc.save(loc, opts);
      }
      catch (IOException e)
      {
         throw new VaultIOException("Could not write vault descriptor", e);
      }
   }

   /**
    * Set the type of vault in the XML.
    * 
    * @param type
    */
   public void setVaultType(String type)
   {
      this.vaultDescriptor.setType(type);
   }

   /**
    * Add a literal parameter to a Vault.
    * 
    * @param name
    *           The name of the parameter
    * @param dataType
    *           The data type of the parameter
    * @param value
    *           The value of the parameter, encoded as a string.
    */
   public void addLiteralVaultParam(String name, SupportedTypes.Enum dataType, String value)
   {
      Parameter param = this.vaultDescriptor.addNewParam();
      addLiteralParam(param, name, dataType, value);
   }

   /**
    * Clear the list of literal vault parameters.
    * 
    */
   public void clearLiteralVaultParams()
   {
      this.vaultDescriptor.setParamArray(null);
   }

   /**
    * Set the IDA kind and referral.
    * 
    * @param name
    *           The name of the parameter
    * @param dataType
    *           The data type of the parameter
    * @param value
    *           The value of the parameter, encoded as a string.
    * @param version
    *           The version of the IDA
    */
   public void setIDA(String kind, String referral, Version version)
   {
      // Ensure that the document has one and only one IDA.
      Reference ida = this.vaultDescriptor.getIda();
      if (ida == null)
      {
         ida = this.vaultDescriptor.addNewIda();
      }

      ida.setParamArray(null);
      ida.setKind(kind);
      ida.setReferral(referral);
      ida.setVersion(version.toString());
   }

   /**
    * Add a parameter to the IDA. An IDA must be defined already.
    * 
    * @param name
    * @param dataType
    * @param value
    * @throws VaultDescriptorException
    */
   public void addLiteralIDAParam(String name, SupportedTypes.Enum dataType, String value)
         throws VaultDescriptorException
   {
      if (this.vaultDescriptor.getIda() == null)
         throw new VaultDescriptorException("Attempted to add param to IDA which does not exist");
      Parameter param = this.vaultDescriptor.getIda().addNewParam();
      addLiteralParam(param, name, dataType, value);
   }

   /**
    * Add a codec to the vault descriptor.
    * 
    * @param kind
    *           Kind of the codec
    * @param referral
    *           Referral of the codec
    * @param parameters
    *           Map containing all the parameters for the codec
    * @param version
    *           Codec version
    */
   public void addCodec(
         String kind,
         String referral,
         Map<String, Tuple2<SupportedTypes.Enum, String>> parameters,
         Version version)
   {
      Reference codec = this.vaultDescriptor.addNewCodec();
      codec.setKind(kind);
      codec.setReferral(referral);
      codec.setVersion(version.toString());

      addParamsToReference(codec, parameters);
   }

   /**
    * Clear the list of pre-dispersal codecs.
    * 
    */
   public void clearAllCodecs()
   {
      this.vaultDescriptor.setCodecArray(null);
   }

   /**
    * Add a slice codec to the vault descriptor.
    * 
    * @param kind
    *           Kind of the codec
    * @param referral
    *           Referral of the codec
    * @param parameters
    *           Map containing all the parameters for the codec
    * @param version
    *           Codec version
    */
   public void addSliceCodec(
         String kind,
         String referral,
         Map<String, Tuple2<SupportedTypes.Enum, String>> parameters,
         Version version)
   {
      Reference codec = this.vaultDescriptor.addNewSliceCodec();
      codec.setKind(kind);
      codec.setReferral(referral);
      codec.setVersion(version.toString());

      addParamsToReference(codec, parameters);
   }

   /**
    * Clear the list of slice codecs.
    * 
    */
   public void clearAllSliceCodecs()
   {
      this.vaultDescriptor.setSliceCodecArray(null);
   }

   /**
    * Add a slice store to the vault descriptor.
    * 
    * @param kind
    *           Kind of the slice store
    * @param referral
    *           Referral of the slice store
    * @param parameters
    *           Map containing all the literal parameters for the slice store
    */
   public void addSliceStore(
         String kind,
         String referral,
         Map<String, Tuple2<SupportedTypes.Enum, String>> parameters)
   {
      Reference store = this.vaultDescriptor.addNewSliceStore();
      store.setKind(kind);
      store.setReferral(referral);

      addParamsToReference(store, parameters);
   }

   /**
    * Clear the list of Slice Stores.
    * 
    */
   public void clearAllSliceStores()
   {
      this.vaultDescriptor.setSliceStoreArray(null);
   }

   /**
    * Helper method which takes a reference object and adds all the parameters in a parameters map
    * to it.
    * 
    * @param ref
    *           Reference object to which to add the parameters
    * @param parameters
    *           Map containing all the literal parameters
    */
   private void addParamsToReference(
         Reference ref,
         Map<String, Tuple2<SupportedTypes.Enum, String>> parameters)
   {
      if (parameters == null)
         return;

      Iterator<String> itr = parameters.keySet().iterator();
      while (itr.hasNext())
      {
         String key = itr.next();
         Tuple2<SupportedTypes.Enum, String> value = parameters.get(key);
         Parameter param = ref.addNewParam();
         addLiteralParam(param, key, value.getFirst(), value.getSecond());
      }
   }

   /**
    * Helper method which converts a name, value, type tuple, converts to a Literal and adds it to a
    * Parameter.
    * 
    * @param param
    *           Parameter to which to add the literal
    * @param name
    *           Name of the literal
    * @param dataType
    *           Data type of the literal
    * @param value
    *           Value of the literal, encoded as a Java String
    */
   private void addLiteralParam(
         Parameter param,
         String name,
         SupportedTypes.Enum dataType,
         String value)
   {
      param.setName(name);
      Literal lit = param.addNewLiteral();
      lit.setValue(value);
      lit.setType(dataType);
   }
}
