//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 25, 2007
//---------------------

package org.cleversafe.vault;

import java.security.Permission;
import java.util.StringTokenizer;
import java.util.UUID;

import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;

/**
 * Describes vault access permissions assigned to a particular grid account.
 * 
 */
public class VaultPermission extends Permission
{
   private static final long serialVersionUID = -2488866894428977851L;

   private VaultPermissionEntry entry;
   
   public static VaultPermission getFullPermissions(UUID vaultIdentifier)
   {
      return new VaultPermission(vaultIdentifier, VaultPermissionEntry.getFullPermission());
   }

   /**
    * Allows an account to read slices from a vault. Implies the "list" permission.
    */
   public static final String READ = "read";

   /**
    * Allows an account to write slices to and remove slices from a vault. Implies the "read"
    * permission.
    */
   public static final String WRITE = "write";

   /**
    * Allows an account to read vault metadata and list slices.
    */
   public static final String LIST = "list";

   /**
    * Allows an account to rebuild damaged or missing slices. Implies the "list" permission.
    * <p>
    * Accounts with the "rebuild" permission can send and receive slices from slice servers but may
    * not decrypt data after assembling the slices.
    */
   public static final String REBUILD = "rebuild";

   /**
    * Allows an account to delete a vault from the grid. Note that individual slices can still be
    * deleted given the "write" permission.
    */
   public static final String DELETE_VAULT = "deleteVault";

   /**
    * Allows an account to modify the vault owner and permissions for other accounts.
    */
   public static final String MODIFY_ACL = "modifyACL";

   /**
    * Parses the action string.
    * 
    * An example string giving read, write, and list permissions: "read, write, list".
    * 
    * @param action
    *           String in comma se
    */
   private void parse(String action)
   {
      StringTokenizer st = new StringTokenizer(action, ",\t ");

      this.entry = new VaultPermissionEntry();

      while (st.hasMoreElements())
      {
         String tok = st.nextToken();
         if (tok.equals(VaultPermission.READ))
         {
            entry.setRead(true);
            entry.setList(true);
         }
         else if (tok.equals(VaultPermission.WRITE))
         {
            entry.setWrite(true);
            entry.setRead(true);
            entry.setList(true);
         }
         else if (tok.equals(VaultPermission.LIST))
         {
            entry.setList(true);
         }
         else if (tok.equals(VaultPermission.REBUILD))
         {
            entry.setRebuild(true);
            entry.setList(true);
         }
         else if (tok.equals(VaultPermission.DELETE_VAULT))
         {
            entry.setDeleteVault(true);
         }
         else if (tok.equals(VaultPermission.MODIFY_ACL))
         {
            entry.setModifyACL(true);
         }
         else
         {
            throw new IllegalArgumentException("Unknown action " + tok);
         }
      }
   }

   /**
    * Constructs a permission object for the indicated vault. Action implications are not guaranteed
    * to remain constant. Thus all permitted actions should be indicated in the action string. If
    * one action implies another they should both indicated.
    * 
    * @param name
    *           The vault identifier. The string "*" indicates all vaults on the slice server.
    * @param action
    *           The actions allowed. For example: "read, write, list".
    */
   public VaultPermission(String name, String action)
   {
      super(name);
      parse(action);
   }

   protected VaultPermission(UUID vaultIdentifier, VaultPermissionEntry permission)
   {
      super(vaultIdentifier.toString());
      this.entry = new VaultPermissionEntry(permission);
   }

   protected VaultPermissionEntry getEntry()
   {
      return this.entry;
   }

   @Override
   public String getActions()
   {
      String sep = ", ";
      String actions = "";

      // This is the canonical order.
      if (entry.canRead())
      {
         actions = READ + sep;
      }
      if (entry.canWrite())
      {
         actions += WRITE + sep;
      }
      if (entry.canList())
      {
         actions += LIST + sep;
      }
      if (entry.canRebuild())
      {
         actions += REBUILD + sep;
      }
      if (entry.canDeleteVault())
      {
         actions += DELETE_VAULT + sep;
      }
      if (entry.canModifyACL())
      {
         actions += MODIFY_ACL + sep;
      }
      // take off last separator
      return actions.substring(0, actions.length() - sep.length());
   }

   @Override
   public boolean implies(Permission permission)
   {
      if (permission == null)
      {
         return false;
      }

      if (!(permission instanceof VaultPermission))
      {
         return false;
      }

      VaultPermission p = (VaultPermission) permission;

      // The name "*" indicates all vaults on the slice server. This can be used to give access
      // to administrative accounts and for testing and development.
      if (!this.getName().equals("*") && !this.getName().equals(p.getName()))
      {
         return false;
      }

      // Generally, if p can do something that this can't the implication is false

      if (p.entry.canModifyACL() && !this.entry.canModifyACL())
      {
         return false;
      }

      if (p.entry.canDeleteVault() && !this.entry.canDeleteVault())
      {
         return false;
      }

      if (p.entry.canWrite() && !this.entry.canWrite())
      {
         return false;
      }

      if (p.entry.canRead() && !this.entry.canRead())
      {
         return false;
      }

      if (p.entry.canList() && !this.entry.canList())
      {
         return false;
      }

      if (p.entry.canRebuild() && !this.entry.canRebuild())
      {
         return false;
      }

      return true;
   }

   @Override
   public String toString()
   {
      return "(\"" + VaultPermission.class.toString() + "\",\"" + getName() + "\",\""
            + getActions() + "\")";
   }

   @Override
   public int hashCode()
   {
      final int PRIME = 31;
      int result = 1;
      result = PRIME * result + ((entry == null) ? 0 : entry.hashCode());
      return result;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;
      if (obj == null)
         return false;
      if (getClass() != obj.getClass())
         return false;
      final VaultPermission other = (VaultPermission) obj;
      if (entry == null)
      {
         if (other.entry != null)
            return false;
      }
      else if (!entry.equals(other.entry))
         return false;
      return true;
   }
}
