//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 15, 2007
//---------------------

package org.cleversafe.vault;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.KeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.cleversafe.authentication.Credentials;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultKeyGenerationException;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyUsage;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;
import org.cleversafe.vault.storage.asn1.KeyInfo.KeyInfoPair;

// TODO: Describe class or interface
public class VaultACLFactory
{
   private static final String ACL_TYPE_PROPERTY = "org.cleversafe.vault.acl.type";
   private static final String ACL_TYPE = "file";

   private static final String WRAP_TRANSFORMATION_PROPERTY =
         "org.cleversafe.vault.keys.wrap.transformation";
   private static String WRAP_TRANSFORMATION = "RSA/ECB/PKCS1Padding";

   static
   {
      // TODO: Replace this with configuration framework lookup.
      WRAP_TRANSFORMATION = System.getProperty(WRAP_TRANSFORMATION_PROPERTY, WRAP_TRANSFORMATION);
   }

   private VaultACL generate(
         URI vaultLocation,
         UUID vaultIdentifier,
         Credentials credentials,
         PublicKey publicKey,
         List<VaultKeyInfo> keyInfoList) throws VaultConfigurationException,
         VaultKeyGenerationException
   {
      Map<Integer, PlainKeyInfo> publicKeys = new HashMap<Integer, PlainKeyInfo>();
      Map<Integer, EncryptedKeyInfo> entryKeys = new HashMap<Integer, EncryptedKeyInfo>();

      // Generate the vault keys and place them in public and entry key maps.
      for (VaultKeyInfo keyInfo : keyInfoList)
      {
         if (keyInfo.getUsage() == KeyUsage.DATASOURCE_VERIFICATION
               || keyInfo.getUsage() == KeyUsage.SLICE_VERIFICATION)
         {
            // We generate a key pair for verification keys
            try
            {
               KeyInfoPair keyPair =
                     PlainKeyInfo.generateKeyPair(keyInfo.getAlgorithm(), keyInfo.getSize(),
                           keyInfo.getUsage());
               // store public key
               publicKeys.put(keyInfo.getStorageIndex(), (PlainKeyInfo) keyPair.getPublic());
               // encrypt private key
               EncryptedKeyInfo privateKey =
                     EncryptedKeyInfo.wrap((PlainKeyInfo) keyPair.getPrivate(), publicKey,
                           WRAP_TRANSFORMATION);
               // store encrypted private key
               entryKeys.put(keyInfo.getStorageIndex(), privateKey);

            }
            catch (NoSuchAlgorithmException e)
            {
               throw new VaultKeyGenerationException("no such asymmetric algorithm: "
                     + keyInfo.getAlgorithm() + " for key index " + keyInfo.getStorageIndex()
                     + "; see vault descriptor", e);
            }
            catch (KeyException e)
            {
               throw new VaultKeyGenerationException("could not wrap vault key index "
                     + keyInfo.getStorageIndex() + "; see vault descriptor", e);
            }
            catch (GeneralSecurityException e)
            {
               throw new VaultKeyGenerationException("could not wrap vault key index "
                     + keyInfo.getStorageIndex() + "; see vault descriptor", e);
            }
         }
         else
         {
            // for other key usages we generate symmetric keys
            try
            {
               PlainKeyInfo key =
                     PlainKeyInfo.generateKey(keyInfo.getAlgorithm(), keyInfo.getSize(),
                           keyInfo.getUsage());
               // encrypt secret key
               EncryptedKeyInfo secretKey =
                     EncryptedKeyInfo.wrap(key, publicKey, WRAP_TRANSFORMATION);
               // store encrypted secret key
               entryKeys.put(keyInfo.getStorageIndex(), secretKey);
            }
            catch (NoSuchAlgorithmException e)
            {
               throw new VaultKeyGenerationException("no such symmetric algorithm: "
                     + keyInfo.getAlgorithm() + " for key index " + keyInfo.getStorageIndex()
                     + "; see vault descriptor", e);
            }
            catch (KeyException e)
            {
               throw new VaultKeyGenerationException("could not wrap vault key index "
                     + keyInfo.getStorageIndex() + "; see vault descriptor", e);
            }
            catch (GeneralSecurityException e)
            {
               throw new VaultKeyGenerationException("could not wrap vault key index "
                     + keyInfo.getStorageIndex() + "; see vault descriptor", e);
            }
         }
      }

      // Create a full permissions object

      VaultPermissionEntry full = VaultPermissionEntry.getFullPermission();

      // Create the VaultACL and populate it
      VaultACL acl;
      try
      {
         acl =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     VaultACL.class);
      }
      catch (ConfigurationItemNotDefinedException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInstantiationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInitializationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ConfigurationLoadException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ClassCastException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }

      acl.storeACLEntry(credentials.getAccountIdentifier(), full, entryKeys);
      acl.storePublicKeys(publicKeys);
      acl.storeOwner(credentials.getAccountIdentifier());
      acl.storeVaultIdentifier(vaultIdentifier);

      return acl;
   }

   /**
    * Creates a new Vault ACL.
    * <p>
    * The new ACL will have a single entry belonging to the owner, whose identifier is taken from
    * the given credentials object.
    * <p>
    * The ACL type (storage format) is indicated by the URI scheme (or "protocol").
    * 
    * @param vaultLocation
    *           The location where vaults are stored.
    * @param vaultIdentifier
    *           The Vault UUID.
    * @param credentials
    *           The credentials identifying the vault owner.
    * @param publicKey
    *           The grid account public key for the account identified by the credentials.
    * @param keyInfoList
    *           A list of vault key parameters. These keys will be generated and stored in the ACL.
    */
   public void create(
         URI vaultLocation,
         UUID vaultIdentifier,
         Credentials credentials,
         PublicKey publicKey,
         List<VaultKeyInfo> keyInfoList) throws VaultException, VaultConfigurationException,
         VaultKeyGenerationException
   {
      VaultACL acl = generate(vaultLocation, vaultIdentifier, credentials, publicKey, keyInfoList);

      acl.flush(vaultLocation);
   }

   /**
    * Creates a new Vault ACL.
    * <p>
    * The new ACL will have a single entry belonging to the owner, whose identifier is taken from
    * the given credentials object.
    * <p>
    * The ACL type (storage format) is indicated by the system property
    * <code>org.cleversafe.vault.acl.type</code>.
    * 
    * @param out
    *           An output stream where the ACL will be written.
    * @param vaultIdentifier
    *           The Vault UUID.
    * @param credentials
    *           The credentials identifying the vault owner.
    * @param publicKey
    *           The grid account public key for the account identified by the credentials.
    * @param keyInfoList
    *           A list of vault key parameters. These keys will be generated and stored in the ACL.
    */
   public void create(
         OutputStream out,
         UUID vaultIdentifier,
         Credentials credentials,
         PublicKey publicKey,
         List<VaultKeyInfo> keyInfoList) throws VaultException, VaultConfigurationException,
         VaultKeyGenerationException
   {
      String aclType = System.getProperty(ACL_TYPE_PROPERTY, ACL_TYPE);
      create(out, aclType, vaultIdentifier, credentials, publicKey, keyInfoList);
   }

   /**
    * Creates a new Vault ACL.
    * <p>
    * The new ACL will have a single entry belonging to the owner, whose identifier is taken from
    * the given credentials object.
    * 
    * @param out
    *           An output stream where the ACL will be written.
    * @param aclType
    *           The type of ACL to create. E.g., "file" for a file-based ACL
    * @param vaultIdentifier
    *           The Vault UUID.
    * @param credentials
    *           The credentials identifying the vault owner.
    * @param publicKey
    *           The grid account public key for the account identified by the credentials.
    * @param keyInfoList
    *           A list of vault key parameters. These keys will be generated and stored in the ACL.
    */
   public void create(
         OutputStream out,
         String aclType,
         UUID vaultIdentifier,
         Credentials credentials,
         PublicKey publicKey,
         List<VaultKeyInfo> keyInfoList) throws VaultConfigurationException,
         VaultKeyGenerationException, VaultIOException
   {
      URI uri;
      try
      {
         uri = new URI(aclType + "://null");
      }
      catch (URISyntaxException e)
      {
         throw new VaultConfigurationException("Bad acl storage type: " + aclType, e);
      }
      VaultACL acl = generate(uri, vaultIdentifier, credentials, publicKey, keyInfoList);
      acl.flush(out);
   }

   public VaultACL getInstance(URI vaultLocation, UUID vaultIdentifier) throws VaultIOException,
         VaultConfigurationException
   {
      try
      {
         VaultACL acl =
               (VaultACL) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("VaultACL",
                     vaultLocation.getScheme());

         acl.loadFromURI(vaultLocation, vaultIdentifier);
         return acl;
      }

      catch (ConfigurationItemNotDefinedException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInstantiationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInitializationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ConfigurationLoadException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ClassCastException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
   }

   /**
    * Instantiates a VaultACL object from an input stream.
    * 
    * @param in
    *           An input stream containing a serialized VaultACL.
    * @throws IOException
    *            If the ACL could not be recovered from the stream.
    */
   public VaultACL getInstance(InputStream in) throws VaultIOException, VaultConfigurationException
   {
      // TODO: Implement vault ACL type detection from input stream. This would allow us to always
      //       be able to recover the ACL regardless of the implementation.
      VaultACL acl;
      try
      {
         acl =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     VaultACL.class);
      }
      catch (ConfigurationItemNotDefinedException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInstantiationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ObjectInitializationException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ConfigurationLoadException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }
      catch (ClassCastException e1)
      {
         throw new VaultConfigurationException("Error configuring VaultACLFactory", e1);
      }

      acl.loadFromStream(in);

      return acl;
   }

   public VaultACL getInstance(byte[] acl) throws VaultIOException, VaultConfigurationException
   {
      return this.getInstance(new ByteArrayInputStream(acl));
   }

}
