//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: May 1, 2008
//---------------------

package org.cleversafe.util.rangelock;

import java.util.LinkedList;
import java.util.List;

//import org.apache.log4j.Logger;

/**
  * This implementation favors write, Read will not proceed if it overlaps
  * with existing writer or a waiting writer
  * It potentially could create a situation that a reader will be in infinite
  * 'wait' as overlapping writers may arrive 
 */
public class RangeReadWriteLockWritePreference implements RangeReadWriteLock
{

   /**
    * Read lock with range implementation
    */
   public class ReadLock extends LockBase
   {
      public ReadLock(long start, long amount)
      {
         super(start, amount);
      }

      /**
       * Acquires a lock as long as there no writers or waiting writers in the range
       */
      public void acquire()
      {
         synchronized (RangeReadWriteLockWritePreference.this.globalSynch)
         {
            RangeReadWriteLockWritePreference.this.readWaitQueue.add(this);
            // While writers in the same range exist or potential writers waiting for this range
            while (rangeBusy(RangeReadWriteLockWritePreference.this.activeWriters, this)
                  || rangeBusy(RangeReadWriteLockWritePreference.this.writeWaitQueue, this))
            {
               try
               {
                  RangeReadWriteLockWritePreference.this.globalSynch.wait();
               }
               catch (final InterruptedException ignore)
               {
               }
            }
            RangeReadWriteLockWritePreference.this.readWaitQueue.remove(this);
            RangeReadWriteLockWritePreference.this.activeReaders.add(this);
         }
      }

      /**
       * Releases lock
       */
      public void release()
      {
         synchronized (RangeReadWriteLockWritePreference.this.globalSynch)
         {
            RangeReadWriteLockWritePreference.this.activeReaders.remove(this);
            RangeReadWriteLockWritePreference.this.globalSynch.notifyAll();
         }
      }

   }
   /**
    * Write lock with range implementation
    */
   public class WriteLock extends LockBase
   {
      public WriteLock(long start, long amount)
      {
         super(start, amount);
      }

      /**
       * Acquires a lock as long as there no active readers or writers in required range
       */
      public void acquire()
      {
         synchronized (RangeReadWriteLockWritePreference.this.globalSynch)
         {
            RangeReadWriteLockWritePreference.this.writeWaitQueue.add(this);
            // There is a potential for unfair scheduling, as a new write may sneak in before another waiting
            // and block it at later point
            while (rangeBusy(RangeReadWriteLockWritePreference.this.activeReaders, this)
                  || rangeBusy(RangeReadWriteLockWritePreference.this.activeWriters, this))
            {
               try
               {
                  RangeReadWriteLockWritePreference.this.globalSynch.wait();
               }
               catch (final InterruptedException ignore)
               {
               }
            }
            RangeReadWriteLockWritePreference.this.writeWaitQueue.remove(this);
            RangeReadWriteLockWritePreference.this.activeWriters.add(this);
         }
      }

      /**
       * Releases lock
       */
      public void release()
      {
         synchronized (RangeReadWriteLockWritePreference.this.globalSynch)
         {
            RangeReadWriteLockWritePreference.this.activeWriters.remove(this);
            RangeReadWriteLockWritePreference.this.globalSynch.notifyAll();
         }
      }

   }
   // Waiters to obtain locks
   private final List<ReadLock> readWaitQueue = new LinkedList<ReadLock>();
   private final List<WriteLock> writeWaitQueue = new LinkedList<WriteLock>();

   // Lock holders
   private final List<WriteLock> activeWriters = new LinkedList<WriteLock>();
   private final List<ReadLock> activeReaders = new LinkedList<ReadLock>();

   private final Object globalSynch = new Object();

   public RangeReadWriteLockWritePreference()
   {
   }

   private boolean rangeBusy(List<? extends LockBase> busyList, LockBase candidate)
   {
      // Should be called under lock
      for (final LockBase l : busyList)
      {
         if (l.intersect(candidate))
         {
            return true;
         }
      }
      return false;
   }

   /**
    * Lock for read
    */
   public RangeReadWriteLock.Lock getWriteRangeLock(long start, long amount)
   {
      return new WriteLock(start, amount);
   }

   /**
    * Lock for write
    */
   public RangeReadWriteLock.Lock getReadRangeLock(long start, long amount)
   {
      return new ReadLock(start, amount);
   }

}
