//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: May 3, 2008
//---------------------

package org.cleversafe.util.rangelock;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.apache.log4j.Logger;

public class RangeReadWriteLockFair implements RangeReadWriteLock
{
   static private Logger _logger = Logger.getLogger(RangeReadWriteLockFair.class);
   /**
    * Read lock with range implementation
    */
   public class ReadLock extends LockBase
   {
      public ReadLock(long start, long amount)
      {
         super(start, amount);
      }

      /**
       * Acquires a lock as long as there no writers or waiting writers in the range
       */
      public void acquire()
      {
         synchronized (RangeReadWriteLockFair.this.globalSynch)
         {
            // While writers in the same range exist or potential writers waiting for this range
            int myIndex = 0;
            if (!RangeReadWriteLockFair.this.waitQueue.isEmpty())
            {
               myIndex = RangeReadWriteLockFair.this.waitQueue.lastKey() + 1;
            }
            if (_logger.isTraceEnabled())
               _logger.trace("Read:my index = " + myIndex);
            RangeReadWriteLockFair.this.waitQueue.put(myIndex, this);

            // Wait until no conflicting writers and we have lowest index
            // after this happen other readers would have a chance to get lock
            while (RangeReadWriteLockFair.this.rangeBusy(RangeReadWriteLockFair.this.activeWriters,
                  this)
                  || myIndex != RangeReadWriteLockFair.this.waitQueue.firstKey())
            {
               try
               {
                  RangeReadWriteLockFair.this.globalSynch.wait();
                  //_logger.debug("Read:my index = " + myIndex + ", min index=" + RangeReadWriteLockFair.this.waitQueue.firstKey());                 
               }
               catch (final InterruptedException ignore)
               {
               }
            }
            RangeReadWriteLockFair.this.waitQueue.remove(myIndex);
            RangeReadWriteLockFair.this.activeReaders.add(this);
            // Some readers may become first and start
            RangeReadWriteLockFair.this.globalSynch.notifyAll();
         }
      }

      /**
       * Releases lock
       */
      public void release()
      {
         synchronized (RangeReadWriteLockFair.this.globalSynch)
         {
            RangeReadWriteLockFair.this.activeReaders.remove(this);
            RangeReadWriteLockFair.this.globalSynch.notifyAll();
         }
      }
   }

   public class WriteLock extends LockBase
   {
      public WriteLock(long start, long amount)
      {
         super(start, amount);
      }

      /**
       * 
       */
      public void acquire()
      {
         synchronized (RangeReadWriteLockFair.this.globalSynch)
         {
            // While writers in the same range exist or potential writers waiting for this range
            int myIndex = 0;
            if (!RangeReadWriteLockFair.this.waitQueue.isEmpty())
            {
               myIndex = RangeReadWriteLockFair.this.waitQueue.lastKey() + 1;
            }
            if (_logger.isTraceEnabled())
               _logger.trace("Write:my index = " + myIndex);
            RangeReadWriteLockFair.this.waitQueue.put(myIndex, this);

            // Wait until no conflicting writers and readers 
            while (true)
            {
               boolean proceed = false;              
               // If range is not busy we have a chance
               if (!RangeReadWriteLockFair.this.rangeBusy(
                     RangeReadWriteLockFair.this.activeWriters, this) && !RangeReadWriteLockFair.this.rangeBusy(
                     RangeReadWriteLockFair.this.activeReaders, this))
               {
                  // first candidate always wins
                  if (myIndex == RangeReadWriteLockFair.this.waitQueue.firstKey())
                  {
                     proceed = true;
                  }
                  // Not first but no readers ahead is also OK
                  // Fairness is only towards readers
                  else {
                     final SortedMap<Integer, RangeReadWriteLock.Lock> higherPriorityWaiters = waitQueue.headMap(myIndex);
                     assert higherPriorityWaiters != null;
                     Collection<RangeReadWriteLock.Lock> others = higherPriorityWaiters.values();

                     // Do we have readers ahead?
                     proceed = true;
                     for (RangeReadWriteLock.Lock other : others)
                     {
                        if (other instanceof ReadLock)
                        {
                           proceed = false;
                           break;
                        }
                     }
                  }

               }
               if (proceed)
               {
                  break;
               }
               try
               {
                  RangeReadWriteLockFair.this.globalSynch.wait();
               }
               catch (final InterruptedException ignore)
               {
               }

            }
            // Acquired
            RangeReadWriteLockFair.this.waitQueue.remove(myIndex);
            RangeReadWriteLockFair.this.activeWriters.add(this);
         }
      }

      /**
       * Releases lock
       */
      public void release()
      {
         synchronized (RangeReadWriteLockFair.this.globalSynch)
         {
            RangeReadWriteLockFair.this.activeWriters.remove(this);
            RangeReadWriteLockFair.this.globalSynch.notifyAll();
         }
      }
   }
   // Waiters to obtain locks
   private final SortedMap<Integer, RangeReadWriteLock.Lock> waitQueue =
         new TreeMap<Integer, RangeReadWriteLock.Lock>();

   // Lock holders
   private final List<WriteLock> activeWriters = new LinkedList<WriteLock>();
   private final List<ReadLock> activeReaders = new LinkedList<ReadLock>();

   private final Object globalSynch = new Object();

   /**
    * Lock for read
    */
   public RangeReadWriteLock.Lock getWriteRangeLock(long start, long amount)
   {
      return new WriteLock(start, amount);
   }

   /**
    * Lock for write
    */
   public RangeReadWriteLock.Lock getReadRangeLock(long start, long amount)
   {
      return new ReadLock(start, amount);
   }

   private boolean rangeBusy(List<? extends LockBase> busyList, LockBase candidate)
   {
      // Should be called under lock
      for (final LockBase l : busyList)
      {
         if (l.intersect(candidate))
         {
            return true;
         }
      }
      return false;
   }
}
