//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
// -----------------------
// @author: ivolvovski
//
// Date: May 1, 2008
// ---------------------

package org.cleversafe.util.rangelock;

/*
 * Protects an integer range for read/write Any readers and writes obtain locks as long as their
 * range don't overlap Any reader's don't conflict which other even if their ranges overlap
 */
public interface RangeReadWriteLock
{
   /**
    * The Lock interface that allows to acquire read or write lock with semantics described above
    */
   public interface Lock
   {
      public void acquire();

      public void release();

      public long getStart();

      public long getAmount();

      public boolean intersect(final Lock another);
   }

   /*
    * Obtain a read lock for a given range
    */
   public Lock getWriteRangeLock(long start, long amount);

   /*
    * Obtain a write lock for a given range
    */
   public Lock getReadRangeLock(long start, long amount);

   // Common reusable base

   abstract class LockBase implements RangeReadWriteLock.Lock
   {

      protected long start;
      protected long amount;

      /**
       * [start, start+amount] - both ends included
       * 
       * @param start
       * @param amount
       */
      public LockBase(final long start, final long amount)
      {
         assert amount >= 0;
         this.start = start;
         this.amount = amount;
      }

      public long getStart()
      {
         return this.start;
      }

      public long getAmount()
      {
         return this.amount;
      }

      /**
       * Checks whether 2 intervals with ends intersect
       * 
       * @param another
       * @return
       */
      public boolean intersect(final Lock another)
      {
         return !((this.start + this.amount <= another.getStart()) || (this.start >= another.getStart()
               + another.getAmount()));
      }

      public String toString()
      {
         return String.format("RangeLock[start=%d, range=%d]", this.start, this.amount);
      }
   }
}
