//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Jun 14, 2007
//---------------------

package org.cleversafe.util;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;

// Utilities used for reflection and other class related activities

public class TypeUtils
{
   /**
    * Class collects primitive class information used to cast, print and parse
    * value of primitive types
    * 
    */
   public static class PrimitiveTypeInfo
   {
      private String name;
      private Class<?> primitiveClass;
      private Class<?> wrapperClass;
      private PrimitiveTypeInfo widensTo;
      private Method parseMethod;
      private Method printMethod;

      private PrimitiveTypeInfo(Class<?> wrapperClass, PrimitiveTypeInfo widensTo)
      {
         this.wrapperClass = wrapperClass;
         this.widensTo = widensTo;
         try
         {
            this.primitiveClass = (Class<?>) this.wrapperClass.getDeclaredField("TYPE").get(null);
            this.name = this.primitiveClass.getName();
            if (this.primitiveClass != void.class)
            {
               if (this.primitiveClass == char.class)
                  this.parseMethod = TypeUtils.class.getDeclaredMethod("parseChar", new Class[]{
                     String.class
                  });
           
               else
               {
                  this.parseMethod =
                     this.wrapperClass.getDeclaredMethod(NamingHelper.convertNameIntoJavaMethod(
                           name, "parse", null), String.class);
                  this.printMethod =
                     this.wrapperClass.getDeclaredMethod("toString", this.primitiveClass);
               }
            }
         }
         catch (Exception e)
         {
            throw new RuntimeException("Incorrect type information", e);
         }
      }

      /**
       *  Primitive name a type is known (int, boolean etc)
       * @return
       */
      public String getName()
      {
         return this.name;
      }

      /**
       * Corresponding primitive type
       * @return
       */
      public Class<?> getPrimitiveClass()
      {
         return this.primitiveClass;
      }

      /**
       * Corresponding wrapper class
       * @return
       */
      public Class<?> getWrapperClass()
      {
         return this.wrapperClass;
      }

      /** 
       * Method to be used to parse it from a string
       * @return Static method
       */
      public Method getParseMethod()
      {
         return this.parseMethod;
      }

      /**
       * Method to convert an object of primitive type into string
       * @return
       */
      public Method getPrintMethod()
      {
         return this.printMethod;
      }

      /**
       * Verifies whether primitive type is assignable.
       * It is assignable if there is widening to in 1 or more steps
       * int --> float
       * short --> double ( shotrt --> int --> float --> double)
       * @param other the other type
       * @return true if assignable
       */
      public boolean isAssignableFrom(PrimitiveTypeInfo other)
      {
         // Find a long widening path.
         for (PrimitiveTypeInfo t = other; t != null; t = t.widensTo)
         {
            if (t == this)
            {
               return true;
            }
         }
         return false;
      }

      /**
       * If one is assignable to another
       * @param other
       * @return
       */
      public boolean isCastableFrom(PrimitiveTypeInfo other)
      {
         // Return true if:
         // - there is a "widensTo" path from other to this, or vice-versa,
         // or
         // - there is a "widensTo" path from other and this to a
         // common-type;
         // (i.e. the "widensTo" paths from this and other "meet" somewhere)
         // char is castable to short, because char --> int and short --> int
         if (isAssignableFrom(other) || other.isAssignableFrom(this))
            return true;
         else
         {
            // No linear path to/from other...
            // If this and other have a widening-path to the same type, then
            // we can cast from this to other (and vice-versa)...
            for (PrimitiveTypeInfo t = this.widensTo; t != null; t = t.widensTo)
               if (t.isAssignableFrom(other))
                  return true;
            return false;
         }
      }

      /**
       * Convinience: object of primitive type converted to a java literal as this primitive
       * @param value to be converted
       * @return java string
       */
      public String printAsJavaLiteral(Object value)
      {
         if (value.getClass().isAssignableFrom(this.wrapperClass))
         {
            return value.toString();
         }
         throw new RuntimeException("Invalid object type " + value.getClass() + "to generate Java "
               + this.wrapperClass.getName() + " value");
      }

      /**
       * Convinience: string representation to a java literal as this primitive
       * @param value
       * @return
       */
      public String printAsJavaLiteral(String value)
      {
         try
         {
            return printAsJavaLiteral(getParseMethod().invoke(null, value));
         }
         catch (InvocationTargetException ex)
         {
            throw new RuntimeException("Can't parse value:" + value + " to "
                  + this.wrapperClass.getName() + " type", ex);
         }
         catch (IllegalAccessException ex)
         {
            throw new RuntimeException("Should not happen!", ex);
         }
      }
   }

   public static final PrimitiveTypeInfo DOUBLE = new PrimitiveTypeInfo(Double.class, null);

   public static final PrimitiveTypeInfo FLOAT = new PrimitiveTypeInfo(Float.class, DOUBLE)
   {
      public String printAsJavaLiteral(Object value)
      {
         if (value.getClass().isAssignableFrom(Float.class))
         {
            return value.toString() + 'f';
         }
         throw new RuntimeException("Invalid object type " + value.getClass()
               + "to generate Java float value");
      }
   };

   public static final PrimitiveTypeInfo LONG = new PrimitiveTypeInfo(Long.class, FLOAT)
   {
      public String printAsJavaLiteral(Object value)
      {
         if (value.getClass().isAssignableFrom(Long.class))
         {
            return value.toString() + 'l';
         }
         throw new RuntimeException("Invalid object type " + value.getClass()
               + "to generate Java long value");
      }
   };

   public static final PrimitiveTypeInfo INT = new PrimitiveTypeInfo(Integer.class, LONG);

   public static final PrimitiveTypeInfo SHORT = new PrimitiveTypeInfo(Short.class, INT);

   public static final PrimitiveTypeInfo CHAR = new PrimitiveTypeInfo(Character.class, INT)
   {
      public String printAsJavaLiteral(Object value)
      {
         if (value.getClass().isAssignableFrom(Character.class))
         {
            return "'" + ((Character) value).charValue() + "'";
         }
         throw new RuntimeException("Invalid object type " + value.getClass()
               + "to generate Java char value");
      }
   };

   public static final PrimitiveTypeInfo BYTE = new PrimitiveTypeInfo(Byte.class, SHORT);

   public static final PrimitiveTypeInfo BOOLEAN = new PrimitiveTypeInfo(Boolean.class, null);

   public static final PrimitiveTypeInfo VOID = new PrimitiveTypeInfo(Void.class, null);

   private static final Map<String, PrimitiveTypeInfo> pnameMap =
         new HashMap<String, PrimitiveTypeInfo>();

   private static final Map<Class<?>, PrimitiveTypeInfo> wclassMap =
         new HashMap<Class<?>, PrimitiveTypeInfo>();

   static
   {
      PrimitiveTypeInfo[] t = {
            BOOLEAN, BYTE, CHAR, SHORT, INT, LONG, FLOAT, DOUBLE, VOID
      };

      int i;
      for (i = 0; i < t.length; i++)
      {
         pnameMap.put(t[i].getName(), t[i]);
         wclassMap.put(t[i].getWrapperClass(), t[i]);
      }
   }

   /**
    * PrimitiveInfo by name (int, long), throws if not a correct name
    * @param name 
    * @return
    */
   public static PrimitiveTypeInfo getByPrimitiveName(String name)
   {
      PrimitiveTypeInfo t = findByPrimitiveName(name);
      if (t == null)
         throw new IllegalArgumentException("Not a primitive type: " + name);
      return t;
   }

   /**
    * PrimitiveInfo by name (int, long)
    * @param name 
    * @return null if not a primitive name
    */
   public static PrimitiveTypeInfo findByPrimitiveName(String name)
   {
      return pnameMap.get(name);
   }

   /**
    * PrimitiveInfo by class (int.class, long.class), throws if not a correct name
    * @param pclass
    * @return
    */
   public static PrimitiveTypeInfo getByPrimitiveClass(Class<?> pclass)
   {
      return getByPrimitiveName(pclass.getName());
   }

   /**
    * PrimitiveInfo by class (int.class, long.class)
    * @param pclass
    * @return null if invalid class
    */
   public static PrimitiveTypeInfo findByPrimitiveClass(Class<?> pclass)
   {
      return findByPrimitiveName(pclass.getName());
   }

   /**
    * PrimitiveInfo by wrapper name (Integer, Long)
    * @param name
    * @return null if doesn't exist
    */
   public static PrimitiveTypeInfo findByWrapperName(String name)
   {
      try
      {
         return findByWrapperClass(Class.forName(name));
      }
      catch (ClassNotFoundException e)
      {
         return null;
      }
   }
   /**
    * PrimitiveInfo by wrapper name (Integer, Long), throws if doesn't exist
    * @param name
    * @return 
    */
   public static PrimitiveTypeInfo getByWrapperName(String name)
   {
      PrimitiveTypeInfo t = findByWrapperName(name);
      if (t == null)
         throw new IllegalArgumentException("Not a primitive wrapper class: " + name);
      return t;
   }

   /**
    * PrimitiveInfo by wrapper class (Integer.class, Long.class)
    * @param sliceName
    * @return 
    */
   public static PrimitiveTypeInfo getByWrapperClass(Class<?> wclass)
   {
      PrimitiveTypeInfo t = findByWrapperClass(wclass);
      if (t == null)
         throw new IllegalArgumentException("Not a primitive wrapper class: " + wclass);
      return t;
   }

   public static PrimitiveTypeInfo findByWrapperClass(Class<?> wclass)
   {
      return (PrimitiveTypeInfo) wclassMap.get(wclass);
   }

   /**
    * Checks whether class is primitive
    * @param pclass
    * @return
    */
   public static boolean isPrimitive(Class<?> pclass)
   {
      return findByPrimitiveClass(pclass) != null;
   }

   /**
    * Checks whether name is of a primitive type
    * @param pclass
    * @return
    */
   public static boolean isPrimitive(String name)
   {
      return findByPrimitiveName(name) != null;
   }

   /**
    * Checks whether type 'to' is assignable 'from' including primitives 
    * @param to
    * @param from
    * @return
    */
   public static boolean isAssignableFrom(Class<?> to, Class<?> from)
   {
      Class<?> primitiveCandidateTo = findPrimitiveFor(to);
      if (primitiveCandidateTo == null)
      {
         primitiveCandidateTo = to;
      }
      Class<?> primitiveCandidateFrom = findPrimitiveFor(from);
      if (primitiveCandidateFrom == null)
      {
         primitiveCandidateFrom = from;
      }
      // Choose how to compare, as primitives or as regulat types
      if (primitiveCandidateTo.isPrimitive() && primitiveCandidateFrom.isPrimitive())
         return getByPrimitiveClass(primitiveCandidateTo).isAssignableFrom(getByPrimitiveClass(primitiveCandidateFrom));
      else
      {
         return getNonPrimitiveFor(to).isAssignableFrom(getNonPrimitiveFor(from));
      }
   }

   /**
    * Wrapper class from a primitive type
    * 
    * @param pclass
    *           primitive type
    * @return corresponding wrapper
    */
   public static Class<?> getWrapperFor(Class<?> pclass)
   {
      return getByPrimitiveClass(pclass).getWrapperClass();
   }

   /**
    * Returns a non primitive type corresponding clazz
    * @param clazz
    * @return return itself if not a primitive, a wrapper otherwise
    */
   public static Class<?> getNonPrimitiveFor(Class<?> clazz)
   {
      if (!clazz.isPrimitive())
      {
         return clazz;
      }
      return getWrapperFor(clazz);
   }

   /**
    * Primitive type for a given wrapper class
    * 
    * @param wclass
    * @return
    */
   public static Class<?> getPrimitiveFor(Class<?> wclass)
   {
      return getByWrapperClass(wclass).getPrimitiveClass();
   }

   /**
    * Primitive type for a given wrapper class, null if it is not primitive
    * 
    * @param wclass
    * @return
    */
   public static Class<?> findPrimitiveFor(Class<?> clazz)
   {
      if (clazz.isPrimitive()) {
         return clazz;
      }
      PrimitiveTypeInfo info = findByWrapperClass(clazz);
      if (info != null) {
         return info.getPrimitiveClass();
      }
      return null;
   }

   /**
    * Character requires a special parsing
    * 
    * @param str
    *           string top parse from
    * @return character
    */
   public static char parseChar(String str)
   {
      if (str.length() == 0)
         return '\0';
      else if (str.length() > 1)
         throw new IllegalArgumentException("String too long to convert to char: " + str);
      else
         return str.charAt(0);
   }

   /**
    * Convinienece method to create list of types
    * 
    * @param args
    * @return
    */
   public static String listOfClassesToStr(Class<?>[] args)
   {
      StringBuffer res = new StringBuffer();
      for (int i = 0; i < args.length; i++)
      {
         if (i > 0)
            res.append(", ");
         res.append(args[i].getName());
      }
      return res.toString();
   }

   /**
    * Tries to find the first signature that could be invoked without run-time error
    * This is not ideal but seems to be quite adequate
    * @param clazz Class upon which a method is looked
    * @param methodName Method name
    * @param parameterTypes argument types
    * @param includeStatic include or not static methdods in search
    * @return Method object if found null otherwise
    */
   public static Method getMatchingMethod(
         Class<?> clazz,
         String methodName,
         Class<?>[] parameterTypes,
         boolean includeStatic)
   {
      Method[] allMethods = clazz.getMethods();
      for (Method method : allMethods)
      {
         if (!method.getName().equals(methodName))
         {
            continue;
         }
         if (!includeStatic && Modifier.isStatic(method.getModifiers()))
         {
            continue;
         }
         // Check wheather each parameter is assignable
         if (parameterTypes.length != method.getParameterTypes().length)
         {
            continue;
         }
         int i;
         for (i = 0; i < parameterTypes.length; i++)
         {
            if (!isAssignableFrom(method.getParameterTypes()[i], parameterTypes[i]))
            {
               break;
            }
         }
         if (i == parameterTypes.length)
         {
            return method;
         }
      }
      return null;
   }

}
