//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jan 4, 2008
//---------------------

package org.cleversafe.util;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Prints a table into an ASCII-art table. Takes a variable number of columns. The table
 * will be sorted by the first column (which must not contain duplicates).
 */
public class TablePrinter
{
   
   private static final String ROW_LEFT_BORDER = "| ";
   private static final String ROW_CENTER_BORDER = " | ";
   private static final String ROW_RIGHT_BORDER = " |";
   private static final String BORDER_LEFT_EDGE = "+-";
   private static final String BORDER_CENTER_EDGE = "-+-";
   private static final String BORDER_RIGHT_EDGE = "-+";
   private static final String BORDER_FILLER = "-";
   
   private final Row header;
   private final List<Row> rows;
   
   private final List<Integer> columnWidth;
   private final int columnCount;
   
   private int tableWidth;
   
   
   private static class Row implements Comparable<Row>
   {
      private final List<String> row;
      private final List<Integer> width;
      
      public Row(String[] elems)
      {
         this.row = Arrays.asList(elems);
         this.width = calculateWidth(row);
      }
      
      public Row(List<String> elems)
      {
         this.row = new ArrayList<String>(elems);
         this.width = calculateWidth(row);
      }
      
      private static List<Integer> calculateWidth(List<String> row)
      {
         List<Integer> widths = new ArrayList<Integer>(row.size());
         for (String elem : row)
         {
            widths.add(elem != null ? elem.length() : 0);
         }
         return widths;
      }
      
      public List<Integer> getWidth()
      {
         return Collections.unmodifiableList(this.width);
      }
      
      public int getMinTableWidth()
      {
         int minWidth = ROW_LEFT_BORDER.length() + ROW_RIGHT_BORDER.length() + 
            (ROW_CENTER_BORDER.length() * this.row.size());
         for (int w : this.getWidth())
         {
            minWidth += w;
         }
         return minWidth;
      }
      
      public void updateWidth(List<Integer> width)
      {
         Iterator<Integer> here = this.getWidth().iterator();
         for (int i = 0; i < width.size(); i++)
         {
            width.set(i, Math.max(width.get(i), here.next()));
         }
      }
      
      public List<String> getColumns(List<Integer> width)
      {
         List<String> columns = new ArrayList<String>(width.size());
         Iterator<Integer> w = width.iterator();
         Iterator<String> it = this.row.iterator();
         while(w.hasNext())
         {
            columns.add(String.format("%1$-" + w.next() + "s", it.next()));
         }
         return columns;
      }

      public int compareTo(Row obj)
      {
         final String here = this.row.size() > 0 ? this.row.get(0) : "";
         final String there = obj.row.size() > 0 ? obj.row.get(0) : "";
         return here.compareToIgnoreCase(there);
      }
      
   }
   
   /**
    * Create a table printer with the given headers.
    */
   public TablePrinter(String... header)
         throws IllegalArgumentException
   {
      this.header = new Row(header);
      this.rows = new ArrayList<Row>();
      
      this.columnCount = header.length;
      this.columnWidth = new ArrayList<Integer>(this.header.getWidth());
      this.tableWidth = this.header.getMinTableWidth();
   }
   
   /**
    * Create a table without headers with the given number of columns.
    */
   public TablePrinter(int columns) throws IllegalArgumentException
   {
      this.header = null;
      this.rows = new ArrayList<Row>();
      
      Row fake = new Row(new String[columns]);
      this.columnCount = columns;
      this.columnWidth = new ArrayList<Integer>(fake.getWidth());
      this.tableWidth = fake.getMinTableWidth(); 
   }
   
   /**
    * Add a row to the table. The number of elements must be equal to the number of columns.
    * Output is sorted by the first column.
    */
   public TablePrinter add(String... row) throws IllegalArgumentException
   {
      if (row.length != this.columnCount)
      {
         throw new IllegalArgumentException(
               "Row contains invalid number of columns (expected " + 
               this.columnCount + "): " + Arrays.toString(row));
      }
      
      Row r = new Row(row);
      this.rows.add(r);
      Collections.sort(this.rows);
      r.updateWidth(this.columnWidth);
      this.tableWidth = Math.max(this.tableWidth, r.getMinTableWidth());
      
      return this;
   }
   
   /**
    * Print the table to a print stream.
    */
   public void print(PrintStream out)
   {
      out.println(getTableHorzSeperator());
      if (this.header != null)
      {
         out.println(getRow(this.header));
         out.println(getTableHorzSeperator());
      }
      for (Row row : this.rows)
      {
         out.println(getRow(row));
      }
      out.println(getTableHorzSeperator());
      
   }
   
   private String getTableHorzSeperator()
   {
      StringBuilder builder = new StringBuilder(tableWidth);
      builder.append(BORDER_LEFT_EDGE);
      for (int i = 0; i < this.columnCount - 1; i++)
      {
         for (int j = 0; j < this.columnWidth.get(i); j++)
         {
            builder.append(BORDER_FILLER);
         }
         builder.append(BORDER_CENTER_EDGE);
      }
      for (int j = 0; j < this.columnWidth.get(this.columnCount-1); j++)
      {
         builder.append(BORDER_FILLER);
      }
      builder.append(BORDER_RIGHT_EDGE);
      return builder.toString();
   }
   
   @SuppressWarnings("unchecked")
   private String getRow(Row row)
   {
      StringBuilder builder = new StringBuilder();
      builder.append(ROW_LEFT_BORDER);
      
      Iterator<String> elems = row.getColumns(this.columnWidth).iterator();
      while (elems.hasNext())
      {
         builder.append(elems.next());
         if (elems.hasNext())
         {
            builder.append(ROW_CENTER_BORDER);
         }
      }
      builder.append(ROW_RIGHT_BORDER);
      return builder.toString();
   }
   
   
   @SuppressWarnings("unchecked")
   public static void main(String[] args)
   {
      
      
   }
   
   

}


