//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: Jun 17, 2007
//---------------------

package org.cleversafe.util;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Bounded, self-regulating ThreadPoolExecutor. No more than bound threads will be created, and no
 * tasks will be kept in a wait queue. If the maximum number of threads are active, any attempt to
 * enqueue a new task will block until an existing task finishes.
 * 
 * Threads that are not used for some period of time are destroyed. An unused
 * BoundedThreadPoolExecutor will have no running threads.
 */
public class BoundedThreadPoolExecutor extends ThreadPoolExecutor
{
   public static final String DEBUG_THREADLESS_PROPERTY =
         "org.cleversafe.util.BoundedThreadPoolExecutor.isDebugThreadless";

   /** Maximum number of threads in pool */
   protected int bound;

   /** Semaphore with bound permits */
   protected Semaphore permitHolder;

   /** Reference to the executor's work queue */
   protected BlockingQueue<Runnable> workQueue;

   /** Threadless mode (DEDBUG) enabled? */
   protected boolean isDebugThreadless;

   /** Priority of threads in this pool */
   protected int priority;

   /**
    * Create a new bounded thread pool
    * 
    * @param threadNamePrefix
    *           Prefix for the name of each thread in this pool
    * @param bound
    *           Maximum number of running task threads
    */
   public BoundedThreadPoolExecutor(String threadNamePrefix, int bound)
   {
      this(threadNamePrefix, bound, Thread.currentThread().getPriority());
   }

   /**
    * Create a new bounded thread pool
    * 
    * @param threadNamePrefix
    *           Prefix for the name of each thread in this pool
    * @param bound
    *           Maximum number of running task threads
    * @param priority
    *           Priority of threads in this pool
    */
   public BoundedThreadPoolExecutor(String threadNamePrefix, int bound, int priority)
   {
      super(0, // Shrinks to 1 thread if not in use
            bound, // Grows to N threads when in use
            60, TimeUnit.SECONDS, // Threads live for 1 minute when not in use
            new SynchronousQueue<Runnable>(), new BoundedThreadFactory(threadNamePrefix, bound,
                  priority));

      this.bound = bound;
      this.permitHolder = new Semaphore(bound);
      this.workQueue = super.getQueue();
      this.priority = priority;

      // Threadless mode, for debugging
      this.isDebugThreadless =
            (System.getProperty(DEBUG_THREADLESS_PROPERTY, "false").equals("true"));
   }

   /**
    * Get current maximum number of task threads
    * 
    * @return Maximum number of running task threads
    */
   public int getBound()
   {
      return this.bound;
   }

   /**
    * Get the priority of threads in this pool
    * 
    * @return
    */
   public int getPriority()
   {
      return this.priority;
   }

   /**
    * Schedule a new task for execution, blocking until it can be scheduled if the bound has been
    * reached.
    * 
    * @see ThreadPoolExecutor.execute
    */
   public void execute(final Runnable command)
   {
      if (!this.isDebugThreadless)
      {
         try
         {
            // Permit is acquired when submitted and released when execution is done
            this.permitHolder.acquire();
            try
            {
               super.execute(command);
            }
            catch (final RejectedExecutionException ex)
            {
               // Since we have a permit, we know a thread will come along
               // looking for a task eventually. The timeout is just for safety
               boolean success = this.workQueue.offer(command, 10, TimeUnit.SECONDS);
               if (!success)
               {
                  // This should never happen
                  throw new RuntimeException("Bug in BoundedThreadPoolExecutor");
               }
            }
         }
         catch (final InterruptedException ex)
         {
            throw new RejectedExecutionException("Interrupted", ex);
         }
      }
      else
      {
         // Handle threadless debugging case, run in this thread
         command.run();
      }
   }

   /**
    * A task's permit is released after it finishes executing
    */
   @Override
   protected void afterExecute(Runnable r, Throwable t)
   {
      super.afterExecute(r, t);
      this.permitHolder.release();
   }

   /**
    * Helper factory for a named thread pool
    */
   private static class BoundedThreadFactory extends NamedThreadFactory
   {
      // Maximum active pool threads
      protected int bound;

      // Priority of threads created
      protected int priority;

      public BoundedThreadFactory(String prefix, int bound, int priority)
      {
         super(prefix);
         this.bound = bound;
         this.priority = priority;
      }

      public Thread newThread(Runnable r)
      {
         Thread t = super.newThread(r);
         t.setName(t.getName() + " (" + this.bound + ")");
         t.setPriority(this.priority);
         return t;
      }
   }
}
