//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: May 31, 2007
//---------------------

package org.cleversafe.util;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.UUID;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DEROctetString;


/**
 * Encodes a universally unique identifier (UUID) as an ASN1 OCTET STRING.
 * 
 * UniqueIdentifier   ::= OCTET STRING (SIZE (16))
 *
 */
public class ASN1UniqueIdentifier extends ASN1Encodable
{
   private UUID identifier;

   public ASN1UniqueIdentifier( UUID identifier )
   {
      this.identifier = identifier;
   }
   
   public ASN1UniqueIdentifier( DEROctetString identifier )
   {
      // TODO: Ensure this always reads in network (MSB) order
      
      if ( identifier.getOctets().length != 16 )
      {
         throw new IllegalArgumentException("not enough bytes for unique identifier");
      }
      
      try
      {
         DataInputStream in = new DataInputStream( identifier.getOctetStream() );
         long high = in.readLong();
         long low = in.readLong();
         
         this.identifier = new UUID( high, low );         
      }
      catch (IOException e)
      {
         throw new IllegalArgumentException("error decoding unique identifier from octet string");
      }
   }
   
   public static ASN1UniqueIdentifier getInstance( ASN1TaggedObject obj , boolean explicit )
   {
      return getInstance(obj.getObject());
   }
   
   public static ASN1UniqueIdentifier getInstance( Object obj )
   {
      if ( obj instanceof ASN1UniqueIdentifier )
      {
         return (ASN1UniqueIdentifier)obj;
      }
      
      if ( obj instanceof DEROctetString )
      {
         return new ASN1UniqueIdentifier( (DEROctetString)obj );
      }
      
      throw new IllegalArgumentException("unknown object in factory");
   }
   
   
   public UUID getIdentifier()
   {
      return this.identifier;
   }
   
   
   @Override
   public DERObject toASN1Object()
   {
      // TODO: Ensure this always writes in network (MSB) order
      ByteArrayOutputStream bytestream = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream( bytestream );
      
      try
      {
         out.writeLong( this.identifier.getMostSignificantBits() );
         out.writeLong( this.identifier.getLeastSignificantBits() );
      }
      catch (IOException e)
      {
         throw new RuntimeException("Unexpected IOException during byte array output", e);
      }
      
      return new DEROctetString( bytestream.toByteArray() );
   }

}


