//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: dhendrickson
//
// Date: May 24, 2007
//---------------------

package org.cleversafe.server.handlers.authentication;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.AuthenticationFacilitator;
import org.cleversafe.authentication.AuthenticationFacilitatorConfiguration;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.util.UUIDGen;

/**
 * A request handler for processing username/password authentication via a JAAS module.
 * 
 * @author Dusty Hendrickson
 */
public class PasswordAuthenticationHandler implements RequestHandler
{
   private static Logger _logger = Logger.getLogger(PasswordAuthenticationHandler.class);

   private AuthenticationFacilitator authenticationFacilitator;
   private final AuthenticationFacilitatorConfiguration configuration;
   private final String authMethod;
   private final List<ApplicationType> applicationTypes;

   /**
    * Constructs the handler with the given configuration file locations and profile name.
    * 
    * @param jaasProfile
    *           The name of the profile in the given JAAS configuration file.
    * @param jaasConfigFile
    *           The filename of the JAAS configuration file.
    */
   public PasswordAuthenticationHandler(
         AuthenticationFacilitatorConfiguration configuration,
         String authMethod)
   {
      this.configuration = configuration;
      this.authMethod = authMethod;
      this.applicationTypes = new ArrayList<ApplicationType>(1);
      this.applicationTypes.add(ApplicationType.TYPE_ANY);
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this.applicationTypes;
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<PasswordAuthenticationRequest> getRequestClass()
   {
      return PasswordAuthenticationRequest.class;
   }

   /**
    * Attempts to authenticate the username and password located within the request. Upon successful
    * authentication, the proper session information is populated. Throws an exception if the
    * request is not of type PasswordAuthenticationRequest.
    * 
    * @param request
    *           The PasswordAuthenticationRequest containing the username and password.
    * @param session
    *           The session for the client connection.
    * @return A PasswordAuthenticationResponse containing the authentication status of the request.
    * @see PasswordAuthenticationRequest
    * @see PasswordAuthenticationResponse
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session)
   {
      _logger.info("Servicing authentication request");

      // Create request
      PasswordAuthenticationRequest passwordAuthenticationRequest;
      passwordAuthenticationRequest = (PasswordAuthenticationRequest) request;

      // Create response
      PasswordAuthenticationResponse passwordAuthenticationResponse;
      passwordAuthenticationResponse = new PasswordAuthenticationResponse();

      UUID gridAccountID = null;

      String username = passwordAuthenticationRequest.getUsername();
      String password = passwordAuthenticationRequest.getPassword();

      PasswordCredentials credentials = new PasswordCredentials(username, password);

      try
      {
         this.authenticationFacilitator =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     AuthenticationFacilitator.class);
         this.authenticationFacilitator.setConfiguration(this.configuration);
      }
      catch (ConfigurationItemNotDefinedException e)
      {
         _logger.error("Configuration item not defined for AuthenticationFacilitator");
         passwordAuthenticationResponse.setAuthenticated(false);
         return passwordAuthenticationResponse;
      }
      catch (ObjectInstantiationException e)
      {
         _logger.error("Could not instantiate AuthenticaitonFaciliator");
         passwordAuthenticationResponse.setAuthenticated(false);
         return passwordAuthenticationResponse;
      }
      catch (ObjectInitializationException e)
      {
         _logger.error("Could not initialize AuthenticationFacilitator");
         passwordAuthenticationResponse.setAuthenticated(false);
         return passwordAuthenticationResponse;
      }
      catch (ConfigurationLoadException e)
      {
         _logger.error("Exception when loading configuration: " + e.getMessage());
         passwordAuthenticationResponse.setAuthenticated(false);
         return passwordAuthenticationResponse;
      }

      boolean authenticated =
            this.authenticationFacilitator.attemptAuthentication(credentials, this.authMethod);

      passwordAuthenticationResponse.setAuthenticated(authenticated);

      // Create grid account UUID
      // FIXME: It is a design flaw to allow the user to set any arbitrary account
      // ID in the authentication request. In the future the server should either
      // authenticate directly against the account ID (not username), or have some
      // mechanism for looking up the account ID for any given username.
      // ### gridAccountID = passwordAuthenticationRequest.getGridAccountID(); ###

      // TODO: For the following line of code to work it requires that the client
      // derive the account ID in the same way. This is a kludge to prevent clients
      // from authenticating to any account they desire. - JKR
      gridAccountID = UUIDGen.getUUIDFromUsername(username);

      // Update session information
      if (passwordAuthenticationResponse.getAuthenticated())
      {
         _logger.debug("Authentication succeeded for user " + username);
         session.put(ClientSession.GRID_ACCOUNT_UUID, gridAccountID);
         session.put(ClientSession.AUTHENTICATED, true);
         _logger.info("Authentication succeeded for user " + username + " with accound ID "
               + gridAccountID);
      }
      else
      {
         _logger.warn("Authentication failed for user " + username);
      }

      return passwordAuthenticationResponse;
   }
}
