//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: May 16, 2007
//---------------------

package org.cleversafe.server;

import java.util.HashMap;

import org.apache.log4j.Logger;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.ProtocolLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.server.exceptions.ServerException;
import org.cleversafe.server.exceptions.ServerRequestException;
import org.cleversafe.server.exceptions.UnauthorizedRequestException;
import org.cleversafe.server.exceptions.UnrecognizedRequestException;

/**
 * This is a base class for implementing generic servers. Handlers must be registered with this
 * server before it can fulfill any purpose.
 */
public abstract class ServerApplication
{
   private static Logger logger = Logger.getLogger(ServerApplication.class);

   protected HashMap<Class<? extends Request>, RequestHandler> commandHandlers =
         new HashMap<Class<? extends Request>, RequestHandler>();

   /**
    * Looks up the appropriate handler for the given request, and processes the request with that
    * handler. Depending on the request, the client session may be read from or written to.
    * 
    * @param request
    * @param session
    * @return a response for the given request
    */
   public <T extends Request> Response service(T request, ClientSession session)
   {
      // Determine the type of the request
      Class<? extends Request> requestType = request.getClass();

      // Get the appropriate handler for the received request
      RequestHandler handler = this.commandHandlers.get(requestType);

      if (logger.isDebugEnabled())
         logger.debug("Request [" + request + "] in " + session);

      Response response = null;

      try
      {
         if (handler != null)
         {
            // Authentication not required
            if (handler.getAllowedApplications().contains(ApplicationType.TYPE_ANY))
            {
               response = handler.service(request, session);
            }
            // Authentication required
            else
            {
               if (session.containsKey(ClientSession.AUTHENTICATED)
                     && session.get(ClientSession.AUTHENTICATED).equals(true))
               {
                  response = handler.service(request, session);
               }
               else
               {
                  throw new UnauthorizedRequestException("Not authenticated. Acces denied", request);
               }
            }
         }
         else
         {
            throw new UnrecognizedRequestException("Handler for request is not registered", request);
         }
      } 
      catch (AssertionError ex)
      {
         logger.error("An asertion failed while processing request " + request + ": " + ex.getMessage());
         
         response = new ErrorResponse();
         response.setException(new ServerRequestException("An assertion failed: " + ex.getMessage(), request, ex));
      }
      catch (Exception ex)
      {
         if (logger.isDebugEnabled())
         {
            logger.debug("Exception '" + ex.getClass().getName() + "' while processing request "
                  + request + " in session context " + session, ex);
         }

         response = new ErrorResponse();

         //FIXME: it would be better if we had a central "isExceptionSerializable(Exception e)" 
         //       method that could be used instead of checking instances
         if (ex instanceof SliceStoreLayerException || ex instanceof ServerException
               || ex instanceof ProtocolLayerException)
         {
            response.setException(ex);
         }
         else
         {
            response.setException(new ServerRequestException(ex.getMessage(), request, ex));
         }
      }

      // If the request and response are sequenced
      if (request instanceof SequencedProtocolMessage
            && response instanceof SequencedProtocolMessage)
      {
         // Set the sequence number of the response
         SequencedProtocolMessage sequencedRequest = (SequencedProtocolMessage) request;
         SequencedProtocolMessage sequencedResponse = (SequencedProtocolMessage) response;
         sequencedResponse.setSequenceNumber(sequencedRequest.getSequenceNumber());
      }
      if (logger.isDebugEnabled())
         logger.debug("Response [" + response + "] in " + session);

      return response;
   }

   /**
    * Registers a handler for a given request so that it may be remotely called by a client.
    * 
    * @param handler
    */
   public void addHandler(RequestHandler handler)
   {
      assert this.commandHandlers.containsKey(handler.getRequestClass()) == false : "A handler for this request has already been registered";

      this.commandHandlers.put(handler.getRequestClass(), handler);
   }

   /**
    * Returns a handler that has been registered for a given request, if none are registered returns
    * null.
    * 
    * @param handler
    */
   public RequestHandler getHandler(Class<? extends Request> requestClass)
   {
      return this.commandHandlers.get(requestClass);
   }

   /**
    * Unregisters a handler for a given request.
    * 
    * @param handler
    */
   public void removeHandler(RequestHandler handler)
   {
      this.commandHandlers.remove(handler.getRequestClass());
   }

   /**
    * Checks if a request is handled.
    * 
    * @return True if the handler class is registered, otherwise false.
    */
   public boolean isRequestHandled(Class<?> requestClass)
   {
      return this.commandHandlers.containsKey(requestClass);
   }

   /**
    * Returns the number of registered request handlers.
    * 
    * @return The number of registered request handlers.
    */
   public int getHandlerCount()
   {
      return this.commandHandlers.size();
   }
}
