//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
// -----------------------
// @author: mmotwani
//
// Date: Oct 4, 2007
// ---------------------

package org.cleversafe.serialization.raw.custom;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.serialization.asn1.ASN1ExceptionWrapper;
import org.cleversafe.serialization.exceptions.EncodeDecodeException;

public abstract class CustomRawSerializationBase implements CustomRawSerialization
{
   private Class<? extends GridSerializable> serializableClass = null;
   private static Logger _logger = Logger.getLogger(CustomRawSerializationBase.class);

   protected CustomRawSerializationBase(final Class<? extends GridSerializable> serializableClass)
   {
      this.serializableClass = serializableClass;
   }

   public Class<? extends GridSerializable> getSerializableClass()
   {
      return this.serializableClass;
   }

   public void write(final DataOutput out, final GridSerializable serializable)
         throws EncodeDecodeException, IOException
   {
      // Empty payload
   }

   public GridSerializable read(final DataInput in) throws EncodeDecodeException, IOException
   {
      try
      {
         // Empty payload
         return getSerializableClass().newInstance();
      }
      catch (final InstantiationException e)
      {
         throw new EncodeDecodeException("No custom serialization defined for "
               + getSerializableClass() + " and could not instantiate using default constructor.",
               e);
      }
      catch (final IllegalAccessException e)
      {
         throw new EncodeDecodeException("No custom serialization defined for "
               + getSerializableClass() + " and could not instantiate using default constructor.",
               e);
      }
   }

   // Convenience method
   public <T extends GridSerializable> byte[] encode(final T serializable)
         throws EncodeDecodeException, IOException
   {
      final ByteArrayOutputStream byteOutput =
            new ByteArrayOutputStream(calculateMaxSerializedSize(serializable));
      final DataOutputStream dataOut = new DataOutputStream(byteOutput);
      write(dataOut, serializable);
      dataOut.flush();
      dataOut.close();
      return byteOutput.toByteArray();
   }

   // Convenience method
   public <T extends GridSerializable> GridSerializable decode(
         final Class<T> cls,
         final byte[] encodedSerializable) throws EncodeDecodeException, IOException
   {
      final ByteArrayInputStream byteInput = new ByteArrayInputStream(encodedSerializable);
      final DataInputStream dataIn = new DataInputStream(byteInput);
      final GridSerializable obj =
            CustomRawSerializationRegistrator.getCustomSerialization(cls).read(dataIn);
      dataIn.close();
      byteInput.close();
      return obj;
   }

   public int getMaxSerializedSize(final GridSerializable serializable)
   {
      // 0 sized payload for empty messages
      return 0;
   }

   protected Exception readException(final DataInput in) throws EncodeDecodeException, IOException
   {
      if (in.readInt() == -1)
      {
         return null;
      }

      // FIXME: currently using ASN.1 exception serialization, use raw
      final byte[] encodedException = readByteArray(in);
      final ASN1Object derException = ASN1Object.fromByteArray(encodedException);
      return ASN1ExceptionWrapper.decodeSingleException(derException);
   }

   protected void writeException(final DataOutput out, final Exception exception)
         throws EncodeDecodeException, IOException
   {
      if (exception == null)
      {
         out.writeInt(-1);
      }
      else
      {
         out.writeInt(0);
         _logger.info("Encountered exception", exception);

         // FIXME: currently using ASN.1 exception serialization, use raw
         final DERSequence derException = ASN1ExceptionWrapper.encodeSingleException(exception);

         writeByteArray(out, derException.getEncoded());
      }
   }

   protected void writeString(final DataOutput out, final String string)
         throws EncodeDecodeException, IOException
   {
      if (string == null)
      {
         throw new EncodeDecodeException("null string");
      }

      out.writeUTF(string);
   }

   protected String readString(final DataInput in) throws IOException
   {
      return in.readUTF();
   }

   // helper method to read the next object off the stream
   @SuppressWarnings("unchecked")
   protected <T extends GridSerializable> T readObject(final Class<T> cls, final DataInput in)
         throws EncodeDecodeException, IOException
   {
      return (T) CustomRawSerializationRegistrator.getCustomSerialization(cls).read(in);
   }

   protected void writeObject(final DataOutput out, final GridSerializable obj)
         throws EncodeDecodeException, IOException
   {
      if (obj == null)
      {
         throw new EncodeDecodeException("null object");
      }
      CustomRawSerializationRegistrator.getCustomSerialization(obj.getClass()).write(out, obj);
   }

   /**
    * reads the length, then the byte array. Be careful to only call this when you know you are at
    * the start of an array
    * 
    * @param in
    *           Input stream that is queued up to the byte array
    * @return byte array that has just been read
    * @throws IOException
    */
   protected byte[] readByteArray(final DataInput in) throws IOException
   {
      final int dataLength = in.readInt();

      if (dataLength == -1)
      {
         return null;
      }
      else
      {
         // Read Data
         final byte[] data = new byte[dataLength];
         in.readFully(data, 0, dataLength);
         return data;
      }
   }

   /**
    * Write out a byte array with the following format: "length" - "data"
    * 
    * @param out -
    *           the output stream to write to
    * @param data -
    *           the data that will be written to the stream
    * @throws IOException
    */
   protected void writeByteArray(final DataOutput out, final byte[] data) throws IOException
   {
      if (data == null)
      {
         out.writeInt(-1);
      }
      else
      {
         // Write Data Size (as integer)
         out.writeInt(data.length);

         // Write Data
         out.write(data);
      }
   }

   protected <T extends GridSerializable> List<T> readObjectList(
         final Class<T> cls,
         final DataInput in) throws EncodeDecodeException, IOException
   {
      // Read the list size
      final int listSize = in.readInt();

      if (listSize == -1)
      {
         return null;
      }

      // Read the list
      final List<T> objList = new ArrayList<T>(listSize);
      for (int i = 0; i < listSize; ++i)
      {
         objList.add(readObject(cls, in));
      }

      return objList;
   }

   protected <T extends GridSerializable> void writeObjectList(
         final DataOutput out,
         final List<T> objList) throws EncodeDecodeException, IOException
   {
      if (objList == null)
      {
         out.writeInt(-1);
      }
      else
      {
         // Write the list size
         out.writeInt(objList.size());

         // Write the list
         for (final T obj : objList)
         {
            if (obj == null)
            {
               throw new EncodeDecodeException("One of the objects in the list was null");
            }

            writeObject(out, obj);
         }
      }
   }

   protected void writeStringMap(final DataOutput out, final Map<String, String> strMap)
         throws EncodeDecodeException, IOException
   {
      if (strMap == null)
      {
         out.writeInt(-1);
      }
      else
      {
         // Write the map size
         out.writeInt(strMap.size());

         // Write the key-value pairs
         for (final String key : strMap.keySet())
         {
            writeString(out, key);
            writeString(out, strMap.get(key));
         }
      }
   }

   protected Map<String, String> readStringMap(final DataInput in) throws IOException
   {
      // Read the map size
      final int mapSize = in.readInt();

      if (mapSize == -1)
      {
         return null;
      }

      // Create the map and read the key-value pairs
      final Map<String, String> map = new HashMap<String, String>(mapSize);
      for (int i = 0; i < mapSize; ++i)
      {
         map.put(readString(in), readString(in));
      }

      return map;
   }

   protected int calculateMaxSerializedSize(final GridSerializable obj)
   {
      if (obj == null)
      {
         return 0;
      }

      return CustomRawSerializationRegistrator.getCustomSerialization(obj.getClass()).getMaxSerializedSize(
            obj);
   }

   public <T extends GridSerializable> int calculateMaxSerializedSize(final List<T> objList)
   {
      int size = 4; // for the initial int

      if (objList != null)
      {
         for (final T obj : objList)
         {
            size += calculateMaxSerializedSize(obj);
         }
      }
      return size;
   }

   public int calculateMaxSerializedSize(final byte[] data)
   {
      if (data == null)
      {
         return 4;
      }

      return 4 + data.length;
   }

   public int calculateMaxSerializedSize(final String string)
   {
      if (string == null)
      {
         return 0;
      }

      return 2 + string.length() * 3;
   }

   public int calculateMaxSerializedSize(final Map<String, String> strMap)
   {
      int size = 4; // map size
      if (strMap != null)
      {
         for (final String key : strMap.keySet())
         {
            size += calculateMaxSerializedSize(key);
            size += calculateMaxSerializedSize(strMap.get(key));
         }
      }
      return size;
   }

}
