//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Oct 5, 2007
//---------------------

package org.cleversafe.serialization.raw.custom;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.serialization.asn1.ASN1ExceptionWrapper;
import org.cleversafe.serialization.exceptions.EncodeDecodeException;

public class BaseResponseSerialization extends CustomRawSerializationBase
{
   public BaseResponseSerialization(Class<? extends GridSerializable> serializableClass)
   {
      super(serializableClass);
   }

   protected void readException(DataInput in, Response response) throws EncodeDecodeException,
         IOException
   {
      if (in.readBoolean())
      {
         response.setException(readException(in));
      }
   }

   protected void writeException(DataOutput out, Response response) throws EncodeDecodeException,
         IOException
   {
      // Write exception boolean
      out.writeBoolean(response.getExceptionFlag());

      // If exception exists, write it
      if (response.getExceptionFlag())
      {
         writeException(out, response.getException());
      }
   }

   @Override
   public GridSerializable read(DataInput in) throws EncodeDecodeException, IOException
   {
      GridSerializable serializable = null;

      try
      {
         serializable = getSerializableClass().newInstance();
      }
      catch (InstantiationException e)
      {
         throw new EncodeDecodeException("Error instantiating object of class "
               + getSerializableClass().getName(), e);
      }
      catch (IllegalAccessException e)
      {
         throw new EncodeDecodeException("Error instantiating object of class "
               + getSerializableClass().getName(), e);
      }

      assert serializable instanceof Response;
      Response response = (Response) serializable;

      // Read the exception
      readException(in, response);

      return response;
   }

   @Override
   public void write(DataOutput out, GridSerializable serializable) throws EncodeDecodeException,
         IOException
   {
      assert serializable instanceof Response;
      Response response = (Response) serializable;

      // Write the exception
      writeException(out, response);
   }

   @Override
   public int getMaxSerializedSize(GridSerializable serializable)
   {
      int size = 1; // boolean (1 byte)
      Response response = (Response) serializable;
      if (response.getExceptionFlag())
      {
         size += 4; // an integer (4 bytes) is written writeException to say whether the exception is null
         if (response.getException() != null)
         {
            Exception ex = response.getException();
            DERSequence derSeq = ASN1ExceptionWrapper.encodeSingleException(ex);
            assert derSeq != null;
            try
            {
               size += derSeq.getEncoded().length;
            }
            catch (IOException e)
            {
               // Should never happen
               throw new RuntimeException("Error calculating encoded exception size", e);
            }
         }
      }

      return size;
   }
}
