//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: vthornton
//
// Date: Aug 21, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;
import java.math.BigInteger;

import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.slicestore.SliceInfo;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link SliceInfo}.
 * 
 * @see SliceInfo
 * @author Vance Thornton
 */
public class SliceInfoSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<SliceInfo> getSerializableClass()
   {
      return SliceInfo.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return SliceInfoSerialization.class.getMethod("encodeSliceInfo", new Class[]{
            getSerializableClass()
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return SliceInfoSerialization.class.getMethod("decodeSliceInfo", new Class[]{
               getSerializableClass(), ASN1Object.class
         });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link SliceInfo}.
    * 
    * @param sliceInfo
    * @return
    */
   public static DERSequence encodeSliceInfo(SliceInfo sliceInfo)
   {
      DERSequence derSourceName =
            SourceNameSerialization.encodeSourceName(sliceInfo.getSourceName());

      DERInteger derTransactionId =
            new DERInteger(BigInteger.valueOf(sliceInfo.getTransactionId()));

      return new DERSequence(new DERObject[]{
            derSourceName, derTransactionId
      });
   }

   /**
    * Decode method for {@link SliceInfo}.
    * 
    * @param sliceInfo
    * @param asnObj
    */
   public static void decodeSliceInfo(SliceInfo sliceInfo, ASN1Object asnObj)
   {
      assert asnObj instanceof DERSequence;
      DERSequence derSliceInfo = (DERSequence) asnObj;

      // Source name
      assert derSliceInfo.getObjectAt(0) instanceof DERSequence;

      DERSequence encodedSliceInfo = (DERSequence) derSliceInfo.getObjectAt(0);
      SourceNameSerialization.decodeSourceName(sliceInfo.getSourceName(), encodedSliceInfo);

      // Transaction id
      assert derSliceInfo.getObjectAt(1) instanceof DERInteger;

      DERInteger transactionId = (DERInteger) derSliceInfo.getObjectAt(1);
      sliceInfo.setTransactionId(transactionId.getValue().longValue());
   }

}
