//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.protocol.MultipleWriteRequest;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link MultipleWriteRequest}.
 * 
 * @see MultipleWriteRequest
 * @author Manish Motwani
 */
public class MultipleWriteRequestSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<MultipleWriteRequest> getSerializableClass()
   {
      return MultipleWriteRequest.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return MultipleWriteRequestSerialization.class.getMethod("encodeMultipleWriteRequest",
               new Class[]{
                  getSerializableClass()
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return MultipleWriteRequestSerialization.class.getMethod("decodeMultipleWriteRequest",
               new Class[]{
                     getSerializableClass(), ASN1Object.class
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link MultipleWriteRequest}.
    * 
    * @param multipleWriteRequest
    * @return
    */
   public static ASN1Object encodeMultipleWriteRequest(MultipleWriteRequest multipleWriteRequest)
   {
      // Data slices
      DataSlice[] dataSlices = multipleWriteRequest.getDataSlices();
      if (dataSlices == null)
      {
         return new DERNull();
      }

      ASN1Encodable[] dataSliceArray = new ASN1Encodable[dataSlices.length];
      for (int i = 0; i < dataSlices.length; ++i)
      {
         if (dataSlices[i] == null)
         {
            dataSliceArray[i] = new DERNull();
         }
         else
         {
            dataSliceArray[i] = DataSliceSerialization.encodeDataSlice(dataSlices[i]);
         }
      }

      return new DERSequence(dataSliceArray);
   }

   /**
    * Decode method for {@link MultipleWriteRequest}.
    * 
    * @param multipleWriteRequest
    * @param asnObj
    */
   public static void decodeMultipleWriteRequest(
         MultipleWriteRequest multipleWriteRequest,
         ASN1Object asnObj)
   {
      if (asnObj instanceof DERNull)
      {
         multipleWriteRequest.setDataSlices((DataSlice[]) null);
      }
      else
      {
         assert asnObj instanceof DERSequence;
         DERSequence derMultipleWriteRequest = (DERSequence) asnObj;

         if (derMultipleWriteRequest.size() == 0)
         {
            multipleWriteRequest.setDataSlices(new DataSlice[0]);
         }
         else
         {
            DataSlice[] dataSlices = new DataSlice[derMultipleWriteRequest.size()];
            for (int i = 0; i < derMultipleWriteRequest.size(); ++i)
            {
               if (derMultipleWriteRequest.getObjectAt(i) instanceof DERNull)
               {
                  dataSlices[i] = null;
               }
               else
               {
                  assert derMultipleWriteRequest.getObjectAt(i) instanceof DERSequence;
                  dataSlices[i] = new DataSlice();
                  DataSliceSerialization.decodeDataSlice(dataSlices[i],
                        (DERSequence) derMultipleWriteRequest.getObjectAt(i));
               }
            }
            multipleWriteRequest.setDataSlices(dataSlices);
         }
      }
   }
}
