//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization.asn1.custom;

import java.lang.reflect.Method;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DERObject;
import org.bouncycastle.asn1.DERSequence;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.protocol.MultipleReadResponse;

/**
 * Implements custom ASN.1 <code>encode</code> and <code>decode</code> methods for
 * {@link MultipleReadResponse}.
 * 
 * @see MultipleReadResponse
 * @author Manish Motwani
 */
public class MultipleReadResponseSerialization implements CustomSerialization
{
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getSerializableClass()
    */
   public Class<MultipleReadResponse> getSerializableClass()
   {
      return MultipleReadResponse.class;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getEncodeMethod()
    */
   public Method getEncodeMethod()
   {
      try
      {
         return MultipleReadResponseSerialization.class.getMethod("encodeMultipleReadResponse",
               new Class[]{
                  getSerializableClass()
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.serialization.asn1.custom.CustomSerialization#getDecodeMethod()
    */
   public Method getDecodeMethod()
   {
      try
      {
         return MultipleReadResponseSerialization.class.getMethod("decodeMultipleReadResponse",
               new Class[]{
                     getSerializableClass(), ASN1Object.class
               });
      }
      catch (NoSuchMethodException nsme)
      {
         throw new RuntimeException("Error in code above.", nsme);
      }
   }

   /**
    * Encode method for {@link MultipleReadResponse}.
    * 
    * @param multipleReadResponse
    * @return
    */
   public static DERSequence encodeMultipleReadResponse(MultipleReadResponse multipleReadResponse)
   {
      // Exception
      ASN1Object exception =
            ResponseExceptionSerializationUtil.encodeResponseException(multipleReadResponse);

      // Data slices
      DataSlice[] dataSlices = multipleReadResponse.getDataSlices();
      if (dataSlices == null)
      {
         return new DERSequence(new DERObject[]{
               exception, new DERNull()
         });
      }

      ASN1Encodable[] dataSliceArray = new ASN1Encodable[dataSlices.length];
      for (int i = 0; i < dataSlices.length; ++i)
      {
         if (dataSlices[i] == null)
         {
            dataSliceArray[i] = new DERNull();
         }
         else
         {
            dataSliceArray[i] = DataSliceSerialization.encodeDataSlice(dataSlices[i]);
         }
      }

      return new DERSequence(new DERObject[]{
            exception, new DERSequence(dataSliceArray)
      });
   }

   /**
    * Decode method for {@link MultipleReadResponse}.
    * 
    * @param multipleReadResponse
    * @param asnObj
    */
   public static void decodeMultipleReadResponse(
         MultipleReadResponse multipleReadResponse,
         ASN1Object asnObj)
   {
      assert asnObj instanceof DERSequence;
      DERSequence derMultipleReadResponse = (DERSequence) asnObj;

      assert derMultipleReadResponse.size() == 2;

      // Exception
      assert derMultipleReadResponse.getObjectAt(0) instanceof ASN1Object;
      ResponseExceptionSerializationUtil.decodeResponseException(multipleReadResponse,
            (ASN1Object) derMultipleReadResponse.getObjectAt(0));

      // Data slices
      if (derMultipleReadResponse.getObjectAt(1) instanceof DERNull)
      {
         multipleReadResponse.setDataSlices((DataSlice[]) null);
      }
      else
      {
         assert derMultipleReadResponse.getObjectAt(1) instanceof DERSequence;
         DERSequence derDataSlices = (DERSequence) derMultipleReadResponse.getObjectAt(1);
         if (derDataSlices.size() == 0)
         {
            multipleReadResponse.setDataSlices(new DataSlice[0]);
         }
         else
         {
            DataSlice[] dataSlices = new DataSlice[derDataSlices.size()];
            for (int i = 0; i < derDataSlices.size(); ++i)
            {
               if (derDataSlices.getObjectAt(i) instanceof DERNull)
               {
                  dataSlices[i] = null;
               }
               else
               {
                  assert derDataSlices.getObjectAt(i) instanceof DERSequence;
                  dataSlices[i] = new DataSlice();
                  DataSliceSerialization.decodeDataSlice(dataSlices[i],
                        (DERSequence) derDataSlices.getObjectAt(i));
               }
            }
            multipleReadResponse.setDataSlices(dataSlices);
         }
      }
   }
}
