//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Jun 22, 2007
//---------------------

package org.cleversafe.serialization.asn1;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Object;
import org.cleversafe.util.NamingHelper;
import org.cleversafe.util.XMLParser;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

// TODO: Describe class or interface
public class ExceptionRegistrator
{
   private static Logger _logger = Logger.getLogger(ExceptionRegistrator.class);

   public static final String ROOT_NAME = "GridProtocol";
   public static final String EXCEPTION_NAME = "Exception";
   public static final String METHOD_NODE = "method";

   public static final String NAME_ATTRIBUTE = "name";
   public static final String CLASS_ATTRIBUTE = "class";
   public static final String ERROR_DEFINITION_FILE = "error-definition.xml";
   public static final String ERROR_DEFINITION_PROPERTY =
         "org.cleversafe.serialization.asn1.error_definition";

   private static final String ENCODER_ATTRIBUTE = "encode";
   private static final String DECODER_ATTRIBUTE = "decode";
   private static final String CODEC_CLASS_ATTRIBUTE = "class";

   private static final String SCHEMA_FILE_NAME =
         NamingHelper.getPackagePathFromClass(ExceptionRegistrator.class) + "/exception-config.xsd";

   static void loadAndRegister()
   {
      Document document;
      try
      {

         String resourceName;
         if (System.getProperty(ERROR_DEFINITION_PROPERTY) != null)
         {
            resourceName = System.getProperty(ERROR_DEFINITION_PROPERTY);
         }
         else
         {
            resourceName =
                  NamingHelper.getPackagePathFromClass(ExceptionRegistrator.class) + File.separator
                        + ERROR_DEFINITION_FILE;
         }
         InputStream inputXml = null;

         try
         {
            inputXml = new FileInputStream(resourceName);
         }
         catch (FileNotFoundException ignore)
         {

         }

         if (inputXml == null)
         {
            inputXml =
                  ExceptionRegistrator.class.getClassLoader().getResourceAsStream(resourceName);
         }

         if (inputXml == null)
         {
            throw new RuntimeException("Can't open resourse '" + resourceName
                  + "' with exception definitions");
         }

         InputStream inputSchema =
               ExceptionRegistrator.class.getClassLoader().getResourceAsStream(SCHEMA_FILE_NAME);
         assert inputSchema != null;

         document = XMLParser.parse(inputXml, inputSchema);
      }
      catch (Exception e)
      {
         throw new RuntimeException(ERROR_DEFINITION_FILE + " not found", e);
      }

      // Get the root node
      final Node protocolNode = document.getChildNodes().item(0);
      assert protocolNode.getNodeName().equals(ROOT_NAME);

      final NodeList children = protocolNode.getChildNodes();
      for (int i = 0; i < children.getLength(); i++)
      {
         final Node exceptionNode = children.item(i);
         if (exceptionNode.getNodeType() == Node.ELEMENT_NODE)
         {
            final String childName = exceptionNode.getNodeName();
            if (childName.equals(EXCEPTION_NAME))
            {
               final Node nameAttr = exceptionNode.getAttributes().getNamedItem(NAME_ATTRIBUTE);
               if (nameAttr == null)
               {
                  throw new RuntimeException(
                        "Can't load network exception definitions: name attribute is missing");
               }
               final String name = nameAttr.getNodeValue();

               final Node classAttr = exceptionNode.getAttributes().getNamedItem(CLASS_ATTRIBUTE);
               if (nameAttr == null)
               {
                  throw new RuntimeException(
                        "Can't load network exception definitions: class attribute is missing");
               }
               final String className = classAttr.getNodeValue();
               Class<?> registeredClass;
               try
               {
                  registeredClass = Class.forName(className);
               }
               catch (ClassNotFoundException e)
               {
                  throw new RuntimeException("Can't load class '" + className + "' for name '"
                        + name + "'");
               }
               assert registeredClass != null;

               _logger.info("Registering exception class '" + className + "' under name '" + name
                     + "'");
               ASN1ExceptionWrapper.registerException(name, registeredClass);

               String encoder = null;
               String decoder = null;
               String codecClassName = null;

               final NodeList exceptionChildren = exceptionNode.getChildNodes();

               for (int j = 0; j < exceptionChildren.getLength(); j++)
               {
                  Node methodNode = exceptionChildren.item(j);
                  if (methodNode.getNodeName().equals(METHOD_NODE))
                  {
                     if (methodNode.getNodeName().equals(METHOD_NODE))
                     {
                        final Node encoderAttr =
                              methodNode.getAttributes().getNamedItem(ENCODER_ATTRIBUTE);
                        if (encoderAttr == null)
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: encoder attribute is missing");
                        }
                        encoder = encoderAttr.getNodeValue();

                        final Node decoderAttr =
                              methodNode.getAttributes().getNamedItem(DECODER_ATTRIBUTE);
                        if (decoderAttr == null)
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: encoder attribute is missing");
                        }
                        decoder = decoderAttr.getNodeValue();

                        final Node codecClassAttr =
                              methodNode.getAttributes().getNamedItem(CODEC_CLASS_ATTRIBUTE);
                        if (codecClassAttr == null)
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: class attribute is missing");
                        }
                        codecClassName = codecClassAttr.getNodeValue();
                     }

                     Method encoderMethod = null;
                     Method decoderMethod = null;
                     Exception codecException = null;

                     if (encoder != null && decoder != null && codecClassName != null)
                     {
                        Class<?> codecClass;
                        try
                        {
                           codecClass = Class.forName(codecClassName);
                        }
                        catch (ClassNotFoundException e)
                        {
                           throw new RuntimeException("Invalid codec definition class "
                                 + codecClassName, e);
                        }
                        // check that methods exist
                        Method m1;

                        try
                        {
                           m1 = codecClass.getMethod(encoder, registeredClass);
                           if (Modifier.isStatic(m1.getModifiers())) // must be
                           // static
                           {
                              encoderMethod = m1;
                           }
                           else
                           {
                              codecException = new Exception("Encoder must be static method");
                           }
                        }
                        catch (Exception ex)
                        {
                           codecException = ex;
                        }
                        if (encoderMethod == null)
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: encoder method "
                                       + encoder + "() is doesn't exist or has a wrong signature",
                                 codecException);
                        }
                        if (!ASN1Object.class.isAssignableFrom(encoderMethod.getReturnType()))
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: encoder method "
                                       + encoder
                                       + "() should return value assignable to ASN1Object");
                        }

                        // check that decoder exist
                        Method m2;

                        try
                        {
                           m2 = codecClass.getMethod(decoder, registeredClass, ASN1Object.class);
                           if (Modifier.isStatic(m2.getModifiers()))
                           {
                              decoderMethod = m2;
                           }
                           else
                           {
                              codecException = new Exception("Decoder must be static method");
                           }
                        }
                        catch (Exception ex)
                        {
                           codecException = ex;
                        }
                        if (decoderMethod == null)
                        {
                           throw new RuntimeException(
                                 "Can't load network exception definitions: decoder method "
                                       + decoder + "() is doesn't exist or has a wrong signature");
                        }

                        // encodeMethod and decodeMethod have to be defined
                        assert encoderMethod != null;
                        assert decoderMethod != null;
                        _logger.info("Registering ASN.1 encode/decode methods, "
                              + encoderMethod.getName() + "/" + decoderMethod.getName()
                              + ", for exception class " + registeredClass.getName());
                        ASN1.DERConverter.registerDefaultEncodeDecode(registeredClass,
                              encoderMethod, decoderMethod);
                     }
                  }
               }
            }
         }
      }
   }

   public static void main(String[] args)
   {
      loadAndRegister();
   }
}
