//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Apr 20, 2007
//---------------------

package org.cleversafe.serialization.asn1;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigInteger;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.DERBoolean;
import org.bouncycastle.asn1.DEREncodable;
import org.bouncycastle.asn1.DERGeneralString;
import org.bouncycastle.asn1.DERInteger;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.DERUTCTime;
import org.cleversafe.serialization.GridSerializable;
import org.cleversafe.serialization.GridSerializable.ElementOrder;
import org.cleversafe.serialization.GridSerializable.NotSerializable;
import org.cleversafe.serialization.GridSerializable.OwnSequence;
import org.cleversafe.serialization.asn1.custom.Registrator;
import org.cleversafe.serialization.exceptions.InvalidClassException;

/*
 * Generic mechanism to ASN.1 serialization/deserialization
 * TODO: Add support for inheritence
 * */
public interface ASN1
{
   public static class DERConverter
   {
      private static final Logger _logger = Logger.getLogger(ASN1.class);

      private static final String encoderMethodName = "asn1Encode";
      private static final String decoderMethodName = "asn1Decode";

      static class TypeConstrDef
      {
         public Class<?> fieldClass; // Field type
         public Class<?> asn1Class; // ASN.1 type

         public Object asn1Creator; // Constructor or static method
         public Method[] asn1Extractor; // Methods used to extract a value
         // that will be submitted to the
         // setter
         public String fieldGetter; // Method to get field value, null if get
         public String fieldSetter; // Method to set field value, null if sett

         public TypeConstrDef(
               final Class<?> fieldClass,
               final Class<?> asn1Class,
               final Object asn1Creator,
               final Method[] asn1Extractor,
               final String fieldGetter,
               final String fieldSetter)
         {
            super();

            this.fieldClass = fieldClass;
            this.asn1Class = asn1Class;
            this.asn1Creator = asn1Creator;
            this.asn1Extractor = asn1Extractor;
            this.fieldGetter = fieldGetter;
            this.fieldSetter = fieldSetter;
         }
      }
      // ASN1 type conversion assembly
      // TODO: Add long, char
      // Static initialization required due to Exceptions
      static private final TypeConstrDef typeConstr[] = buildTypeConversionMap();

      static private TypeConstrDef[] buildTypeConversionMap()
      {
         // Method[] m = ASN1.DERConverter.class.getMethods();
         try
         {
            return new TypeConstrDef[]{
                  new TypeConstrDef(int.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{})
                        }, "getInt", "setInt"),

                  new TypeConstrDef(Integer.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{})
                        }, null, null),

                  new TypeConstrDef(short.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{}),
                              Integer.class.getMethod("shortValue", new Class[]{})
                        }, "getShort", "setShort"),

                  new TypeConstrDef(Short.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{}),
                              Integer.class.getMethod("shortValue", new Class[]{}),
                        }, null, null),

                  new TypeConstrDef(byte.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{}),
                              Integer.class.getMethod("byteValue", new Class[]{}),
                        }, "getByte", "setByte"),

                  new TypeConstrDef(Byte.class, DERInteger.class,
                        DERInteger.class.getConstructor(new Class[]{
                           int.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("intValue", new Class[]{}),
                              Integer.class.getMethod("byteValue", new Class[]{}),
                        }, null, null),

                  new TypeConstrDef(byte[].class, DEROctetString.class,
                        DEROctetString.class.getConstructor(new Class[]{
                           byte[].class
                        }), new Method[]{
                           DEROctetString.class.getMethod("getOctets", new Class[]{}),
                        }, null, null),

                  new TypeConstrDef(long.class, DERInteger.class,
                        ASN1.DERConverter.class.getMethod("convertFromLong", new Class[]{
                           Long.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("longValue", new Class[]{}),
                        }, "getLong", "setLong"),

                  new TypeConstrDef(Long.class, DERInteger.class,
                        ASN1.DERConverter.class.getMethod("convertFromLong", new Class[]{
                           Long.class
                        }), new Method[]{
                              DERInteger.class.getMethod("getValue", new Class[]{}),
                              BigInteger.class.getMethod("longValue", new Class[]{}),
                        }, null, null),

                  new TypeConstrDef(char.class, DERGeneralString.class,
                        ASN1.DERConverter.class.getMethod("convertFromChar", new Class[]{
                           Character.class
                        }), new Method[]{
                           ASN1.DERConverter.class.getMethod("convertToChar", new Class[]{
                              DERGeneralString.class
                           }),
                        }, "getChar", "setChar"),

                  new TypeConstrDef(Character.class, DERGeneralString.class,
                        ASN1.DERConverter.class.getMethod("convertFromChar", new Class[]{
                           Character.class
                        }), new Method[]{
                           ASN1.DERConverter.class.getMethod("convertToChar", new Class[]{
                              DERGeneralString.class
                           }),
                        }, null, null),

                  new TypeConstrDef(String.class, DERGeneralString.class,
                        DERGeneralString.class.getConstructor(new Class[]{
                           String.class
                        }), new Method[]{
                           DERGeneralString.class.getMethod("getString", new Class[]{})
                        }, null, null),

                  new TypeConstrDef(Date.class, DERUTCTime.class,
                        DERUTCTime.class.getConstructor(new Class[]{
                           Date.class
                        }), new Method[]{
                           DERUTCTime.class.getMethod("getDate", new Class[]{})
                        }, null, null),

                  new TypeConstrDef(boolean.class, DERBoolean.class,
                        DERBoolean.class.getConstructor(new Class[]{
                           boolean.class
                        }), new Method[]{
                           DERBoolean.class.getMethod("isTrue", new Class[]{}),
                        }, "getBoolean", "setBoolean"),

                  new TypeConstrDef(Boolean.class, DERBoolean.class,
                        DERBoolean.class.getConstructor(new Class[]{
                           boolean.class
                        }), new Method[]{
                           DERBoolean.class.getMethod("isTrue", new Class[]{}),
                        }, null, null),
            };
         }
         catch (final Exception ex)
         {
            // This exception will only be thrown if the code above is incorrect
            throw new RuntimeException("Invalid ASN.1 definition", ex);
         }
      }
      /*
       * Static map of registered ASN1 interface classes Caches field ordering based on annotations
       */

      private static final Map<Class<?>, Field[]> asn1Order = new HashMap<Class<?>, Field[]>();

      /**
       * The following codec could be omnly applied when an object of a sepcified class resides
       * within a class defined in the second argument The one that is closest in hierarchy will be
       * chosen For example: B --> D B --> C
       */
      private static final Map<Class<?>, Map<Class<?>, Method[]>> externalCodecs =
            new HashMap<Class<?>, Map<Class<?>, Method[]>>();

      public static void clearCachedOrders()
      {
         asn1Order.clear();
      }

      /**
       * Orders class' fields for ASN1 encoding/decoding. Registers class and all base classes that
       * implement ASN1 interface The following annotations are used:
       * 
       * @ASN1ElementOrder(order=1) - will be used at specified position
       * @ASN1ElementOrder(next=true) - next available position is used (default)
       * @NotSerializable - not serialized
       * 
       * @param clazz
       */
      public static void registerASN1Order(final Class<?> clazz)
      {
         if (!asn1Order.containsKey(clazz))
         {
            // Do inheritance traversal
            Field[] orderedSuperFields = null;
            Field[] orderedFields = null;

            final Type superType = clazz.getGenericSuperclass();
            if (superType != null && superType instanceof Class)
            {
               final Class<?> superClass = (Class<?>) superType;
               orderedSuperFields = new Field[0]; // Empty array of superfields
               if (GridSerializable.class.isAssignableFrom(superClass))
               {
                  orderedSuperFields = registerHierarchy(superClass, new Field[0]);
               }
               orderedFields = registerHierarchy(clazz, orderedSuperFields);
            }

            // Finally cache the result
            asn1Order.put(clazz, orderedFields);

            // One may be interest to observe field orders
            if (_logger.isTraceEnabled())
            {
               _logger.trace("ASN1 order for " + clazz.getName());
               int ind = 1;
               for (int i = 0; i < orderedFields.length; i++)
               {
                  if (orderedFields[i] != null)
                  {
                     _logger.trace(ind++ + ". " + orderedFields[i].getName());
                  }
               }
            }
         }
      }

      /**
       * Registers a single class and appends array of field to those created for a base classes
       * 
       * @param clazz
       * @param orderedSuperFields
       * @return
       */
      private static Field[] registerHierarchy(
            final Class<?> clazz,
            final Field[] orderedSuperFields)
      {
         final Field[] fields = clazz.getDeclaredFields();

         final Field[] orderedFields = new Field[fields.length + orderedSuperFields.length];
         // Fill super fields
         System.arraycopy(orderedSuperFields, 0, orderedFields, 0, orderedSuperFields.length);

         // First place numbered
         final int startIndexForThisClass = orderedSuperFields.length;
         for (int i = 0; i < fields.length; i++)
         {
            // first scan all that have index defined
            final ElementOrder annOrder = fields[i].getAnnotation(ElementOrder.class);
            if (Modifier.isStatic(fields[i].getModifiers()) || annOrder == null
                  || annOrder.order() < 1)
            {
               continue;
            }
            final int realOrder = annOrder.order() - 1; // Indexes starts with
            // 1
            if (realOrder >= fields.length)
            {
               // This error is caused by invalid class annotation
               throw new RuntimeException("Invalid ASN1 field number (" + annOrder.order() + ") "
                     + clazz.getName() + "." + fields[i].getName());
            }
            if (orderedFields[realOrder + startIndexForThisClass] != null)
            {
               // This error is caused by invalid class annotation
               throw new RuntimeException("ASN1 field number (" + annOrder.order()
                     + ") used more then once in " + clazz.getName() + "." + fields[i].getName());
            }
            orderedFields[realOrder + startIndexForThisClass] = fields[i];
         }
         // Placed next() and not marked in remaining positions
         int orderedIndex = startIndexForThisClass;
         for (int i = 0; i < fields.length; i++)
         {
            // Skip static and specially annotated
            if (fields[i].isAnnotationPresent(NotSerializable.class)
                  || Modifier.isStatic(fields[i].getModifiers()))
            {
               continue;
            }
            // first scan all that have index defined
            final ElementOrder annOrder = fields[i].getAnnotation(ElementOrder.class);
            if (annOrder == null || annOrder.order() < 0)
            {
               while (orderedFields[orderedIndex] != null)
               {
                  orderedIndex++;
               }
               assert orderedIndex <= fields.length;
               orderedFields[orderedIndex++] = fields[i];
            }
         }
         return orderedFields;
      }

      /**
       * Returns fields in the order defined by class annotations. Some positions may be missed
       * (null will be used). Example:
       * 
       * <pre>
       * class Xyz implements ASN1
       * {
       *    @ASN1ElementOrder(order = 1)
       *    private int a; // position 0
       *    @NotSerializable
       *    private boolean cached; // not serialized
       *    @ASN1ElementOrder(order = 3)
       *    private String b; // position 2
       * }
       * </pre>
       * 
       * @param clazz
       *           class fields to be returned
       * @return array of relevant fields. Some position may be missed
       */
      static protected Field[] getASN1OrderedFields(final Class<?> clazz)
      {
         registerASN1Order(clazz);

         assert asn1Order.containsKey(clazz) == true;
         return asn1Order.get(clazz);
      }

      /**
       * Find an appropriate conversion definition for a field class
       * 
       * @param fieldClass
       * @return
       */
      static public TypeConstrDef getTypeContsrDef(final Class<?> fieldClass)
      {
         for (int convInd = 0; convInd < typeConstr.length; convInd++)
         {
            final TypeConstrDef curConv = typeConstr[convInd];
            if (fieldClass.equals(curConv.fieldClass))
            {
               return curConv;
            }
         }
         return null;
      }

      static private void registerCustomObjects()
      {
         if (!Registrator.getRegistered())
         {
            Registrator.registorCustomObjects();
         }
      }

      /**
       * Checks whether a class implements special serialization. Checks that specvified methods
       * have proper signatures
       * 
       * Array of 4 methods is returned: Method[0] - non-static en-code Method[1] - static en-code
       * Method[2] - non-static de-code Method[3] - static de-code
       * 
       * @param obj
       *           object for which class a special codec is sought
       * @return null if no serialzization is available; array of 4 Methods 2 of which are null
       */
      static private Method[] getSpecialSerialization(final GridSerializable obj)
      {
         // Method[] m = obj.getClass().getMethods();
         final Method[] codecs = new Method[4];
         try
         {
            try
            {
               codecs[0] = obj.getClass().getDeclaredMethod(encoderMethodName, new Class[]{});
               if (Modifier.isStatic(codecs[0].getModifiers()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + encoderMethodName
                        + " can't be static without arguments");
               }
               if (!ASN1Object.class.isAssignableFrom(codecs[0].getReturnType()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + encoderMethodName
                        + " doesn't return ASN1Object");
               }

            }
            catch (final Exception ex)
            {
               codecs[1] = obj.getClass().getDeclaredMethod(encoderMethodName, new Class[]{
                  obj.getClass()
               });
               if (!Modifier.isStatic(codecs[1].getModifiers()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + encoderMethodName
                        + " has to be static 1 argument");
               }
               if (!ASN1Object.class.isAssignableFrom(codecs[1].getReturnType()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + encoderMethodName
                        + " doesn't return ASN1Object");
               }
            }
            try
            {
               codecs[2] = obj.getClass().getDeclaredMethod(decoderMethodName, new Class[]{
                  ASN1Object.class
               });
               if (Modifier.isStatic(codecs[2].getModifiers()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + decoderMethodName
                        + " can't be static with a single argument");
               }
            }
            catch (final Exception ex)
            {
               codecs[3] = obj.getClass().getDeclaredMethod(decoderMethodName, new Class[]{
                     obj.getClass(), ASN1Object.class
               });
               if (!Modifier.isStatic(codecs[3].getModifiers()))
               {
                  // This error is caused by invalid class annotation
                  throw new RuntimeException(obj.getClass() + "." + decoderMethodName
                        + " has to bes tatic with 2 arguments");
               }
            }
         }
         catch (final SecurityException e)
         {
            return null;
         }
         catch (final NoSuchMethodException e)
         {
            return null;
         }
         return codecs;
      }

      /**
       * Registeres method to be used to ASN1 serialize instances of a specified class
       * 
       * @param which
       *           class to use specialized seriallization
       * @param codec
       *           encode/decode ,ethods
       */
      public static void registerDefaultEncodeDecode(
            final Class<?> classToSerialize,
            final Method encode,
            final Method decode)
      {

         registerClassEncodeDecode(classToSerialize, Object.class, encode, decode);
      }

      /**
       * Registeres method to be used to ASN1 serialize instances of a specified class when it is
       * encounter within a specified class
       * 
       * @param which
       *           class to use specialized seriallization
       * @param codec
       *           encode/decode ,ethods
       */
      public static void registerClassEncodeDecode(
            final Class<?> classToSerialize,
            final Class<?> outerClass,
            final Method encode,
            final Method decode)
      {
         Map<Class<?>, Method[]> map = externalCodecs.get(classToSerialize);
         if (map == null)
         {
            map = new HashMap<Class<?>, Method[]>();
            externalCodecs.put(classToSerialize, map);
         }
         map.put(outerClass, checkEncodeDecode(classToSerialize, encode, decode));
      }

      public static void unregisterDefaultEncodeDecode(final Class<?> classToSerialize)
      {
         unregisterClassEncodeDecode(classToSerialize, Object.class);
      }

      public static void unregisterClassEncodeDecode(
            final Class<?> classToSerialize,
            final Class<?> outerClass)
      {
         Map<Class<?>, Method[]> map = externalCodecs.get(classToSerialize);
         if (map != null && map.containsKey(outerClass))
         {
            map.remove(outerClass);
         }
      }

      /**
       * Checks whether external serialization method have proper signatures
       * 
       * @param classToSerialize
       * @param encode
       *           method
       * @param decode
       *           method
       * @return
       */
      private static Method[] checkEncodeDecode(
            final Class<?> classToSerialize,
            final Method encode,
            final Method decode)
      {
         // Make sure the signature is proper
         // public static <ASN1Object-assignable> <method>(<classToSerialize>)
         // public static void <method>(<classToSerialize> obj, ASN1Object
         // asn1Rep)
         if (encode == null || decode == null)
            throw new IllegalArgumentException("Both encode and decode should be defined");
         final String encodeRef = "Specialized serialization " + encode.getName();
         if (!Modifier.isStatic(encode.getModifiers()))
            throw new RuntimeException(encodeRef + " has to be static ");
         if (encode.getParameterTypes().length != 1)
            throw new RuntimeException(encodeRef + " has to have 1 argument");
         if (!encode.getParameterTypes()[0].equals(classToSerialize))
            throw new RuntimeException(encodeRef + " has to have 1-st arg "
                  + classToSerialize.getName());
         if (!ASN1Object.class.isAssignableFrom(encode.getReturnType()))
            throw new RuntimeException(encodeRef + " has to have return type ASN1Object assignable");

         final String decodeRef = "Specialized serialization " + decode.getName();
         if (!Modifier.isStatic(decode.getModifiers()))
            throw new RuntimeException(decodeRef + " has to be static ");
         if (decode.getParameterTypes().length != 2)
            throw new RuntimeException(decodeRef + " has to have 2 arguments");
         if (!decode.getParameterTypes()[0].equals(classToSerialize))
            throw new RuntimeException(decodeRef + " has to have 1-st arg "
                  + classToSerialize.getName());
         if (!decode.getParameterTypes()[1].equals(ASN1Object.class))
            throw new RuntimeException(decodeRef + " has to have 2-st arg ASN1Object");
         return new Method[]{
               encode, decode
         };
      }

      /**
       * External codec finder
       * 
       * @param clazz
       *           for which it should be found
       * @return encode/decode method or , null if not found
       */
      public static Method[] getExternalCodecs(final Class<?> clazz, final Class<?> outerClass)
      {
         registerCustomObjects();

         if (clazz.equals(Object.class))
            return null;

         // Try to find a direct match
         Method[] codec = getExternalCodecsExact(clazz, outerClass);
         if (codec != null)
         {
            return codec;
         }
         // Try all recursively interfaces of this class
         codec = getExternalCodecsInterfaces(clazz.getInterfaces(), outerClass);
         if (codec != null)
            return codec;
         // Finally try recursively base class
         return getExternalCodecs(clazz.getSuperclass(), outerClass);
      }

      /**
       * Try to find an interface for which special codec is available
       * 
       * @param interfaces
       * @param outerClass
       * @return
       */
      private static Method[] getExternalCodecsInterfaces(
            final Class<?>[] interfaces,
            final Class<?> outerClass)
      {
         if (interfaces == null)
         {
            return null;
         }
         // First immediate inetrfaces
         for (int i = 0; i < interfaces.length; i++)
         {
            final Method[] codec = getExternalCodecsExact(interfaces[i], outerClass);
            if (codec != null)
            {
               return codec;
            }
         }
         // All subinterfaces
         for (int i = 0; i < interfaces.length; i++)
         {
            final Method[] codec =
                  getExternalCodecsInterfaces(interfaces[i].getInterfaces(), outerClass);
            if (codec != null)
            {
               return codec;
            }
         }

         return null;
      }

      // Searches for a best match for a specific class
      private static Method[] getExternalCodecsExact(final Class<?> clazz, final Class<?> outerClass)
      {
         final Map<Class<?>, Method[]> classCodecs = externalCodecs.get(clazz);
         Class<?> bestFit = null;
         Method[] bestMethods = null;
         if (classCodecs != null)
         {
            final Set<Class<?>> candiates = classCodecs.keySet();
            for (final Class<?> cnd : candiates)
            {
               if (outerClass != null && !cnd.isAssignableFrom(outerClass)) // can
               // be
               // used?
               {
                  continue;
               }
               if (bestFit == null || bestFit.isAssignableFrom(cnd))
               {
                  bestFit = cnd;
                  bestMethods = classCodecs.get(bestFit);
               }
            }
         }
         return bestMethods;
      }

      /**
       * Encodes any data members that are of supported types into an ASN.1 byte array.
       * 
       * @return ASN.1 DER encoded byte array
       * @throws InvalidClassException
       */
      @SuppressWarnings("unchecked")
      static public byte[] encode(final Object object) throws InvalidClassException
      {
         final ByteArrayOutputStream out;

         try
         {
            // Create DERSequence and byte stream
            out = new ByteArrayOutputStream();
            final DERSequenceGenerator derGenerator = new DERSequenceGenerator(out);

            // Recursevely cerate a list of objects that have to be added to
            // serialization
            // Any primitive class is represented with ASN1Object
            // Any array of primitives is represented with a DERSequence
            // Non primitive is represnted with its
            // Array of non-primitives is represented with a list
            final List<Object> asn1Values = new LinkedList<Object>();
            encodeLevel(object, null, asn1Values, false);
            for (Object value : asn1Values)
            {
               if (value instanceof List)
               {
                  derGenerator.addObject(generateASN1((List) value));
               }
               else
               {
                  assert value instanceof ASN1Object;
                  derGenerator.addObject((ASN1Object) value);
               }
            }

            // Close DERSequence generator and byte stream
            derGenerator.close();
            out.close();
         }
         catch (final IOException e)
         {
            // If this occurs, there is a serious problem
            throw new RuntimeException("Encoding IO error", e);
         }

         return out.toByteArray();
      }

      // 2-nd pass of encoding
      @SuppressWarnings("unchecked")
      private static DERSequence generateASN1(final List<Object> asn1Values)
      {
         final ASN1Object[] derValues = new ASN1Object[asn1Values.size()];
         int derIndex = 0;

         for (Object value : asn1Values)
         {
            if (value instanceof List)
            {
               derValues[derIndex++] = generateASN1((List) value);
            }
            else
            {
               assert value instanceof ASN1Object;
               derValues[derIndex++] = (ASN1Object) value;
            }
         }
         return new DERSequence(derValues);
      }

      /**
       * Encodes one level only. Both decode and encoder should use the same class definition in
       * order to work properly
       * 
       * @param derGenerator -
       *           generator keeps currents state of encoding
       * @param asn1Obj -
       *           Object to be serialized
       */
      static private void encodeLevel(
            final Object obj,
            final Class<?> outerClass,
            final List<Object> genList,
            final boolean newList) throws InvalidClassException
      {
         try
         {
            // Extrenal serialization is allowed to all classes
            // That's how 3-rd party object can be converted to ASN1
            final Method[] externalCodec = getExternalCodecs(obj.getClass(), outerClass);
            if (externalCodec != null)
            {
               final Object asn1Object = externalCodec[0].invoke(null, obj);
               genList.add(asn1Object);
               return;
            }

            // The rest assumes ASN1Serializable
            GridSerializable asn1Obj = null;
            if (obj instanceof GridSerializable)
            {
               asn1Obj = (GridSerializable) obj;
            }
            else
            {
               throw new InvalidClassException(
                     "Class "
                           + obj.getClass().getName()
                           + " does not implement ASN1Serializable and no custom serialization was found.");
            }

            // Check whether this serializable has special mechanism
            // to serialize itself into ASN1
            final Method[] inClassCodec = getSpecialSerialization(asn1Obj);
            if (inClassCodec != null)
            {
               assert inClassCodec[0] != null || inClassCodec[1] != null;
               Object asn1Object;
               if (inClassCodec[0] != null)
               {
                  asn1Object = inClassCodec[0].invoke(asn1Obj, new Object[]{});
               }
               else if (inClassCodec[1] != null)
               {
                  asn1Object = inClassCodec[1].invoke(null, new Object[]{
                     asn1Obj
                  });
               }
               else
               {
                  throw new IllegalStateException("Can't be here");
               }
               genList.add(asn1Object);
               return;
            }

            // Non-special processing, do the best
            final Field[] fields = getASN1OrderedFields(asn1Obj.getClass());

            // Load all data member into ASN.1 sequence
            List<Object> buildList = null;
            if (newList == false)
            {
               buildList = genList; // Keep adding to the old list
            }
            else
            {
               buildList = new LinkedList<Object>(); // Own sequence was specified
               genList.add(buildList);
            }

            // Looping by all fields
            for (final Field field : fields)
            {
               boolean isArray = false;
               boolean isList = false;

               if (field == null)
               {
                  continue;
               }

               // Skip static fields
               if (Modifier.isStatic(field.getModifiers()))
               {
                  continue;
               }

               field.setAccessible(true);
               Class<?> elementClass = null;
               final Class<?> fieldClass = field.getType();

               // Check for null value
               if (field.get(obj) == null)
               {
                  buildList.add(new DERNull());
                  continue;
               }

               TypeConstrDef curConv = getTypeContsrDef(fieldClass);
               if (curConv == null) // Try special processing if array or list
               {
                  final Class<?> listElementSuspect = getListElementClass(field);
                  if (fieldClass.isArray())
                  {
                     isArray = true;
                     elementClass = fieldClass.getComponentType();
                  }
                  else if (listElementSuspect != null)
                  {
                     isList = true;
                     elementClass = listElementSuspect;
                  }
                  if (elementClass != null) // try it with array or list
                  {
                     curConv = getTypeContsrDef(elementClass);
                  }
               }
               else
               {
                  elementClass = fieldClass;
               }

               if (curConv != null)
               {
                  // Type found
                  if (isArray)
                  {
                     // Create and initialize an array to be set as a sequence
                     final Object array = field.get(asn1Obj);
                     final int arrayLength = Array.getLength(array);

                     final ASN1Object[] derValues = new ASN1Object[arrayLength];
                     for (int arrayInd = 0; arrayInd < arrayLength; arrayInd++)
                     {
                        final Object elementValue = Array.get(array, arrayInd);
                        derValues[arrayInd] = createASN1Value(elementValue, curConv.asn1Creator);
                     }
                     buildList.add(new DERSequence(derValues));
                     if (_logger.isTraceEnabled())
                        _logger.trace("E: Array[" + elementClass.getName() + "]");

                  }
                  else if (isList)
                  {
                     final List l = (List) field.get(asn1Obj);
                     final ASN1Object[] derValues = new ASN1Object[l.size()];
                     for (int arrayInd = 0; arrayInd < l.size(); arrayInd++)
                     {
                        final Object elementValue = l.get(arrayInd);
                        derValues[arrayInd] = createASN1Value(elementValue, curConv.asn1Creator);
                     }
                     buildList.add(new DERSequence(derValues));
                     if (_logger.isTraceEnabled())
                        _logger.trace("E: Array[" + elementClass.getName() + "]");
                  }
                  else
                  {
                     // Set a single value.
                     Object fieldValue = null;
                     if (curConv.fieldGetter != null)
                     {
                        final Method fieldGetter =
                              Field.class.getMethod(curConv.fieldGetter, new Class[]{
                                 Object.class
                              });
                        fieldValue = fieldGetter.invoke(field, new Object[]{
                           asn1Obj
                        });
                     }
                     else
                     {
                        // Default getter from a field is get()
                        fieldValue = field.get(asn1Obj);
                     }
                     buildList.add(createASN1Value(fieldValue, curConv.asn1Creator));
                     if (_logger.isTraceEnabled())
                        _logger.trace("E: " + elementClass.getName());
                  }
               }
               else
               {
                  // Non primitive and not array of primitives
                  assert elementClass == null || !elementClass.isPrimitive();

                  if (isArray)
                  {
                     // This is an array of non ASN1 serializable types
                     // Create a list of lists
                     final Object array = field.get(asn1Obj);
                     final int arrayLength = Array.getLength(array);
                     final List<Object> subGenlist = new LinkedList<Object>();
                     for (int ind = 0; ind < arrayLength; ind++)
                     {
                        encodeLevel(Array.get(array, ind), asn1Obj.getClass(), subGenlist, false);
                     }
                     buildList.add(subGenlist);
                     if (_logger.isTraceEnabled())
                        _logger.trace("E: Array[" + elementClass.getName() + "]");
                  }
                  else if (isList)
                  {
                     final List l = (List) field.get(asn1Obj);

                     final List<Object> subGenlist = new LinkedList<Object>();
                     for (int ind = 0; ind < l.size(); ind++)
                     {
                        encodeLevel(l.get(ind), asn1Obj.getClass(), subGenlist, false);
                     }
                     buildList.add(subGenlist);
                     if (_logger.isTraceEnabled())
                        _logger.trace("E: List[" + elementClass.getName() + "]");

                  }
                  else
                  {
                     // Traverse a complex type recursevely
                     final boolean isSequence =
                           field.isAnnotationPresent(OwnSequence.class) ? true : false;

                     encodeLevel(field.get(asn1Obj), asn1Obj.getClass(), buildList, isSequence);
                  }
                  // Array of non primitives
               }
            }
         }
         catch (final IllegalAccessException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InvocationTargetException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final NoSuchMethodException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }

      }

      /**
       * Simple convinience method to instantiate ASN1 field
       * 
       * @param fieldValue
       * @param asn1Creator
       * @return
       * @throws InvalidClassException
       */
      static private ASN1Object createASN1Value(final Object fieldValue, final Object asn1Creator)
            throws InvalidClassException
      {
         try
         {
            if (asn1Creator instanceof Method)
            {
               final Method ctr = (Method) asn1Creator;
               return (ASN1Object) ctr.invoke(null, fieldValue);
            }
            else
            {
               final Constructor<?> ctr = (Constructor<?>) asn1Creator;
               Object asn1Value = ctr.newInstance(new Object[]{
                  fieldValue
               });
               assert asn1Value instanceof ASN1Object;
               return (ASN1Object) asn1Value;
            }
         }
         catch (final IllegalAccessException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InvocationTargetException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InstantiationException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
      }

      /**
       * Decodes a given ASN.1 encoded byte array. The encoded byte array must be the same structure
       * as the class responsible for decoding. For example, if the class contains a String and an
       * Integer, then the byte array must also contain an encoded String and Integer. An exception
       * will occur if the decoding process cannot proceed.
       * 
       * @param bytes
       *           A properly formatted ASN.1 DER byte array
       * @throws InvalidClassException
       */
      static public void decode(final Object object, final byte[] bytes)
            throws InvalidClassException
      {
         try
         {
            // Create DERSequence from byte stream
            final ASN1InputStream aIn = new ASN1InputStream(bytes);
            final DERSequence derSequence = (DERSequence) aIn.readObject();
            decodeLevel(object, null, derSequence, 0);
         }
         catch (final IOException e)
         {
            throw new RuntimeException("IO error in decode", e);
         }
      }

      /**
       * Fills object obj from ASN.1 coding
       * 
       * @param asn1Obj -
       *           object to be filled
       * @param derSequence
       *           ASN.1 sequence used to buid an object
       * @param count -
       *           global counter if extracted objects
       * @return count after object is processed
       * @throws InvalidClassException
       */
      @SuppressWarnings("unchecked")
      static private int decodeLevel(
            final Object obj,
            final Class<?> outreClass,
            final DERSequence derSequence,
            int count) throws InvalidClassException
      {
         assert obj != null;

         try
         {
            // Try external decoding
            final Method[] externalCodec = getExternalCodecs(obj.getClass(), outreClass);
            if (externalCodec != null)
            {
               externalCodec[1].invoke(null, new Object[]{
                     obj, derSequence.getObjectAt(count)
               });
               return count + 1;
            }
            // The rest assumes ASN1Serializable
            GridSerializable asn1Obj = null;
            if (obj instanceof GridSerializable)
            {
               asn1Obj = (GridSerializable) obj;
            }
            else
            {
               throw new InvalidClassException(
                     "Class "
                           + obj.getClass().getName()
                           + " does not implement ASN1Serializable and no custom serialization was found.");
            }

            // Check whether this serializable has special mechanism
            // to serialize itself into ASN1
            final Method[] inClassCodec = getSpecialSerialization(asn1Obj);
            if (inClassCodec != null)
            {
               assert inClassCodec[2] != null || inClassCodec[3] != null;
               if (inClassCodec[2] != null)
               {
                  inClassCodec[2].invoke(asn1Obj, new Object[]{
                     derSequence.getObjectAt(count)
                  });
               }
               else if (inClassCodec[3] != null)
               {
                  inClassCodec[3].invoke(null, new Object[]{
                        asn1Obj, derSequence.getObjectAt(count)
                  });
               }
               else
               {
                  throw new IllegalStateException("Can't be here");
               }
               return count + 1;
            }

            // Try built-in methods
            // Sort fields by name
            final Field[] fields = getASN1OrderedFields(asn1Obj.getClass());

            // Load ASN.1 elements into class members
            for (int i = 0; i < fields.length; i++)
            {
               boolean isArray = false;
               boolean isList = false;

               final Field field = fields[i];
               if (field == null)
               {
                  continue;
               }

               // Skip static fields
               if (Modifier.isStatic(field.getModifiers()))
               {
                  continue;
               }
               field.setAccessible(true);
               final Class<?> fieldClass = field.getType();
               Class<?> elementClass = null; // type of a field or type of collection
               // element
               // Extract next value, it will be used to populate a current field
               final DEREncodable derEncodable = derSequence.getObjectAt(count);

               if (derEncodable instanceof DERNull)
               {
                  field.set(obj, null);
                  count++;
                  continue;
               }

               // Special processing for array of field
               TypeConstrDef curConv = getTypeContsrDef(fieldClass);

               final Class<?> listElementSuspect = getListElementClass(field);
               if (curConv == null) // Try array or list
               {

                  if (fieldClass.isArray())
                  {
                     elementClass = fieldClass.getComponentType();
                     isArray = true;
                  }
                  else if (listElementSuspect != null)
                  {
                     isList = true;
                     elementClass = listElementSuspect;
                  }
                  if (elementClass != null) // try it with array or list
                  {
                     curConv = getTypeContsrDef(elementClass);
                  }

               }
               else
               {
                  elementClass = fieldClass;
               }

               if (curConv != null)
               {
                  // Type found
                  if (isArray)
                  {
                     // Create a sequence and populate it with element values
                     final DERSequence array = (DERSequence) derEncodable;
                     final Object values = Array.newInstance(elementClass, array.size());
                     for (int arrayInd = 0; arrayInd < array.size(); arrayInd++)
                     {
                        // Retrieve a value from ASN1 and convert it to a field
                        // element value
                        final Object value =
                              createFieldValue(array.getObjectAt(arrayInd), curConv.asn1Extractor);
                        Array.set(values, arrayInd, value);
                     }
                     if (_logger.isTraceEnabled())
                        _logger.trace("D: Array[" + elementClass.getName() + "]");
                     field.set(asn1Obj, values);
                  }
                  else if (isList)
                  {
                     final DERSequence array = (DERSequence) derEncodable;
                     final List l = (List) field.get(asn1Obj);
                     if (l == null)
                     {
                        throw new InvalidClassException("List " + field.getName() + " in class "
                              + asn1Obj.getClass().getName() + " should be created before decoding");
                     }
                     l.clear();

                     for (int listInd = 0; listInd < array.size(); listInd++)
                     {
                        // Retrieve a value from ASN1 and convert it to a field
                        // element value
                        final Object value =
                              createFieldValue(array.getObjectAt(listInd), curConv.asn1Extractor);
                        l.add(value);
                     }
                     if (_logger.isTraceEnabled())
                        _logger.trace("D: List[" + elementClass.getName() + "]");

                  }
                  else
                  {
                     // Retrive a value from ASN1 and convert it to a field element
                     // value
                     final Object value = createFieldValue(derEncodable, curConv.asn1Extractor);
                     if (curConv.fieldSetter == null)
                     {
                        field.set(asn1Obj, value);
                     }
                     else
                     {
                        final Method fieldSetter =
                              Field.class.getMethod(curConv.fieldSetter, new Class[]{
                                    Object.class, elementClass
                              });
                        fieldSetter.invoke(field, new Object[]{
                              asn1Obj, value
                        });
                     }
                     if (_logger.isTraceEnabled())
                        _logger.trace("D: " + elementClass.getName());
                  }
                  count++;
               }
               else
               {
                  // Non primitive and not array of ASN.1 primitives
                  assert !fieldClass.isPrimitive();

                  if (isArray || isList)
                  {
                     // Create a list of elements that will be added to array
                     List<Object> list;
                     final Class<?> listType = field.getType();
                     if (!isArray && !listType.isInterface()) // List of known
                     // type
                     {
                        // Create the desired class
                        list = (List) listType.newInstance();
                     }
                     else
                     {
                        list = new LinkedList<Object>(); // best we can do
                     }

                     final DERSequence sequence = (DERSequence) derEncodable;
                     int subCount = 0;
                     while (subCount < sequence.size())
                     {
                        final Object newObj = elementClass.newInstance();
                        subCount = decodeLevel(newObj, asn1Obj.getClass(), sequence, subCount);
                        list.add(newObj);
                     }
                     // Finally add an array to the obj
                     if (isArray)
                     {
                        final Object[] finalArray =
                              (Object[]) Array.newInstance(elementClass, list.size());
                        list.toArray(finalArray);
                        field.set(asn1Obj, finalArray);
                     }
                     else
                     {
                        field.set(asn1Obj, list);
                     }
                     count++;
                  }
                  else
                  {
                     // Traverse a complex type recursevely
                     final boolean isSequence =
                           field.isAnnotationPresent(OwnSequence.class) ? true : false;
                     // Try to instantiate an object if it is not done yet
                     if (field.get(asn1Obj) == null)
                     {
                        try
                        {
                           Constructor fieldConstructor = fieldClass.getConstructor();
                           field.set(asn1Obj, fieldConstructor.newInstance());
                        }
                        catch (Exception ex)
                        {
                           throw new InvalidClassException(
                                 "Can't construct a default object of type '"
                                       + elementClass.getName() + "' and it was not created before");
                        }
                     }
                     if (isSequence)
                     {
                        // Must be sequence
                        final DERSequence newSequence =
                              (DERSequence) derSequence.getObjectAt(count);
                        decodeLevel(field.get(asn1Obj), asn1Obj.getClass(), newSequence, 0);
                        count++; // Onlu sequence was processed
                     }
                     else
                     {
                        count =
                              decodeLevel(field.get(asn1Obj), asn1Obj.getClass(), derSequence,
                                    count);
                     }
                  }
               }
            }
         }
         catch (final IllegalAccessException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InvocationTargetException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final NoSuchMethodException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InstantiationException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }

         return count;
      }

      static private Object createFieldValue(Object value, final Method[] extractors)
            throws InvalidClassException
      {
         try
         {
            // Retrive a value from ASN1 and convert it to a field element value
            for (int j = 0; j < extractors.length; j++)
            {
               final Method m = extractors[j];
               if (Modifier.isStatic(m.getModifiers()))
               {
                  value = m.invoke(null, value);
               }
               else
               {
                  value = m.invoke(value, new Object[]{});
               }
            }
         }
         catch (final IllegalAccessException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }
         catch (final InvocationTargetException e)
         {
            // Reflection-related
            throw new InvalidClassException(e);
         }

         return value;
      }

      private static Class<?> getListElementClass(final Field field)
      {
         if (List.class.isAssignableFrom(field.getType()))
         {
            final Type type = field.getGenericType();
            if (type == null || !(type instanceof ParameterizedType))
            {
               throw new RuntimeException(
                     "ASN1 serialization doesn't support non-parametrized List");
            }
            final ParameterizedType generic = (ParameterizedType) type;
            assert generic.getActualTypeArguments().length == 1;
            return (Class<?>) generic.getActualTypeArguments()[0];
         }
         return null;
      }

      /**
       * long value is converted to ASN.1 sequence of 2 DERInteger
       * 
       * @param value
       *           to convert
       * @return ASN1Object
       */
      @SuppressWarnings("unused")
      static public DERInteger convertFromLong(final Long value)
      {
         final BigInteger b = new BigInteger(value.toString());
         return new DERInteger(b);
      }

      /**
       * char value is converted to DERGeneralString
       * 
       * @param value
       *           to convert
       * @return ASN1Object
       */
      @SuppressWarnings("unused")
      static public DERGeneralString convertFromChar(final Character value)
      {
         return new DERGeneralString(new String(new char[]{
            value
         }));
      }

      /**
       * char value is restored from a ASN.1 string
       * 
       * @param ints
       * @return
       */
      @SuppressWarnings("unused")
      static public Character convertToChar(final DERGeneralString character)
      {
         final String s = character.getString();
         return s.charAt(0);
      }

      /**
       * Convinience method to print serialization maps used for different classes
       * 
       * @param clazz
       * @return
       */
      static public String serializationMap(final Class<?> clazz)
      {
         final Field[] fields = (Field[]) asn1Order.get(clazz);

         if (fields == null)
         {
            return clazz.getName() + "is not registred for ASN1";
         }
         final StringBuffer res = new StringBuffer();
         res.append("ASN1 map for " + clazz.getName());
         // Load all data member into ASN.1 sequence
         for (int i = 0; i < fields.length; i++)
         {
            boolean isArray = false;

            final Field field = fields[i];
            if (field == null)
            {
               continue;
            }

            // Skip static fields
            if (Modifier.isStatic(field.getModifiers()))
            {
               continue;
            }

            Class<?> fieldClass = field.getType();

            if (fieldClass.isArray())
            {
               fieldClass = fieldClass.getComponentType();
               isArray = true;
            }

            res.append("\n" + fieldClass.getName());
            if (isArray)
            {
               res.append("[]");
            }
         }
         return res.toString();
      }
   }
}
