//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER


package org.cleversafe.serialization;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Set;

import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException;
import org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;

/**
 * Interface for Protocol Message Factories that serialize and deserialize protocol message objects
 * to and from byte arrays.
 */
public interface ProtocolMessageFactory
{
   /**
    * This method serializes a protocol message and writes it to the output stream
    * 
    * @param message
    *           The ProtocolMessage object to be serialized.
    * @throws ProtocolSerializationException
    */
   void serialize(DataOutputStream out, ProtocolMessage message)
         throws ProtocolSerializationException;

   /**
    * This method deserializes a data input stream protocol payload to a ProtocolMessage object.
    * 
    * @param operationCode
    *           The operation that is being requested, extracted from the header
    * @param payload
    *           The payload data input stream.
    * @return The deserialized ProtocolMessage object.
    * @throws OperationNotRegisteredException,
    *            DeserializationException
    */
   ProtocolMessage deserialize(int operationCode, DataInputStream payload)
         throws OperationNotRegisteredException, ProtocolDeserializationException;

   /**
    * This method is used to lookup what classes this factory can handle
    * 
    * @return Set of classes
    */
   Set<Class<?>> getMessageTypes();

   /**
    * Get the class that is used to encode this message
    */
   Class<?> getEncoderClass();

   /**
    * Get the class that is used to decode this message
    */
   Class<?> getDecoderClass();

   /**
    * Calculates the maximum size of a serializes message
    * 
    * @return size of serialized message.
    */
   public int getMaxSerializedSize(ProtocolMessage message);
}
