//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: May 10, 2007
//---------------------

package org.cleversafe.layer.slicestore.remote;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Authenticator;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.authenticators.PasswordAuthenticator;
import org.cleversafe.authentication.exceptions.AuthenticationException;
import org.cleversafe.authentication.exceptions.RemoteAuthenticationException;
import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.EndSessionRequest;
import org.cleversafe.layer.protocol.ExistsRequest;
import org.cleversafe.layer.protocol.ExistsResponse;
import org.cleversafe.layer.protocol.IntegrityVerificationRequest;
import org.cleversafe.layer.protocol.IntegrityVerificationResponse;
import org.cleversafe.layer.protocol.ListBeginRequest;
import org.cleversafe.layer.protocol.ListContinueRequest;
import org.cleversafe.layer.protocol.ListContinueResponse;
import org.cleversafe.layer.protocol.ListInProgressRequest;
import org.cleversafe.layer.protocol.ListInProgressResponse;
import org.cleversafe.layer.protocol.ListStopRequest;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleRemoveRequest;
import org.cleversafe.layer.protocol.MultipleRemoveResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.RemoveStoreRequest;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.RollbackTransactionRequest;
import org.cleversafe.layer.protocol.VaultBindRequest;
import org.cleversafe.layer.protocol.VaultBindResponse;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreInfo;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.server.exceptions.ServerRequestException;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultDescriptorException;

/**
 * Implements SliceStore interface through remote procedure calls sent over the network to a remote
 * SliceServer. The remote SliceServer in turn utilizes another instance of a SliceStore to store
 * slices locally.
 */
public class RemoteSliceStore extends SliceStoreBase
{
   private static Logger logger = Logger.getLogger(RemoteSliceStore.class);

   private UUID vaultIdentifier = null;
   private Connector connection;

   private SliceStoreInfo remoteSliceStoreInfo = null;

   // application connection settings
   // During startSession() these members are only used if ( conn == null )

   // TODO: Fix it with support of non primitive types
   private Credentials credentials = null;

   private boolean isSessionActive = false;

   /**
    * Constructor required by framework to instantiate from configuration
    */
   public RemoteSliceStore()
   {
   }

   /**
    * Constructs a Remote Slice Store with an existing application connection. Binding the vault to
    * the session is handled internally by the Remote Slice Store when startSession() is called.
    * setCredentials must be used before attempting to call startSession().
    * 
    * @param vaultIdentifier
    *           The vault which this slice store belongs to.
    * @param conn
    *           A connection to an application.
    * @param remoteSliceStoreInfo
    *           Remote slice store information.
    */
   public RemoteSliceStore(
         final UUID vaultIdentifier,
         final Connector conn,
         final SliceStoreInfo remoteSliceStoreInfo)
   {
      this();
      assert (conn != null);
      this.connection = conn;
      this.vaultIdentifier = vaultIdentifier;
      this.remoteSliceStoreInfo = remoteSliceStoreInfo;
   }

   /**
    * Setter for remote slice store info
    * 
    * @param remoteSliceStoreInfo
    */
   public void setRemoteSliceStoreInfo(final SliceStoreInfo remoteSliceStoreInfo)
   {
      this.remoteSliceStoreInfo = remoteSliceStoreInfo;
   }

   /**
    * Constructs a Remote Slice Store with an existing application connection. Binding the vault to
    * the session is handled internally by the Remote Slice Store when startSession() is called.
    * 
    * @param vaultIdentifier
    *           The vault which this slice store belongs to.
    * @param credentials
    *           the credentials for the account
    * @param conn
    *           A connection to an application.
    */
   public RemoteSliceStore(
         final UUID vaultIdentifier,
         final Credentials credentials,
         final Connector conn)
   {
      this();
      assert (conn != null);

      this.vaultIdentifier = vaultIdentifier;
      this.connection = conn;
      this.credentials = credentials;
   }

   public Connector getConnection()
   {
      return this.connection;
   }

   /**
    * Setter for connection
    * 
    * @param connection
    *           the connection to set
    */
   @RequiredForInitialization
   public void setConnection(final Connector connection)
   {
      assert (connection != null);
      this.connection = connection;
   }

   /**
    * Getter for credentials
    * 
    * @return the credentials
    */
   public Credentials getCredentials()
   {
      return this.credentials;
   }

   /**
    * @param credentials
    *           the credentials to set
    */
   @RequiredForInitialization
   public void setCredentials(final Credentials credentials)
   {
      this.credentials = credentials;
   }

   /**
    * Getter for vaultIdentifier
    * 
    * @return the vaultIdentifier
    */
   public UUID getVaultIdentifier()
   {
      return this.vaultIdentifier;
   }

   /**
    * Setter for vault identifier
    * 
    * @param vaultIdentifier
    *           the vaultIdentifier to set
    */
   @RequiredForInitialization
   public void setVaultIdentifier(final UUID vaultIdentifier)
   {
      this.vaultIdentifier = vaultIdentifier;
   }

   /**
    * Initializes the RemoteSliceStore
    */
   public void initialize()
   {
      if (this.connection == null || this.vaultIdentifier == null || this.credentials == null)
      {
         throw new InitializationException(
               "connection, credentials and/or vault UUID and other remote slice store information should be defined for a remote store");
      }

      logger.debug("Initilized remote slice store for " + getIdentification());
   }

   /**
    * @see SliceStore
    */
   public String getIdentification()
   {
      return "Remote [" + connection.getIdentification() + "]";
   }

   /**
    * @see SliceStore
    */
   public void createStore(
         final String vaultType,
         final long maxSliceSize,
         final long sliceStoreSize,
         final VaultACL accessControlList,
         final byte[] vaultDescriptorBytes,
         final Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      // Store must not have an open session
      if (this.isSessionActive)
      {
         throw new IllegalStateException(
               "RemoteSliceStore.createStore() cannot be called on a store with an open session");
      }

      assert (this.vaultIdentifier != null) : "The vault identifier must be set";

      try
      {
         this.connection.ensureConnected();
      }
      catch (CommunicationException e)
      {
         throw new SliceStoreIOException("Unable to connect to slice store: " + this, e);
      }

      // Authenticate
      authenticate();

      // Send a Create Store Request
      // TODO: Check for the case when the vault already exists, allow a new vault ID to be
      // selected.
      // This would have to be done at a higher level to coordinate the selection of the same
      // UUID. - JKR
      try
      {
         if (this.remoteSliceStoreInfo == null)
         {
            // Default slice store type, if not specified in
            // vault descriptor
            this.remoteSliceStoreInfo = new SliceStoreInfo()
            {
            };
         }

         // Copy the options over to remote slice store info
         for (String key : options.keySet())
         {
            this.remoteSliceStoreInfo.setProperty(key, options.get(key));
         }

         send(new CreateStoreRequest(vaultType, maxSliceSize, sliceStoreSize,
               this.remoteSliceStoreInfo.getSliceStoreType(), this.vaultIdentifier,
               accessControlList, vaultDescriptorBytes, this.remoteSliceStoreInfo.getProperties()));
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      try
      {
         this.connection.disconnect();
      }
      catch (final CommunicationIOException e)
      {
         throw new SliceStoreIOException("Communication error", e);
      }
      catch (final CommunicationInterruptedException e)
      {
         throw new SliceStoreIOException("Communication interrupted error", e);
      }

      logger.debug("RemoteSliceStore successfully created store");
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void deleteStore() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      // Store must not have an open session
      if (this.isSessionActive)
      {
         throw new IllegalStateException(
               "RemoteSliceStore.deleteStore() cannot be called on a store with an open session");
      }

      assert (this.connection != null) : "The connection must be set";
      assert (this.vaultIdentifier != null) : "The vault identifier must be set";

      try
      {
         ensureConnected();

         // KLUDGE: This fix is to compensate for ensureConencted also starting the session
         try
         {
            if (this.isSessionActive)
            {
               this.isSessionActive = false;

               send(new EndSessionRequest());
            }
         }
         catch (SliceStoreLayerException e)
         {
            logger.error("Could not end session before requesting delete vault", e);
         }

         // Send a Remove Store Request
         try
         {
            send(new RemoveStoreRequest(this.vaultIdentifier));
         }
         catch (final SliceStoreTransactionException e)
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }
         catch (final IllegalSourceNameException e)
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }
         catch (final SliceStoreExistsException e)
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }
         catch (final SliceStoreQuotaException e)
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }
         catch (final SliceStoreNotFoundException e)
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }

         try
         {
            this.connection.disconnect();
         }
         catch (final CommunicationInterruptedException e)
         {
            throw new SliceStoreIOException("Communication interrupted error", e);
         }
      }
      catch (final CommunicationIOException e)
      {
         throw new SliceStoreIOException("Connection error", e);
      }
   }

   public void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException
   {
      throw new RuntimeException("not yet implemented");
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void beginTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      final RemoteSliceStoreTransaction trans = (RemoteSliceStoreTransaction) transaction;

      try
      {
         send(new BeginTransactionRequest(transaction.getID()));
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      trans.isActive = true;
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void commitTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         long beginTime = System.currentTimeMillis();
         if (logger.isTraceEnabled())
            logger.trace(String.format("RemoteSliceStore.commit(tx=%d): started",
                  transaction.getID()));

         send(new CommitTransactionRequest(transaction.getID()));

         if (logger.isTraceEnabled())
            logger.trace(String.format("RemoteSliceStore.commit(tx=%d): successful - %dms",
                  transaction.getID(), System.currentTimeMillis() - beginTime));
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      ((RemoteSliceStoreTransaction) transaction).isActive = false;

      // TODO: Decide how this will be implemented over the network remotely
      // Notify all registered handlers of the writes that took place during
      // this commit
      // signalHandlers(trans);
   }

   /**
    * @see SliceStore
    */
   public SliceStoreTransaction createTransaction(final long transactionId)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      final SliceStoreTransaction transaction =
            new RemoteSliceStoreTransaction(transactionId, this);

      return transaction;
   }

   /**
    * @see SliceStore
    */
   public synchronized void endSession() throws SliceStoreIOException
   {
      // ensureConnected() is intentionally not called here

      try
      {
         // if not operational, we can't end the session
         if (this.isSessionActive)
         {
            this.isSessionActive = false;

            send(new EndSessionRequest());
         }
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      finally
      {
         try
         {
            this.connection.disconnect();
         }
         catch (final CommunicationException ex)
         {
            logger.error("Error shutting down client engine", ex);
            throw new SliceStoreIOException("Error occured closing the connection", ex);
         }
      }
      logger.debug("successfully terminated RemoteSliceStore session");
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public boolean exists(final SliceName name) throws SliceStoreIOException,
         SliceStoreNotFoundException
   {
      ensureConnected();

      final ExistsRequest existsRequest = new ExistsRequest(name);
      ExistsResponse existsResponse;
      try
      {
         existsResponse = (ExistsResponse) send(existsRequest);
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      return existsResponse.doesExist();
   }

   /**
    * Note that this method is only needed by actual slice servers used by the server, as the long
    * transactionId is sent in the protocol message and used to lookup the corresponding transaction
    * from the map.
    * 
    * @see SliceStore
    */
   public SliceStoreTransaction getTransaction(final long transactionId)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      return new RemoteSliceStoreTransaction(transactionId, this);
   }

   /**
    * Note that since the RemoteSliceStore does not keep track of transactions locally. Instead it
    * relies on the server to keep track of and manage transactions.
    * 
    * @see SliceStore
    */
   public boolean isActiveTransaction(final SliceStoreTransaction transaction)
   {
      return ((RemoteSliceStoreTransaction) transaction).isActive;
   }

   /**
    * A RemoteSliceStore can manage its own sessions, so it is always operational. This does not
    * necessarily mean that it is or can be connected.
    */
   public boolean isOperational()
   {
      return true;
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void listBegin() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         send(new ListBeginRequest());
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   /**
    * @see SliceStore
    */
   public void listBegin(final SliceName name)
   {
      throw new NotImplementedException(RemoteSliceStore.class.getName()
            + ".listBegin not implemented");
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public List<SliceInfo> listContinue() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         final ListContinueResponse response =
               (ListContinueResponse) send(new ListContinueRequest());
         return response.getSlices();
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public boolean listInProgress() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         final ListInProgressResponse response =
               (ListInProgressResponse) send(new ListInProgressRequest());
         
         return response.listInProgress();
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void listStop() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         send(new ListStopRequest());
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   @Override
   public List<DataSlice> readImpl(final List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      ensureConnected();

      final MultipleReadRequest readRequest = new MultipleReadRequest(names);
      MultipleReadResponse readResponse;
      try
      {
         long beginTime = System.currentTimeMillis();
         if (logger.isTraceEnabled())
            logger.trace("RemoteSliceStore.read(): started");

         readResponse = (MultipleReadResponse) send(readRequest);

         if (logger.isTraceEnabled())
         {
            long totalTime = System.currentTimeMillis() - beginTime;
            long bytesRead = 0;
            for (DataSlice slice : readResponse.getDataSlices())
            {
               byte[] data = slice.getData();
               if (data != null)
               {
                  bytesRead += data.length;
               }
            }
            logger.trace(String.format("RemoteSliceStore.read(): successful - %.3fMB, %dms",
                  bytesRead / (1024 * 1024.), totalTime));
         }
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      return Arrays.asList(readResponse.getDataSlices());
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void registerForNorNotification(final NotificationHandler handler)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      throw new NotImplementedException();
   }

   public boolean remove(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException
   {
      return remove(Arrays.asList(new SliceName[]{
         name
      }))[0];
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   @Override
   public boolean[] remove(final List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      ensureConnected();

      Response response = null;
      try
      {
         response = send(new MultipleRemoveRequest(names));
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      assert response instanceof MultipleRemoveResponse;
      return ((MultipleRemoveResponse) response).getRemoved();
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   public void rollbackTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreIOException, SliceStoreTransactionException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         send(new RollbackTransactionRequest(transaction.getID()));
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      ((RemoteSliceStoreTransaction) transaction).isActive = false;
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @throws SliceStoreStateException
    * @throws SecurityException
    *            If authentication failed or access to vault was denied by slice server.
    * 
    * @see SliceStore
    */
   public void startSession() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      try
      {
         // Call connection manager's ensureConnected() directly
         this.connection.ensureConnected();
      }
      catch (final CommunicationException ex)
      {
         throw new SliceStoreIOException(
               "communication connection exception encountered while connecting");
      }
      authenticate();

      // Bind vault to current session
      bindVault();

      // Send a begin session request to establish server-side SliceStore instance
      try
      {
         send(new BeginSessionRequest());
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }

      this.isSessionActive = true;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.slicestore.SliceStore#read(org.cleversafe.layer.grid.SliceName)
    */
   public DataSlice readImpl(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      return readImpl(Arrays.asList(new SliceName[]{
         name
      })).get(0);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.slicestore.SliceStore#write(org.cleversafe.layer.grid.DataSlice,
    *      boolean)
    */
   public void writeImpl(DataSlice data, boolean allowOverwriteNewer) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException
   {
      writeImpl(Arrays.asList(new DataSlice[]{
         data
      }), allowOverwriteNewer);
   }

   /**
    * @throws SliceStoreNotFoundException 
    * @see SliceStore
    */
   @Override
   public void writeImpl(final List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      ensureConnected();

      try
      {
         long beginTime = System.currentTimeMillis();
         if (logger.isTraceEnabled())
            logger.trace(String.format("RemoteSliceStore.write(tx=%d): started",
                  dataSlices.get(0).getTransactionId()));

         send(new MultipleWriteRequest(dataSlices, allowOverwriteNewer));

         if (logger.isTraceEnabled())
         {
            long totalTime = System.currentTimeMillis() - beginTime;
            long bytesWritten = 0;
            for (DataSlice slice : dataSlices)
            {
               bytesWritten += slice.getData().length;
            }
            logger.trace(String.format("RemoteSliceStore.write(tx=%d): successful - %.3fMB, %dms",
                  dataSlices.get(0).getTransactionId(), bytesWritten / (1024 * 1024.), totalTime));
         }
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   public List<SliceInfo> verifyIntegrity(List<SliceInfo> slices) throws VaultDescriptorException,
         SliceStoreIOException, IllegalSourceNameException, SliceStoreNotFoundException
   {
      ensureConnected();

      IntegrityVerificationRequest request = new IntegrityVerificationRequest(slices);

      IntegrityVerificationResponse response;
      try
      {
         response = (IntegrityVerificationResponse) send(request);
         return response.getCorruptedSlices();
      }
      catch (SliceStoreTransactionException e)
      {
         // This should never occur
         throw new RuntimeException("An unexpected exception was thrown", e);
      }
      catch (SliceStoreExistsException e)
      {
         // This should never occur
         throw new RuntimeException("An unexpected exception was thrown", e);
      }
      catch (SliceStoreQuotaException e)
      {
         // This should never occur
         throw new RuntimeException("An unexpected exception was thrown", e);
      }
      catch (final SliceStoreNotFoundException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
   }

   // TODO: Implement this method
   public VaultDescriptor getVaultDescriptor()
   {
      throw new NotImplementedException("RemoteSliceStore.getVaultDescriptor is not implemented");
   }

   /**
    * Performs authentication to the server for this SliceStore
    * 
    * @throws SliceStoreIOException
    */
   private void authenticate() throws SliceStoreIOException
   {
      assert this.credentials != null : "Credentials are null";
      assert this.vaultIdentifier != null : "Vault identifier is null";
      assert this.connection != null : "Connection is null";

      // TODO: We hard-code password authenticator currently; in the future
      // we will use some sort of AuthenticationFactory.
      final Authenticator authenticator = new PasswordAuthenticator(this.connection);

      logger.info("Authenticating to slice server " + getIdentification() + " using "
            + authenticator.getClass().getName());

      authenticator.setCredentials(this.credentials);
      try
      {
         if (!authenticator.login())
         {
            throw new RemoteAuthenticationException("Authentication to slice server "
                  + getIdentification() + " failed: no reason provided by server");
         }
         logger.info("Authenticated to slice server " + getIdentification());
      }
      catch (final AuthenticationException e)
      {
         logger.warn("Authentication error", e);
         throw new SliceStoreIOException("Authentication error: " + e.getMessage());
      }
   }

   /**
    * Binds Vault to this instance of RemoteSliceStore. Called when starting a session or deleting a
    * store.
    * @throws SliceStoreNotFoundException 
    * 
    * @throws CommunicationException
    */
   private void bindVault() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      final VaultBindRequest bindRequest = new VaultBindRequest(this.vaultIdentifier);

      VaultBindResponse bindResponse = null;

      try
      {
         bindResponse = (VaultBindResponse) send(bindRequest);
      }
      catch (final ClassCastException e)
      {
         logger.error("Slice server " + getIdentification()
               + "returned invalid vault session bind response.");
         throw e;
      }
      catch (final SliceStoreNotFoundException e)
      {
         logger.warn("Slice store not found for vault " + this.vaultIdentifier + " on " + this);
         throw e;
      }
      catch (final SliceStoreIOException e)
      {
         logger.warn("Error reading slice store for vault " + this.vaultIdentifier + " on " + this);
         throw e;
      }
      catch (final SliceStoreTransactionException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final IllegalSourceNameException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreExistsException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final SliceStoreQuotaException e)
      {
         throw new RuntimeException("Unexpected Exception type was received from server", e);
      }
      catch (final Exception e)
      {
         if (bindResponse != null)
         {
            if (bindResponse.getExceptionFlag())
            {
               logger.warn("Error binding vault to session: slice server " + getIdentification()
                     + " denied bind request");
               throw new SecurityException("Error binding vault to session: slice server "
                     + getIdentification() + " denied bind request");
            }
         }
         else
         {
            throw new RuntimeException("Unexpected Exception type was received from server", e);
         }
      }

   }

   /**
    * Sends a request in a way to check if an exception was set on the response
    * 
    * @param request
    * @return the received response
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    * @throws SliceNotFoundException
    * @throws SliceStoreQuotaException
    */
   private Response send(final Request request) throws SliceStoreIOException,
         SliceStoreTransactionException, IllegalSourceNameException, SliceStoreExistsException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      Response response;
      try
      {
         response = this.connection.exchange(request);
      }
      catch (final NotConnectedException e)
      {
         throw new SliceStoreIOException("Remote slice store not connected", e);
      }
      catch (final CommunicationIOException e)
      {
         throw new SliceStoreIOException("Communication IO error", e);
      }
      catch (final CommunicationInterruptedException e)
      {
         throw new SliceStoreIOException("Communication interrupted error", e);
      }
      catch (final CommunicationResponseException e)
      {
         throw new SliceStoreIOException("Communication response error", e);
      }
      catch (final CommunicationTransmissionException e)
      {
         throw new SliceStoreIOException("Communication transmission error", e);
      }

      if (response.getExceptionFlag())
      {
         if (response.getException() instanceof SliceStoreIOException)
         {
            throw (SliceStoreIOException) response.getException();
         }
         else if (response.getException() instanceof SliceStoreTransactionException)
         {
            throw (SliceStoreTransactionException) response.getException();
         }
         else if (response.getException() instanceof IllegalSourceNameException)
         {
            throw (IllegalSourceNameException) response.getException();
         }
         else if (response.getException() instanceof SliceStoreQuotaException)
         {
            throw (SliceStoreQuotaException) response.getException();
         }
         else if (response.getException() instanceof SliceStoreExistsException)
         {
            throw (SliceStoreExistsException) response.getException();
         }
         else if (response.getException() instanceof SliceStoreNotFoundException)
         {
            throw (SliceStoreNotFoundException) response.getException();
         }
         else if (response.getException() instanceof ServerRequestException)
         {
            // All server request exceptions get turned into generic IO exceptions
            throw new SliceStoreIOException(this + " error '"
                  + response.getException().getClass().getName() + "' when processing request: "
                  + request, response.getException());
         }
         else
         {
            // FIXME: Handle actual error here, when we get that worked out
            throw new RuntimeException(
                  "The server threw an unknown exception processing the request",
                  response.getException());
         }
      }

      return response;
   }

   /**
    * Attempts to re-establish a connection when it is lost
    * @throws SliceStoreNotFoundException 
    * 
    * @throws SliceStoreStateException
    * @throws CommunicationException
    */
   private synchronized void ensureConnected() throws SliceStoreIOException,
         SliceStoreNotFoundException
   {
      try
      {
         // check if the connection had been disconnected at the network layer and we just
         // reconnected
         if (this.connection.ensureConnected())
         {
            // Re-authenticate
            authenticate();

            // Re-bind to vault
            bindVault();
            try
            {
               send(new BeginSessionRequest());
               logger.debug("Session restarted");
               isSessionActive = true;
            }
            catch (final SliceStoreTransactionException e)
            {
               throw new RuntimeException("Unexpected Exception type was received from server", e);
            }
            catch (final IllegalSourceNameException e)
            {
               throw new RuntimeException("Unexpected Exception type was received from server", e);
            }
            catch (final SliceStoreExistsException e)
            {
               throw new RuntimeException("Unexpected Exception type was received from server", e);
            }
            catch (final SliceStoreQuotaException e)
            {
               throw new RuntimeException("Unexpected Exception type was received from server", e);
            }
         }
      }
      catch (final CommunicationException ex)
      {
         try
         {
            endSession();
         }
         catch (SliceStoreLayerException ssle)
         {
            // Stick to our original exception
         }

         throw new SliceStoreIOException("connection is unusable", ex);
      }
   }
}
