//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Jan 28, 2008
//---------------------

package org.cleversafe.layer.slicestore.memory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * In-memory {@link SliceStore} implementation. Used for testing.
 * 
 * @author Manish Motwani
 */
public class MemorySliceStore extends SliceStoreBase
{
   private final Map<Long, MemorySliceStoreTransaction> transactionMap =
         new ConcurrentHashMap<Long, MemorySliceStoreTransaction>();

   private final Map<SliceName, DataSlice> committedSlices =
         new ConcurrentHashMap<SliceName, DataSlice>();

   private boolean operational = false;

   @Override
   public VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException
   {
      throw new NotImplementedException();
   }

   @Override
   protected DataSlice readImpl(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      if (this.committedSlices.containsKey(name))
      {
         return this.committedSlices.get(name);
      }
      else
      {
         return new DataSlice(name);
      }
   }

   @Override
   protected void writeImpl(final DataSlice data, final boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException
   {
      ensureOperational();
      ensureActiveTransaction(data.getTransactionId());

      final MemorySliceStoreTransaction transaction =
            this.transactionMap.get(data.getTransactionId());

      this.addUncommittedSlice(data);
      transaction.write(data);
   }

   public void beginTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      ensureOperational();

      assert transaction != null;
      ensureNotActiveTransaction(transaction);
      ensureTransactionFound(transaction.getID());

      assert transaction instanceof MemorySliceStoreTransaction;
      final MemorySliceStoreTransaction memorySliceStoreTransaction =
            (MemorySliceStoreTransaction) transaction;

      memorySliceStoreTransaction.begin();
   }

   public void commitTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      ensureOperational();
      ensureActiveTransaction(transaction);

      final MemorySliceStoreTransaction memorySliceStoreTransaction =
            (MemorySliceStoreTransaction) transaction;

      this.transactionMap.remove(memorySliceStoreTransaction.getID());

      // Synchronize for making an atomic commit
      synchronized (memorySliceStoreTransaction.unwrittenSlices)
      {
         // If no exception is thrown, do the real commit - add them
         for (final DataSlice dataSlice : memorySliceStoreTransaction.unwrittenSlices)
         {
            this.committedSlices.put(dataSlice.getSliceName(), dataSlice);
         }

         this.cleanTrasactionSources(memorySliceStoreTransaction);
         memorySliceStoreTransaction.commit(); // resets transaction to its inactive state
      }
   }

   private void cleanTrasactionSources(MemorySliceStoreTransaction tx)
   {
      synchronized (tx.unwrittenSlices)
      {
         for (DataSlice dataSlice : tx.unwrittenSlices)
         {
            removeUncommittedSlices(dataSlice.getSliceName());
         }
      }
   }

   public SliceStoreTransaction createTransaction(final long transactionId)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      ensureOperational();

      if (this.transactionMap.containsKey(transactionId))
      {
         throw new SliceStoreTransactionException("A transaction with that ID already exists");
      }

      final MemorySliceStoreTransaction transaction =
            new MemorySliceStoreTransaction(transactionId, this);

      this.transactionMap.put(transactionId, transaction);
      return transaction;
   }

   public void endSession() throws SliceStoreIOException
   {
      ensureOperational();

      this.transactionMap.clear();
      this.operational = false;
   }

   public boolean exists(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      return this.committedSlices.containsKey(name);
   }

   public String getIdentification()
   {
      return "MemorySliceStore";
   }

   public SliceStoreTransaction getTransaction(final long transactionId)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      ensureOperational();

      final SliceStoreTransaction transaction = this.transactionMap.get(transactionId);
      if (transaction == null)
      {
         throw new SliceStoreTransactionException("Requested transaction " + transactionId
               + " does not exist");
      }
      return transaction;
   }

   public boolean isActiveTransaction(final SliceStoreTransaction transaction)
   {
      return isActiveTransaction(transaction.getID());
   }

   public boolean isOperational()
   {
      return this.operational;
   }

   public void listBegin() throws SliceStoreIOException
   {
   }

   public void listBegin(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
   }

   public List<SliceInfo> listContinue() throws SliceStoreIOException
   {
      List<SliceInfo> sliceList = new ArrayList<SliceInfo>();
      
      for(Map.Entry<SliceName, DataSlice> sliceEntry : committedSlices.entrySet())
      {
         SourceName sourceName = sliceEntry.getKey().getSourceName();
         long transactionId = sliceEntry.getValue().getTransactionId();
         sliceList.add(new SliceInfo(sourceName, transactionId));
      }
      
      return sliceList;
   }

   public boolean listInProgress() throws SliceStoreIOException
   {
      return false;
   }

   public void listStop() throws SliceStoreIOException
   {
   }

   public void registerForNorNotification(final NotificationHandler handler)
         throws SliceStoreIOException
   {
      throw new NotImplementedException();
   }

   public boolean remove(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      if (this.committedSlices.containsKey(name))
      {
         this.committedSlices.remove(name);
         return true;
      }
      else
      {
         return false;
      }
   }

   public void rollbackTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException
   {
      ensureOperational();
      ensureActiveTransaction(transaction);

      this.transactionMap.remove(transaction.getID());

      final MemorySliceStoreTransaction memorySliceStoreTransaction =
            (MemorySliceStoreTransaction) transaction;
      this.cleanTrasactionSources(memorySliceStoreTransaction);
      memorySliceStoreTransaction.rollback();
   }

   public void startSession() throws SliceStoreIOException
   {
      ensureNotOperational();

      this.operational = true;
   }

   /**
    * Ensures that the slice store is operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureOperational()
   {
      if (!isOperational())
      {
         throw new IllegalStateException("The slice store must be operational");
      }
   }

   /**
    * Ensures that the slice store is not operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureNotOperational()
   {
      if (isOperational())
      {
         throw new IllegalStateException("The slice store must not be operational");
      }
   }

   /**
    * Ensures that a transaction is active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureActiveTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (!isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction was not active");
      }
   }

   /**
    * Ensures that a transaction is active, otherwise throws a transaction exception
    * 
    * @param transactionId
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureActiveTransaction(final long transactionId)
         throws SliceStoreTransactionException
   {
      if (!isActiveTransaction(transactionId))
      {
         throw new SliceStoreTransactionException("The supplied transaction was not active");
      }
   }

   /**
    * Ensures that transaction exists but is not active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureNotActiveTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction should not be active");
      }
   }

   /**
    * Ensures that transaction is found, otherwise throws a transaction exception
    * 
    * @param transactionId
    * @throws SliceStoreTransactionException
    */
   protected void ensureTransactionFound(final long transactionId)
         throws SliceStoreTransactionException
   {
      if (!this.transactionMap.containsKey(transactionId))
      {
         throw new SliceStoreTransactionException("A transaction by that ID already exists");
      }
   }

   protected boolean isActiveTransaction(final long transactionId)
   {
      ensureOperational();

      boolean isActive = false;

      if (this.transactionMap.containsKey(transactionId))
      {
         isActive = this.transactionMap.get(transactionId).isActive();
      }
      return isActive;
   }

   public void createStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      if (this.isOperational())
      {
         throw new SliceStoreExistsException("Slice store exists and is operational");
      }
   }

   public void deleteStore() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      if (this.isOperational())
      {
         throw new IllegalStateException("Cannot create when session is active");
      }
   }

   public void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      throw new NotImplementedException();
   }
}
