//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Oct 20, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.security.InvalidParameterException;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * This class should replace a single BlockFileStore. It supports a single file but also makes use
 * of multiple files to represent a single store. It should save 2 important goals: - to overcome a
 * single file size limitation imposed by OS - to allow to grow slice store by adding multiple
 * directories on possibly different hard drives (cross mount) in order to increase storage.
 * 
 * Block multi file slice store is created with a list of directories that it should operate upon. A
 * selection, disk storage considerations and algorithm to choose directories resides in
 * BlockMultiFileSliceStoreManager
 * 
 * The old BlockFileStore should be deprecated and not used. It is preserved for now for performance
 * comparison reasons.
 */
public class BlockMultiFileSliceStore extends SliceStoreBase
{
   // This implementation uses transaction content in memory
   // It should be very efficient for a block device

   static Logger _logger = Logger.getLogger(BlockFileSliceStore.class);

   // Constants
   private static final int EXISTING_BLOCK_MAGIC = 0x13579BDF;
   private static final int REMOVED_BLOCK_MAGIC = 0xACE02468;

   public static final String BLOCK_FILE_BASE = "BlockDevice-";
   public static final String BLOCK_FILE_EXT = ".slice";

   public static final String PROPERTIES_FILE_NAME = "BlockDevice.properties";

   public static final String BLOCK_SIZE_PROPERTY_NAME = "block-size";

   public static final String MAX_SLICE_NUMBER_PROPERTY_NAME = "max-slice-number";

   public static final String SYNC_PROPERTY_NAME = "sync";

   // Configuration definable parameters
   List<FileStorageDefinition> fileStoreDefinitions = null;
   // One or more directories where data will be stored. Each directory represents a
   // a single file. The same directory may be listed more then once for large slice stores
   protected RuntimeStorage[] fileStores = null;

   protected long maxSliceNumber = SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED;
   protected int blockSize = -1; // maximum block size
   protected boolean blockSizeInitialized = false;
   protected boolean synchronous; // Use synchronous file access?

   // Directory in which access information is stored, such as directories, sizes
   protected int realBlockSize;
   protected byte[] padding = null;
   /*
    * Number of physical blocks written into a slice file It could only increase and it is not
    * really good :(
    */
   protected long eofBlockNumber = 0;

   protected boolean listInProgress = false;
   protected int listBlockCount = 4096; // Number of blocks per list call
   protected long listBlockNumber = -1; // Current block number for listing functions

   private final ConcurrentMap<Long, SliceStoreTransactionImpl> transactionMap =
         new ConcurrentHashMap<Long, SliceStoreTransactionImpl>();

   protected Properties blockDeviceProperties = null;

   /**
    * BlockFile implementation of SliceStoreTransaction
    * 
    * @see SliceStoreTransaction
    */
   public class SliceStoreTransactionImpl implements SliceStoreTransaction
   {
      private final long transactionID;

      Map<Long, DataSlice> content = null; // Blocks written for a transaction

      /**
       * Constructs BlockFileSliceStoreTransaction taking the transactionId
       * 
       * @param transactionID
       */
      public SliceStoreTransactionImpl(final long transactionID)
      {
         this.transactionID = transactionID;
      }

      /**
       * Begins transaction, creating a map to store data slices
       */
      private void begin()
      {
         this.content = Collections.synchronizedSortedMap(new TreeMap<Long, DataSlice>());
      }

      /**
       * Cleanup a transaction content. Nothing is written to the storage
       * 
       * @throws Exception
       */
      private synchronized void rollback()
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Rolling back transaction " + getID());
         this.content = null;
      }

      /**
       * Write content to file
       * 
       * @throws IOException
       */
      @SuppressWarnings("unchecked")
      private synchronized void commit() throws SliceStoreIOException
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Commiting transaction " + getID());
         try
         {
            final Set<Long> blockNumbers = this.content.keySet();
            Tuple2<Long, DataSlice>[] sources = new Tuple2[blockNumbers.size()];

            int i = 0;
            for (final long blockNumber : blockNumbers)
            {
               sources[i++] =
                     new Tuple2<Long, DataSlice>(blockNumber, this.content.get(blockNumber));
            }
            doRealWrite(getID(), sources);
         }
         finally
         {
            this.content = null;
         }
      }

      /**
       * Returns true if transaction has been begun but not committed or rolled back
       * 
       * @return active status of transaction
       */
      public synchronized boolean isActive()
      {
         return this.content != null;
      }

      /**
       * @see SliceStoreTransaction
       */
      public long getID()
      {
         return this.transactionID;
      }

      /**
       * @see SliceStoreTransaction
       */
      public SliceStore getSliceStore()
      {
         return BlockMultiFileSliceStore.this;
      }

      /**
       * Creates a block image in memory
       * 
       * @param blockNumber
       * @param data
       * @throws SliceStoreTransactionException
       * @throws IOException
       */
      protected void saveBlock(final long blockNumber, DataSlice dataSlice)
            throws SliceStoreTransactionException
      {
         assert this.transactionID == dataSlice.getTransactionId() : "An attempt to executr wtite witin a different transaction";
         if (dataSlice.getData().length > BlockMultiFileSliceStore.this.blockSize)
         {
            throw new IllegalArgumentException("Block storage doesn't support block larger then "
                  + BlockMultiFileSliceStore.this.blockSize);
         }
         synchronized (this)
         {
            // rollback could be issued asynchronously and could sneak before write was completed, thus 
            // causing put to fail
            if (this.content == null)
            {
               throw new SliceStoreTransactionException("Transaction " + this.transactionID
                     + " was likely rollbacked");
            }
            this.content.put(blockNumber, dataSlice);
         }
      }
   }

   public static class FileStorageDefinition
   {
      private String dataPath;
      private long blockCapacity;

      public FileStorageDefinition()
      {
         // For framework, will be initialized through setters
      }

      /**
       * Initialization all fields
       * 
       * @param dataPath
       * @param blockCapacity
       */
      public FileStorageDefinition(String dataPath, long blockCapacity)
      {
         super();
         this.dataPath = dataPath;
         this.blockCapacity = blockCapacity;
      }

      /**
       * Copy
       * 
       * @param other
       */
      public FileStorageDefinition(FileStorageDefinition other)
      {
         super();
         this.dataPath = other.dataPath;
         this.blockCapacity = other.blockCapacity;
      }

      /**
       * @return the dataPath
       */
      public String getDataPath()
      {
         return this.dataPath;
      }

      /**
       * @param dataPath
       *           the dataPath to set
       */
      public void setDataPath(String dataPath)
      {
         this.dataPath = dataPath;
      }

      /**
       * @return the blockCapacity
       */
      public long getBlockCapacity()
      {
         return this.blockCapacity;
      }

      /**
       * @param blockCapacity
       *           the blockCapacity to set
       */
      public void setBlockCapacity(long blockCapacity)
      {
         this.blockCapacity = blockCapacity;
      }
   }

   private static class RuntimeStorage extends FileStorageDefinition
   {
      private long startBlock;
      private RandomAccessFile randomAccessor;

      public RuntimeStorage(long startBlock, FileStorageDefinition fileDef)
      {
         super(fileDef);
         this.startBlock = startBlock;
         this.randomAccessor = null;
      }

      /**
       * @return the startBlock
       */
      public long getStartBlock()
      {
         return this.startBlock;
      }

      /**
       * @param startBlock
       *           the startBlock to set
       */
      public void setStartBlock(long startBlock)
      {
         this.startBlock = startBlock;
      }

      /**
       * @return the randomAccessor
       */
      public RandomAccessFile getRandomAccessor()
      {
         return this.randomAccessor;
      }

      /**
       * @param randomAccessor
       *           the randomAccessor to set
       */
      public void setRandomAccessor(RandomAccessFile randomAccessor)
      {
         this.randomAccessor = randomAccessor;
      }

   }

   /**
    * Default constructor used by configuration framework
    * 
    */
   public BlockMultiFileSliceStore()
   {
      this.blockDeviceProperties = new Properties();
   }

   /**
    * Constructs a slice store object with a single {@link FileStorageDefinition}.
    * 
    * @param fileStore
    * @throws IOException
    */
   public BlockMultiFileSliceStore(final FileStorageDefinition fileStore) throws IOException
   {
      this();
      setFileStore(fileStore);
      initialize();
   }

   /**
    * Constructs a slice store object given parameters. The blockSize and synchronous parameters are
    * loaded from a properties file that was created during createStore().
    * 
    * @param fileStores
    * @throws IOException
    */
   public BlockMultiFileSliceStore(final List<FileStorageDefinition> fileStores) throws IOException
   {
      this();
      setAllFileStores(fileStores);
      initialize();
   }

   /**
    * Sets all paths at once
    * 
    * @param dataDirectories
    */
   protected void setAllFileStores(List<FileStorageDefinition> fileStores)
   {
      this.fileStoreDefinitions = fileStores;
   }

   /**
    * This method should be only called before initialize In order to enforce, it is stored in
    * temporary variables and will be officially available after initialize() is invoked. This
    * approach guarantees that later calls will have no effect
    * 
    * @param path
    */
   public void setFileStore(final FileStorageDefinition path)
   {
      if (this.fileStoreDefinitions == null)
      {
         this.fileStoreDefinitions = new ArrayList<FileStorageDefinition>();
      }
      this.fileStoreDefinitions.add(path);
   }

   /**
    * Plain getter
    * 
    */
   public List<String> getAllPaths()
   {
      if (this.fileStoreDefinitions == null)
      {
         throw new RuntimeException("Can't call getAllPaths() before object is filly initialized");
      }
      List<String> paths = new ArrayList<String>(this.fileStoreDefinitions.size());
      for (FileStorageDefinition fileStoredef : this.fileStoreDefinitions)
      {
         paths.add(fileStoredef.getDataPath());
      }
      return paths;
   }

   /**
    * This method should be only called before initialize In order to enforce, it is stored in
    * temporary variables and will be officially available after initialize() is invoked. This
    * approach guarantees that later calls will have no effect
    * 
    * @param blockSize
    * 
    */
   protected void setBlockSize(final int blockSize)
   {
      this.blockSize = blockSize;
      // magic + transaction + length + data
      this.realBlockSize = getBlockSizeOverhead() + this.blockSize;

      this.padding = new byte[blockSize]; // used to pad a real block
      Arrays.fill(this.padding, (byte) 0xFF);

      this.blockSizeInitialized = true;
   }

   /**
    * CValculates overhead for each block stored
    * 
    * @return
    */
   static public int getBlockSizeOverhead()
   {
      return (Integer.SIZE + Integer.SIZE + Long.SIZE) / 8;
   }

   /**
    * block size getter
    * 
    * @return maximum block size
    */
   public long getBlockSize()
   {
      return this.blockSize;
   }

   /**
    * This method should only be called before initialize
    * 
    * @param synchronous
    */
   public void setSynchronous(boolean synchronous)
   {
      this.synchronous = synchronous;
   }

   /**
    * Are writes performed synchronously?
    * 
    * @return True if writes are performed synchronously
    */
   public boolean getSynchronous()
   {
      return this.synchronous;
   }

   /**
    * @return
    */
   public long getMaxSliceNumber()
   {
      return this.maxSliceNumber;
   }

   /**
    * Used by manager to restore properties
    * 
    * @param maxSliceNumber
    */
   void setMaxSliceNumber(long maxSliceNumber)
   {
      this.maxSliceNumber = maxSliceNumber;
   }

   /**
    * Configuration framework code
    * 
    * @throws IOException
    * 
    */
   public void initialize() throws IOException
   {
      if (this.fileStoreDefinitions == null || this.fileStoreDefinitions.size() == 0)
      {
         throw new InitializationException(
               "BlockFileStore is not properly initialized: at least one data path must be set");
      }

      this.fileStores = new RuntimeStorage[this.fileStoreDefinitions.size()];
      int dataInd = 0;
      long startIndex = 0;
      for (FileStorageDefinition fileStoreDefinition : this.fileStoreDefinitions)
      {
         RuntimeStorage fs = new RuntimeStorage(startIndex, fileStoreDefinition);
         startIndex += fileStoreDefinition.getBlockCapacity();

         this.fileStores[dataInd++] = fs;
      }
      _logger.info("Completed initializing Block file store (sync=" + this.synchronous + ") in "
            + getAllPaths());
   }

   /**
    * @see SliceStore
    */
   public String getIdentification()
   {
      return "BlockMultiFile:[" + getDirectoryList() + "] " + this.blockSize + ","
            + this.eofBlockNumber;
   }

   private String getDirectoryList()
   {
      StringBuffer paths = new StringBuffer();
      paths.append("{");
      boolean first = true;
      for (RuntimeStorage fileStore : this.fileStores)
      {
         if (!first)
         {
            paths.append(",");
         }
         paths.append(fileStore.getDataPath());
         first = false;
      }
      paths.append("}");
      return paths.toString();
   }

   /**
    * @see SliceStore
    */
   public boolean isOperational()
   {
      for (RuntimeStorage fileStore : this.fileStores)
      {
         if (fileStore.getRandomAccessor() == null)
         {
            return false;
         }
      }

      return true;
   }

   /**
    * Creates a block store for a single vault
    * 
    * @see SliceStore
    */
   public void createStore(
         final String vaultType,
         final long maxSliceSize,
         final long sliceStoreSize,
         final VaultACL accessControlList,
         final byte[] vaultDescriptorBytes,
         final Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      // TODO: check whether vault type is supported by this slice store
      // TODO: check long value is within int bounds
      setBlockSize((int) maxSliceSize);

      // Calculate the maximum number of slices given the Store Size
      if (sliceStoreSize == SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED)
      {
         throw new InvalidParameterException(
               "Maximum block number for block storage must be specified");
      }
      this.maxSliceNumber = (sliceStoreSize / maxSliceSize) - 1;

      _logger.info("Creating a new block store in {" + getDirectoryList() + "}");

      // Create a storage file for each directory
      try
      {
         for (int i = 0; i < this.fileStores.length; i++)
         {
            createSingleStorageFile(i, this.fileStores[i].getBlockCapacity() * this.realBlockSize);
         }
         for (String key : options.keySet())
         {
            this.blockDeviceProperties.setProperty(key, options.get(key));
         }
         saveProperties();
      }
      catch (SliceStoreExistsException ex1)
      {
         throw ex1;
      }
      catch (final Exception e)
      {
         throw new SliceStoreIOException("Can't create a slice store", e);
      }
      // Create ACL and XML
      createSupportFiles(accessControlList, vaultDescriptorBytes);
   }

   void createSupportFiles(final VaultACL accessControlList, final byte[] vaultDescriptorBytes)
         throws SliceStoreExistsException, SliceStoreIOException
   {
      // Serialize ACL and write it
      writeVaultACL(this.fileStores[0].getDataPath() + File.separator + ACL_FILE_NAME,
            accessControlList);

      // Write the vault descriptor
      writeVaultDescriptor(this.fileStores[0].getDataPath() + File.separator
            + accessControlList.getVaultIdentifier().toString() + VAULT_DESCRIPTOR_EXTENSION,
            vaultDescriptorBytes);

      _logger.info("Created block file store (block=" + this.blockSize + ") in "
            + getDirectoryList());
   }

   /**
    * Creates a single storage file used by the manager
    * 
    * @param storeIndex
    * @throws SliceStoreIOException
    */
   void createSingleStorageFile(int storeIndex, long size) throws SliceStoreExistsException,
         SliceStoreIOException
   {
      String fullStorageFileName =
            this.fileStores[storeIndex].getDataPath() + File.separator + BLOCK_FILE_BASE
                  + storeIndex + BLOCK_FILE_EXT;
      File sliceFile = new File(fullStorageFileName);
      if (sliceFile.exists() == true)
      {
         throw new SliceStoreExistsException("The slice store file " + sliceFile.getAbsolutePath()
               + " already exists");
      }
      sliceFile.getParentFile().mkdirs();
      try
      {
         RandomAccessFile f = new RandomAccessFile(fullStorageFileName, "rw");
         // write the last byte, to ensure file is fully created, may cause significant delay
         // if the full size should be initialized, just write a single 0 byte at the end
         //f.seek(size-1);
         //f.write(new byte[]{0x55});
         f.setLength(size); // This is much faster then to write at least on Windows
         f.close();

         this.fileStores[storeIndex].setRandomAccessor(f);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Can't create a slice store", e);
      }
      this.fileStores[storeIndex].setRandomAccessor(null); // SHOULD NOT BE OPERATIONAL
   }

   /**
    * Returns the vault descriptor that was saved during this store's creation.
    * 
    * @return
    * @throws SliceStoreIOException
    * @throws SliceStoreNotFoundException
    */
   public VaultDescriptor getVaultDescriptor(PrivateKey privateKey, Credentials credentials)
         throws SliceStoreIOException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException
   {
      try
      {
         return this.getVaultDescriptor(this.fileStores[0].getDataPath(), privateKey, credentials);
      }
      catch (VaultACLException e)
      {
         throw new VaultSecurityException("Error accessing vault ACL", e);
      }
   }

   /**
    * This method is used to store persistent information about a BlockFileSlice during the creation
    * of the SliceStore, so that subsequent loadings don't need to worry about providing an invalid
    * parameter that could lead to data corruption.
    * 
    * Currently only the blockSize value is stored, however the maxBlockNumber when it is utilized
    * should also be stored here.
    * 
    * @throws SliceStoreIOException
    */
   private void saveProperties() throws SliceStoreIOException
   {
      this.blockDeviceProperties = convertToProperties();

      final File outFile = new File(constructPropertyFileName());
      saveProperiesToFile(outFile, this.blockDeviceProperties);
   }

   /**
    * Created required properties
    * 
    * @return
    */
   Properties convertToProperties()
   {
      final Properties props = new Properties();
      props.setProperty(MAX_SLICE_NUMBER_PROPERTY_NAME, Long.toString(this.maxSliceNumber));
      props.setProperty(BLOCK_SIZE_PROPERTY_NAME, Integer.toString(this.blockSize));
      props.setProperty(SYNC_PROPERTY_NAME, Boolean.toString(this.synchronous));

      return props;
   }

   /**
    * Used for repair
    * 
    * @param outFile
    */
   void saveProperiesToFile(File outFile, Properties props) throws SliceStoreIOException
   {
      FileOutputStream fileOutputStream = null;
      try
      {
         fileOutputStream = new FileOutputStream(outFile);
         props.store(fileOutputStream, null);
      }
      catch (final IOException ex)
      {
         throw new SliceStoreIOException("Unable to save the BlockDevice properties", ex);
      }
      finally
      {
         try
         {
            if (fileOutputStream != null)
            {
               fileOutputStream.close();
            }
         }
         catch (IOException ignore)
         {
            _logger.warn("Unable to close file " + fileOutputStream);
         }
      }
   }

   /**
    * This method is used to load persistent information about a BlockFileSlice during the
    * construction of the SliceStore, it uses information saved about the SliceStore during the
    * creation of it.
    * 
    * Currently only the blockSize value is loaded, however the maxBlockNumber when it is utilized
    * should also be loaded here.
    * 
    * @throws SliceStoreIOException
    */
   void loadProperties() throws IOException
   {
      final File inFile = new File(constructPropertyFileName());
      FileInputStream fileInputStream = new FileInputStream(inFile);
      this.blockDeviceProperties.load(fileInputStream);

      try
      {
         if (fileInputStream != null)
         {
            fileInputStream.close();
         }
      }
      catch (IOException ignore)
      {
         _logger.warn("Unable to close file " + fileInputStream);
      }

      if (this.blockDeviceProperties.containsKey(MAX_SLICE_NUMBER_PROPERTY_NAME))
      {
         String parsedMaxSliceNumber =
               this.blockDeviceProperties.getProperty(MAX_SLICE_NUMBER_PROPERTY_NAME);
         this.maxSliceNumber = Long.parseLong(parsedMaxSliceNumber);
      }

      if (this.blockDeviceProperties.containsKey(BLOCK_SIZE_PROPERTY_NAME))
      {
         String parsedBlockSize = this.blockDeviceProperties.getProperty(BLOCK_SIZE_PROPERTY_NAME);
         int parsedBlockSizeValue = Integer.parseInt(parsedBlockSize);
         setBlockSize(parsedBlockSizeValue);
      }

      if (this.blockDeviceProperties.containsKey(SYNC_PROPERTY_NAME))
      {
         String parsedSynchronous = this.blockDeviceProperties.getProperty(SYNC_PROPERTY_NAME);
         boolean synchronous = Boolean.parseBoolean(parsedSynchronous);
         setSynchronous(synchronous);
      }
   }

   /**
    * @see SliceStore
    */
   public void deleteStore() throws SliceStoreIOException
   {
      ensureNotOperational();

      // TODO: Make sure that a store is not opened by someone else?
      try
      {
         // First try to delete all files. If traverse twice nothing would be left after first try
         for (RuntimeStorage fs : this.fileStores)
         {
            File storageDir = new File(fs.getDataPath());
            for (File file : storageDir.listFiles())
            {
               _logger.debug("Deleted  Block file store " + file.getAbsolutePath());
               boolean success = file.delete();
               _logger.info("Deleting file " + file + (success ? " success" : " failed"));
            }
         }
         // Now delete directory itself
         for (RuntimeStorage fs : this.fileStores)
         {
            _logger.info("Deleted  Block file store directory" + fs.getDataPath());
            File storageDir = new File(fs.getDataPath());
            storageDir.delete();
         }
      }
      catch (final Exception ex)
      {
         _logger.error("Failed to delete store", ex);
         throw new SliceStoreIOException("Failed to delete store", ex);
      }
   }

   public void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException
   {
      throw new RuntimeException("not yet implemented");
   }

   /**
    * @see SliceStore
    */
   public void startSession() throws SliceStoreIOException
   {
      ensureNotOperational();
      try
      {
         loadProperties();

         if (this.blockSizeInitialized == false)
         {
            throw new InitializationException(
                  "BlockFileStore is not properly initialized: block size must be set");
         }

         this.eofBlockNumber = 0;
         for (int dataInd = 0; dataInd < this.fileStores.length; dataInd++)
         {
            RuntimeStorage fs = this.fileStores[dataInd];
            RandomAccessFile randomAccessor =
                  new RandomAccessFile(constructStoreFileName(dataInd), this.synchronous
                        ? "rwd"
                        : "rw");
            randomAccessor.setLength(fs.getBlockCapacity() * this.realBlockSize);
            fs.setRandomAccessor(randomAccessor);
            // Calculate the last block available. The last file with blocks available
            // would define the last block really available
            if (randomAccessor.length() > 0)
            {
               this.eofBlockNumber =
                     fs.getStartBlock() + randomAccessor.length() / this.realBlockSize;
            }
         }
         // POST CONDITION
         ensureOperational();
      }
      catch (final FileNotFoundException e)
      {
         throw new SliceStoreIOException("Data file not found", e);
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException(e);
      }
   }

   /**
    * @see SliceStore
    */
   public void endSession() throws SliceStoreIOException
   {
      ensureOperational();

      // TODO: check uncommitted transactions. What to do with them?
      try
      {
         // Rollback any open transactions
         for (Entry<Long, SliceStoreTransactionImpl> entry : this.transactionMap.entrySet())
         {
            SliceStoreTransaction trans = entry.getValue();
            if (isActiveTransaction(trans))
            {
               try
               {
                  rollbackTransaction(trans);
               }
               catch (SliceStoreTransactionException e)
               {
                  _logger.warn("Unable to rollback transaction id=" + trans.getID(), e);
               }
            }
         }
         for (int dataInd = 0; dataInd < this.fileStores.length; dataInd++)
         {
            this.fileStores[dataInd].getRandomAccessor().close();
         }
      }
      catch (final IOException e)
      {
         _logger.error("Failed to end session", e);
         throw new SliceStoreIOException("Failed to end session", e);
      }
      finally
      {
         for (int dataInd = 0; dataInd < this.fileStores.length; dataInd++)
         {
            this.fileStores[dataInd].setRandomAccessor(null);
         }
         this.transactionMap.clear();
      }
      _logger.info("Session ended for slice store " + getAllPaths());

   }

   /**
    * @see SliceStore
    */
   public SliceStoreTransaction createTransaction(final long transactionId)
         throws SliceStoreTransactionException
   {
      ensureOperational();

      final SliceStoreTransactionImpl newTx = new SliceStoreTransactionImpl(transactionId);

      // The semantics of putIfAbsent are strange. It returns the existing value if there is one, or
      // null if newTx is inserted
      SliceStoreTransactionImpl tx = this.transactionMap.putIfAbsent(transactionId, newTx);
      return tx != null ? tx : newTx;
   }

   /**
    * @see SliceStore
    */
   public void beginTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      ensureOperational();
      assert transaction != null;
      ensureNotActiveTransaction(transaction);
      ensureTransactionFound(transaction.getID());

      assert transaction instanceof SliceStoreTransactionImpl;
      final SliceStoreTransactionImpl blockFileTransaction =
            (SliceStoreTransactionImpl) transaction;

      blockFileTransaction.begin();

      if (_logger.isTraceEnabled())
         _logger.trace("Begin transaction " + blockFileTransaction.getID());
   }

   /**
    * @see SliceStore
    */
   public void commitTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreIOException, SliceStoreTransactionException
   {
      assert transaction != null;
      ensureOperational();
      ensureActiveTransaction(transaction);

      long beginTime = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace(String.format("BlockMultiFileSliceStore.commit(tx=%d): started",
               transaction.getID()));

      this.transactionMap.remove(transaction.getID());

      // Note: This used to be synchronized, but it does not seem necessary (famous last words ;)
      final SliceStoreTransactionImpl blockFileTransaction =
            (SliceStoreTransactionImpl) transaction;

      cleanTrasactionSources(blockFileTransaction);
      blockFileTransaction.commit();

      if (_logger.isTraceEnabled())
         _logger.trace(String.format("BlockMultiFileSliceStore.commit(tx=%d): successful - %dms",
               transaction.getID(), System.currentTimeMillis() - beginTime));
   }

   /**
    * @see SliceStore
    */
   public void rollbackTransaction(final SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      ensureOperational();
      ensureActiveTransaction(transaction);

      this.transactionMap.remove(transaction.getID());

      // Note: This used to be synchronized, but it does not seem necessary (famous last words ;)
      final SliceStoreTransactionImpl blockFileTransaction =
            (SliceStoreTransactionImpl) transaction;
      cleanTrasactionSources(blockFileTransaction);
      blockFileTransaction.rollback();
   }

   //TODO: Create a more consistent way of guaranteeing that slice name is a part of a single transaction
   private void cleanTrasactionSources(SliceStoreTransactionImpl blockFileTransaction)
   {
      for (Map.Entry<Long, DataSlice> entry : blockFileTransaction.content.entrySet())
      {
         removeUncommittedSlices(entry.getValue().getSliceName());
      }
   }

   /**
    * @see SliceStore
    */
   public boolean isActiveTransaction(final SliceStoreTransaction transaction)
   {
      ensureOperational();

      assert transaction != null;

      boolean isActive = false;

      if (this.transactionMap.containsKey(transaction.getID()))
      {
         final SliceStoreTransactionImpl blockFileTransaction =
               (SliceStoreTransactionImpl) transaction;
         isActive = blockFileTransaction.isActive();
      }
      return isActive;
   }

   /**
    * @see SliceStore
    */
   public SliceStoreTransaction getTransaction(final long transactionId)
         throws SliceStoreTransactionException
   {
      ensureOperational();

      final SliceStoreTransaction transaction = this.transactionMap.get(transactionId);
      if (transaction == null)
      {
         throw new SliceStoreTransactionException("Requested transaction " + transactionId
               + " does not exist");
      }
      return transaction;
   }

   /**
    * @see SliceStore
    */
   public boolean exists(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      final long blockNumber = convertNameIntoBlockNumber(name);

      // Don't need synchronization here, even if it increases during this
      // operation it is still not valid
      int accessFileNumber = getFileByBlockNumber(blockNumber);
      if (accessFileNumber == -1)
      {
         return false;
      }

      try
      {
         RandomAccessFile accessFile = this.fileStores[accessFileNumber].getRandomAccessor();
         final long position =
               (blockNumber - this.fileStores[accessFileNumber].getStartBlock())
                     * this.realBlockSize;
         byte[] magicBytes = new byte[Integer.SIZE];
         synchronized (accessFile)
         {
            accessFile.seek(position);
            accessFile.read(magicBytes);
         }
         final ByteArrayInputStream block = new ByteArrayInputStream(magicBytes);
         final DataInputStream input = new DataInputStream(block);

         final int magic = input.readInt();
         return (magic == EXISTING_BLOCK_MAGIC);
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Error reading a block# " + blockNumber, ex);
      }
   }

   /**
    * @see SliceStore
    */
   public boolean remove(final SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      try
      {
         if (exists(name) == false)
         {
            return false;
         }
         else
         {
            final long blockNumber = convertNameIntoBlockNumber(name);
            int accessFileNumber = getFileByBlockNumber(blockNumber);

            RandomAccessFile accessFile = this.fileStores[accessFileNumber].getRandomAccessor();
            final long position =
                  (blockNumber - this.fileStores[accessFileNumber].getStartBlock())
                        * this.realBlockSize;

            final ByteArrayOutputStream block = new ByteArrayOutputStream(this.realBlockSize);
            final DataOutputStream output = new DataOutputStream(block);

            output.writeInt(REMOVED_BLOCK_MAGIC);
            synchronized (accessFile)
            {
               accessFile.seek(position);
               accessFile.write(block.toByteArray());
            }

            if (_logger.isTraceEnabled())
               _logger.trace("Block # transaction " + blockNumber + " deleted");

            return true;
         }
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException("Failed to write transaction data", e);
      }
   }

   /**
    * @see SliceStore
    */
   public void registerForNorNotification(final NotificationHandler handler)

   {
      ensureOperational();
      throw new NotImplementedException(this.getClass().getName() + ".registerForNotification()");
   }

   /**
    * Setter for the list block count (number of blocks to return per list call).
    * 
    * @param listBlockCount
    */
   public void setListBlockCount(int listBlockCount)
   {
      _logger.trace("Setting listBlockCount to " + listBlockCount);
      this.listBlockCount = listBlockCount;
   }

   /**
    * Getter for the list block count (number of blocks to return per list call).
    * 
    * @param listBlockCount
    */
   public int getListBlockCount()
   {
      return this.listBlockCount;
   }

   /**
    * @see SliceStore
    */
   public void listBegin() throws SliceStoreIOException
   {
      ensureOperational();

      this.listBlockNumber = 0;
      this.listInProgress = true;
   }

   /**
    * @see SliceStore
    */
   public void listBegin(final SliceName name)
   {
      ensureOperational();
      throw new NotImplementedException(this.getClass().getName() + ".listBegin()");
   }

   /**
    * @see SliceStore
    */
   public List<SliceInfo> listContinue() throws SliceStoreIOException
   {
      ensureOperational();

      long endBlockNumber = this.listBlockNumber + this.listBlockCount;

      if (endBlockNumber >= this.eofBlockNumber)
      {
         endBlockNumber = this.eofBlockNumber;
         this.listInProgress = false;
      }

      List<SliceInfo> sliceInfoList = new ArrayList<SliceInfo>();

      for (; this.listBlockNumber < endBlockNumber; this.listBlockNumber++)
      {
         SourceName sourceName = new SourceName(String.valueOf(this.listBlockNumber));
         int accessFileNumber = getFileByBlockNumber(this.listBlockNumber);

         assert (accessFileNumber != -1);
         RandomAccessFile accessFile = this.fileStores[accessFileNumber].getRandomAccessor();
         final long position =
               (this.listBlockNumber - this.fileStores[accessFileNumber].getStartBlock())
                     * this.realBlockSize;

         try
         {
            synchronized (accessFile)
            {
               accessFile.seek(position);
               final int magic = accessFile.readInt();
               if (magic == EXISTING_BLOCK_MAGIC)
               {
                  final long transactionId = accessFile.readLong();
                  sliceInfoList.add(new SliceInfo(sourceName, transactionId));
               }
            }
         }
         catch (IOException ex)
         {
            throw new SliceStoreIOException("Error reading from file at position " + position, ex);
         }
      }

      return sliceInfoList;
   }

   /**
    * @see SliceStore
    */
   public boolean listInProgress()
   {
      ensureOperational();

      return this.listInProgress;
   }

   /**
    * @see SliceStore
    */
   public void listStop()
   {
      ensureOperational();

      this.listInProgress = false;
   }

   /**
    * Interface implementation. Depending on transaction either stores in transaction or writes into
    * file
    * 
    * @throws SliceStoreNotFoundException
    * 
    * @see SliceStore
    */
   @Override
   public void writeImpl(DataSlice dataSlice, boolean allowOverwriteNewer)
         throws SliceStoreTransactionException, SliceStoreIOException, IllegalSourceNameException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      ensureOperational();

      SliceStoreTransaction transaction = getTransaction(dataSlice.getTransactionId());
      ensureActiveTransaction(transaction);

      addUncommittedSlice(dataSlice);

      final long blockNumber = convertNameIntoBlockNumber(dataSlice.getSliceName());
      if (dataSlice.getData().length > this.blockSize)
      {
         throw new IllegalArgumentException("Data must be no more than " + this.blockSize
               + " bytes");
      }

      if ((blockNumber > this.maxSliceNumber)
            && (this.maxSliceNumber != SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED))
      {
         throw new SliceStoreQuotaException(
               "Attempted to write a block number greater than the specified maximum");
      }

      if (!allowOverwriteNewer)
      {
         checkOverwriteNewer(dataSlice);
      }

      final SliceStoreTransactionImpl blockFileTransaction =
            (SliceStoreTransactionImpl) transaction;
      blockFileTransaction.saveBlock(blockNumber, dataSlice);
   }

   /**
    * @throws SliceStoreNotFoundException
    * @see SliceStore
    */
   @Override
   public void writeImpl(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreTransactionException, SliceStoreIOException, IllegalSourceNameException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      ensureOperational();

      long beginTime = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace(String.format("BlockMultiFileSliceStore.write(tx=%d): started",
               dataSlices.get(0).getTransactionId()));

      for (int i = 0; i < dataSlices.size(); i++)
      {
         SliceStoreTransaction transaction = getTransaction(dataSlices.get(i).getTransactionId());
         ensureActiveTransaction(transaction);
         final SliceStoreTransactionImpl blockFileTransaction =
               (SliceStoreTransactionImpl) transaction;

         long blockNumber = convertNameIntoBlockNumber(dataSlices.get(i).getSliceName());

         DataSlice dataSlice = dataSlices.get(i);

         addUncommittedSlice(dataSlice);

         if (dataSlice.getData().length > this.blockSize)
         {
            throw new IllegalArgumentException("Data must be no more than " + this.blockSize
                  + " bytes");
         }

         if ((blockNumber > this.maxSliceNumber)
               && (this.maxSliceNumber != SliceStoreBase.SLICE_STORE_SIZE_UNLIMITED))
         {
            throw new SliceStoreQuotaException(
                  "Attempted to write a block number greater than the specified maximum");
         }

         if (!allowOverwriteNewer)
         {
            checkOverwriteNewer(dataSlice);
         }

         blockFileTransaction.saveBlock(blockNumber, dataSlice);
      }

      long totalTime = System.currentTimeMillis() - beginTime;
      if (_logger.isTraceEnabled())
      {
         long bytesWritten = 0;
         for (DataSlice slice : dataSlices)
         {
            bytesWritten += slice.getData().length;
         }
         _logger.trace(String.format(
               "BlockMultiFileSliceStore.write(tx=%d): successful - %.3fMB, %dms",
               dataSlices.get(0).getTransactionId(), bytesWritten / (1024 * 1024.), totalTime));
      }
   }

   /**
    * Performs write of data slices to disk
    * 
    * @param transactionId
    * @param sources
    * @throws SliceStoreIOException
    */
   protected void doRealWrite(long transactionId, Tuple2<Long, DataSlice>[] sources)
         throws SliceStoreIOException
   {
      int startIndex = 0;
      // A single write() should go into a single access file

      while (startIndex < sources.length) // Do we have anything not processed
      {
         // Index of a file to be read for the next operation
         int accessFileIndex = getFileByBlockNumber(sources[startIndex].getFirst());
         assert accessFileIndex != -1;
         // Find sequential block numbers within the same storage file
         int index = startIndex;
         while (index < sources.length - 1
               && (sources[index + 1].getFirst() == sources[index].getFirst() + 1))
         {
            if (accessFileIndex != getFileByBlockNumber(sources[index + 1].getFirst()))
            {
               // changed file storage, need to break
               break;
            }
            index++;
         }
         multipleWriteImpl(transactionId, startIndex, index, accessFileIndex, sources);
         startIndex = index + 1;
      }
   }

   /**
    * Helper method for do real write which writes multiple data slices to disk at once
    * 
    * @param transactionId
    * @param startIndex
    * @param endIndex
    * @param sources
    * @throws SliceStoreIOException
    */
   private void multipleWriteImpl(
         long transactionId,
         int startIndex,
         int endIndex,
         int accessFileIndex,
         Tuple2<Long, DataSlice>[] sources) throws SliceStoreIOException
   {
      int nblocks = (endIndex - startIndex + 1);
      long startBlock = sources[startIndex].getFirst();
      long endBlock = sources[endIndex].getFirst();

      RandomAccessFile accessFile = this.fileStores[accessFileIndex].getRandomAccessor();
      long relativeStartBlock = startBlock - this.fileStores[accessFileIndex].getStartBlock();

      // Make sure that block numbers in correct range for this file
      assert startBlock >= this.fileStores[accessFileIndex].getStartBlock()
            && endBlock < this.fileStores[accessFileIndex].getStartBlock()
                  + this.fileStores[accessFileIndex].getBlockCapacity();

      final ByteArrayOutputStream blocks = new ByteArrayOutputStream(this.realBlockSize * nblocks);
      final DataOutputStream output = new DataOutputStream(blocks);
      try
      {
         for (int i = startIndex; i <= endIndex; i++)
         {
            byte data[] = sources[i].getSecond().getData();
            output.writeInt(EXISTING_BLOCK_MAGIC);
            output.writeLong(transactionId);
            output.writeInt(data.length);
            output.write(data);
            // Append if a block is not full
            if (data.length < this.blockSize)
            {
               output.write(this.padding, 0, this.blockSize - data.length);
            }
         }
         output.flush();
         synchronized (accessFile)
         {
            accessFile.seek(relativeStartBlock * this.realBlockSize);
            accessFile.write(blocks.toByteArray());
            if (endBlock >= this.eofBlockNumber)
            {
               this.eofBlockNumber = endBlock + 1;
            }
         }
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Failed to write into slice store", ex);
      }
   }

   /**
    * Reads a block and parses content TODO: consider reading not max size. may be policy?
    * 
    * @throws SliceStoreNotFoundException
    * 
    * @see SliceStore
    */
   @Override
   protected DataSlice readImpl(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      return read(Arrays.asList(new SliceName[]{
         name
      })).get(0);
   }

   /**
    * @see SliceStore
    */
   @Override
   protected List<DataSlice> readImpl(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException
   {
      ensureOperational();

      long beginTime = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace("BlockMultiFileSliceStore.read(): started");

      // Why not to allow reading empty list. makes little sense but may be simplify logic
      List<DataSlice> results = new ArrayList<DataSlice>(names.size());

      // Normal processing
      long blockNumbers[] = new long[names.size()];
      for (int i = 0; i < names.size(); i++)
      {
         blockNumbers[i] = convertNameIntoBlockNumber(names.get(i));
      }
      // We assume that numbers are given in the order no fancy order rearrangement is done
      // Potentially any sequence can be given, we try to optimize sequential operation
      int startIndex = 0;
      while (startIndex < names.size()) // Do we have anything not processed
      {
         int accessFileIndex = getFileByBlockNumber(blockNumbers[startIndex]);

         if (accessFileIndex == -1)
         {
            throw new SliceStoreIOException("device block request number out of range: "
                  + blockNumbers[startIndex]);
         }

         // Find sequential numbers
         int index = startIndex;
         while (index < names.size() - 1 && (blockNumbers[index + 1] == blockNumbers[index] + 1))
         {
            if (accessFileIndex != getFileByBlockNumber(blockNumbers[index + 1]))
            {
               // changed file storage, need to break
               break;
            }
            index++;
         }
         List<DataSlice> partialResults =
               multipleReadImpl(names.subList(startIndex, index + 1), blockNumbers[startIndex],
                     blockNumbers[index], accessFileIndex);
         // Copy results to the final destination
         results.addAll(partialResults);

         startIndex = index + 1;
      }

      long totalTime = System.currentTimeMillis() - beginTime;
      if (_logger.isTraceEnabled())
      {
         long bytesRead = 0;
         for (DataSlice slice : results)
         {
            byte[] data = slice.getData();
            if (data != null)
            {
               bytesRead += data.length;
            }
         }
         _logger.trace(String.format("BlockMultiFileSliceStore.read(): successful - %.3fMB, %dms",
               bytesRead / (1024 * 1024.), totalTime));
      }

      return results;
   }

   /**
    * @see SliceStore
    */
   private List<DataSlice> multipleReadImpl(
         List<SliceName> names,
         long startBlock,
         long endBlock,
         int accessFileIndex) throws SliceStoreIOException, IllegalSourceNameException
   {
      int nblocks = (int) (endBlock - startBlock + 1);
      RandomAccessFile accessFile = this.fileStores[accessFileIndex].getRandomAccessor();

      // Make sure that block numbers in correct range for this file
      assert startBlock >= this.fileStores[accessFileIndex].getStartBlock()
            && endBlock < this.fileStores[accessFileIndex].getStartBlock()
                  + this.fileStores[accessFileIndex].getBlockCapacity();

      final long position =
            (startBlock - this.fileStores[accessFileIndex].getStartBlock()) * this.realBlockSize;
      try
      {

         List<DataSlice> results = new ArrayList<DataSlice>(nblocks);

         final byte blocksContent[] = new byte[this.realBlockSize * nblocks];

         synchronized (accessFile)
         {
            accessFile.seek(position);
            accessFile.read(blocksContent);
         }
         for (int i = 0; i < nblocks; i++)
         {
            final byte singleBlockContent[] = new byte[this.realBlockSize];
            System.arraycopy(blocksContent, i * this.realBlockSize, singleBlockContent, 0,
                  this.realBlockSize);

            final ByteArrayInputStream block = new ByteArrayInputStream(singleBlockContent);
            final DataInputStream input = new DataInputStream(block);

            final int magic = input.readInt();
            if (magic != EXISTING_BLOCK_MAGIC)
            {
               results.add(new DataSlice(names.get(i)));
               continue;
            }

            final long transactionId = input.readLong();
            final int length = input.readInt();
            // TODO: This is extremely inefficient - we copy 2
            final byte[] data = new byte[length];
            input.read(data);

            if (_logger.isTraceEnabled())
            {
               _logger.trace("Read Block #" + startBlock + i);
            }
            results.add(new DataSlice(names.get(i), transactionId, data));
         }
         return results;
      }
      catch (final IOException e)
      {
         throw new SliceStoreIOException("Error reading from file at position " + position, e);
      }
   }

   /**
    * Returns string representation of this SliceStore
    */
   @Override
   public String toString()
   {
      return getDirectoryList() + "[" + getBlockSize() + "]" + "transactions: "
            + this.transactionMap.size();
   }

   @Override
   public VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException, SliceStoreNotFoundException
   {
      return getVaultDescriptor(null, null);
   }

   /**
    * Attempts to convert a slice name into a long block number, if unable to throws an
    * IllegalSourceNameException
    * 
    * @param name
    * @return long representation of slice name
    * @throws IllegalSourceNameException
    */
   protected long convertNameIntoBlockNumber(final SliceName name)
         throws IllegalSourceNameException
   {
      try
      {
         return Long.parseLong(name.getSourceName().getName());
      }
      catch (final Exception ex)
      {
         throw new IllegalSourceNameException("The slice name must be an integer", ex);
      }
   }

   /**
    * Ensures that the slice store is operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureOperational()
   {
      if (!isOperational())
      {
         throw new IllegalStateException("The slice store must be operational");
      }
   }

   /**
    * Ensures that the slice store is not operational, otherwise throws a SliceStoreStateException
    * 
    * @throws SliceStoreStateException
    */
   protected void ensureNotOperational()
   {
      if (isOperational())
      {
         throw new IllegalStateException("The slice store must not be operational");
      }
   }

   /**
    * Ensures that a transaction is active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureActiveTransaction(SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (!isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction was not active");
      }
   }

   /**
    * Ensures that transaction exists but is not active, otherwise throws a transaction exception
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreStateException
    */
   protected void ensureNotActiveTransaction(SliceStoreTransaction transaction)
         throws SliceStoreTransactionException
   {
      if (isActiveTransaction(transaction))
      {
         throw new SliceStoreTransactionException("The supplied transaction should not be active");
      }
   }

   /**
    * Ensures that transaction is found, otherwise throws a transaction exception
    * 
    * @param transactionId
    * @throws SliceStoreTransactionException
    */
   protected void ensureTransactionFound(long transactionId) throws SliceStoreTransactionException
   {
      if (!this.transactionMap.containsKey(transactionId))
      {
         throw new SliceStoreTransactionException("A transaction by that ID already exists");
      }
   }

   private String constructPropertyFileName()
   {
      return this.fileStores[0].getDataPath() + File.separator + PROPERTIES_FILE_NAME;
   }

   private String constructStoreFileName(int index)
   {
      return this.fileStoreDefinitions.get(index).getDataPath() + File.separator + BLOCK_FILE_BASE
            + index + BLOCK_FILE_EXT;
   }

   private int getFileByBlockNumber(long blockNo)
   {
      for (int dataInd = 0; dataInd < this.fileStores.length; dataInd++)
      {
         RuntimeStorage fs = this.fileStores[dataInd];
         if (blockNo >= fs.getStartBlock() && blockNo < fs.getStartBlock() + fs.getBlockCapacity())
         {
            return dataInd;
         }
      }
      return -1;
   }

}
