//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Jul 31, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.slicestore.GenericRepairAdvice;
import org.cleversafe.layer.slicestore.RepairAdvice;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.SliceStoreManagerBase;
import org.cleversafe.layer.slicestore.exceptions.RepairException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultPermission;
import org.cleversafe.vault.exceptions.VaultConfigurationException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;

/**
 * BlockFile Implementation of SliceStoreManager
 * 
 * @see SliceStoreManager
 */
public class BlockFileSliceStoreManager extends SliceStoreManagerBase
{
   private static Logger _logger = Logger.getLogger(BlockFileSliceStoreManager.class);

   private String basePath = null;
   private int listBlockCount = -1;

   private final String aclFileName = "ACL.der";

   /**
    * Default Constructor
    */
   public BlockFileSliceStoreManager()
   {
   }

   /**
    * Constructs BlockFileSliceStoreManager taking a base path
    * 
    * @param basePath
    */
   public BlockFileSliceStoreManager(String basePath)
   {
      setBasePath(basePath);
   }

   /**
    * Setter for base path
    * 
    * @param basePath
    */
   public void setBasePath(String basePath)
   {
      _logger.trace("Base path was set to " + basePath);
      this.basePath = basePath;
   }

   /**
    * Setter for the list block count (number of blocks to return per list call).
    * This method is here so that the listBlockCount is configurable, but it is not
    * a required parameter to initialize the slicestore.
    * 
    * @param listBlockCount
    */
   public void setListBlockCount(int listBlockCount)
   {
      _logger.trace("Setting listBlockCount to " + listBlockCount);
      this.listBlockCount = listBlockCount;
   }

   /**
    * Initializes BlockFileSliceStoreManager
    */
   public void initialize()
   {
      if (this.basePath == null)
      {
         throw new InitializationException("Parameter basePath was not set");
      }
   }

   /**
    * Creates a BlockFile slice store
    * 
    * @see SliceStoreManagerBase
    */
   public SliceStore loadSliceStoreImpl(UUID vaultID) throws SliceStoreIOException
   {
      String blockFilePath = basePath + File.separator + vaultID.toString();

      _logger.debug("Loading Block File Slice Store");
      BlockFileSliceStore sliceStore = new BlockFileSliceStore(vaultID, blockFilePath);

      if (this.listBlockCount != -1)
      {
         sliceStore.setListBlockCount(this.listBlockCount);
      }
      return sliceStore;
   }
   
   

   @Override
   public void createSliceStoreImpl(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         UUID vaultID,
         VaultACL vaultACL,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      this.loadSliceStoreImpl(vaultID);
   }

   /**
    * @see SliceStoreManager
    */
   public VaultPermission loadVaultPermissions(UUID vaultID, UUID accountID)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      try
      {
         _logger.debug("Loading Block File Slice Store permissions");
         VaultACL vaultACL = readFileVaultACL(vaultID);
         return vaultACL.getPermission(accountID);
      }
      catch (VaultException e)
      {
         throw new SliceStoreIOException("Could not load permissions for vault " + vaultID);
      }
   }

   /**
    * Reads a File Vault ACL for a given Vault ID
    * 
    * @param vaultIdentifier
    * @return VaultACL
    */
   private VaultACL readFileVaultACL(UUID vaultIdentifier) throws VaultIOException,
         VaultConfigurationException
   {

      // TODO: remove direct dependency fro File impelementation. Replace with Rsolver
      String vaultACLFile =
            this.basePath + File.separator + vaultIdentifier.toString() + File.separator
                  + this.aclFileName;

      FileInputStream fileInputStream = null;
      try
      {
         fileInputStream = new FileInputStream(vaultACLFile);
         VaultACLFactory factory = new VaultACLFactory();
         VaultACL vaultACL = factory.getInstance(fileInputStream);
         return vaultACL;
      }
      catch (IOException ex)
      {
         throw new VaultIOException("Failed to read Vault ACL:" + vaultACLFile, ex);
      }
      finally
      {
         if (fileInputStream != null)
         {
            try
            {
               fileInputStream.close();
            }
            catch (final IOException e)
            {
               throw new VaultIOException("Cannot close ACL file", e);
            }
         }
      }
   }

   /**
    * 
    */
   public RepairAdvice checkIntegrity(Vault vault, VaultACL acl)
   {
      return new GenericRepairAdvice();      
   }

   public void repair(RepairAdvice advice, Vault vault, byte[] vaultDescriptorBytes, VaultACL acl)
         throws RepairException, SliceStoreExistsException
   {
     
   }
}
