//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Jason Resch
//
// Date: Aug 27, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.util.Map;
import java.util.UUID;

import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.vault.VaultACL;

/**
 * This class is extended by SliceStoreManagers which wish to support being wrapped by unreliable
 * slice stores. This class returns SliceStores that have been wrapped with an UnreliableStore which
 * has its unreliability options set from configuration.
 */
public abstract class SliceStoreManagerBase implements SliceStoreManager
{
   private SliceStoreWrapper wrapper = null;

   /**
    * Creates a SliceStore. Create operation is not unreliable.
    */
   public void createSliceStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         UUID vaultID,
         VaultACL vaultACL,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
   {
      createSliceStoreImpl(vaultType, maxSliceSize, sliceStoreSize, vaultID, vaultACL,
            vaultDescriptorBytes, options);
   }

   /**
    * Deletes a slice store after wrapping it with an unreliable one
    * @throws SliceStoreNotFoundException 
    */
   public void deleteSliceStore(UUID vaultID, Map<String, String> options)
         throws SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStore sliceStore = loadSliceStore(vaultID);
      sliceStore.deleteStore();
   }

   /**
    * Uses loadSliceStoreImpl to obtain a reliable SliceStore, then wraps it with an unreliable
    * SliceStore and sets which methods should be unreliable before returning it
    */
   public final SliceStore loadSliceStore(UUID vaultID) throws SliceStoreIOException,
         SliceStoreNotFoundException
   {
      SliceStore sliceStore = loadSliceStoreImpl(vaultID);

      if (this.wrapper != null)
         sliceStore = this.wrapper.wrap(sliceStore);

      return sliceStore;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.slicestore.SliceStoreManager#setWrapper(org.cleversafe.layer.slicestore.SliceStoreWrapper)
    */
   public void setWrapper(SliceStoreWrapper wrapper)
   {
      this.wrapper = wrapper;
   }

   /**
    * Implemented by SliceStoreManagers, simply returns a SliceStore object
    * 
    * @param vaultID
    * @param options
    * @return
    * @throws SliceStoreIOException
    */
   public abstract SliceStore loadSliceStoreImpl(UUID vaultID) throws SliceStoreIOException,
         SliceStoreNotFoundException;

   public abstract void createSliceStoreImpl(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         UUID vaultID,
         VaultACL vaultACL,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException;

}
