//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: Jul 31, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.serialization.asn1.ASN1;
import org.cleversafe.serialization.exceptions.InvalidClassException;

/**
 * This class is used for loading and saving slice store descriptor files in a 
 * consistent manner.  Before a SliceStore can be loaded, this class must be used
 * to get the descriptor of the vault to determine what type it is.
 */
public class SliceStoreDescriptorManager
{
   private static Logger _logger = Logger.getLogger(SliceStoreDescriptorManager.class);

   // TODO: check for null before being used; throw exception
   private String fileName = null;
   private String basePath = null;

   /**
    * Default Constructor
    */
   public SliceStoreDescriptorManager()
   {

   }

   /**
    * Sets the base path under which all slice store descriptors are kept. Should be separate from
    * any location where SliceStores are stored as these are independent of slice store
    * implementation.
    * 
    * @param basePath
    */
   public void setBasePath(String basePath)
   {
      _logger.trace("Set base path to " + basePath);
      this.basePath = basePath;
   }

   /**
    * Sets the file name of the slice store descriptor file. For example "Descriptor.der". All slice
    * store descriptors are kept under a directory with a name derived from the vault identifier.
    * 
    * @param fileName
    */
   public void setFileName(String fileName)
   {
      _logger.trace("Set file name to " + fileName);
      this.fileName = fileName;
   }

   /**
    * Checks whether all parameters are initialized correctly. Needed by framework to ensure correct
    * object initialization.
    */
   public void initialize()
   {
      if(this.fileName == null || this.basePath == null)
      {
         throw new InitializationException("Parameters must initialized: basePath and fileName");
      }
   }

   /**
    * Reads and deserializes a slice store descriptor file, returning a SliceStoreDescriptor object
    * 
    * @param vaultIdentifier
    * @return SliceStoreDescriptor created from file
    * @throws SliceStoreNotFoundException
    * @throws SliceStoreIOException
    */
   public SliceStoreDescriptor loadDescriptor(UUID vaultIdentifier)
         throws SliceStoreNotFoundException, SliceStoreIOException
   {
      File descriptorFile = getDescriptorFile(vaultIdentifier);

      if (descriptorFile.exists() == false)
      {
         _logger.warn("Attempted to read a non-existant slice store descriptor file "
               + descriptorFile);
         throw new SliceStoreNotFoundException("Slice Store Descriptor not found");
      }

      SliceStoreDescriptor sliceStoreDescriptor = new SliceStoreDescriptor();

      try
      {
         FileInputStream inFile = new FileInputStream(descriptorFile);
         byte[] encodedBuffer = new byte[inFile.available()];
         inFile.read(encodedBuffer);
         inFile.close();
         ASN1.DERConverter.decode(sliceStoreDescriptor, encodedBuffer);
         _logger.trace("Read a slice store descriptor file " + descriptorFile);
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Unable to read Slice Store Descriptor");
      }
      catch (InvalidClassException ex)
      {
         throw new SliceStoreIOException("Unable to deserialize Slice Store Descriptor");
      }

      return sliceStoreDescriptor;
   }

   /**
    * Serializes and writes a SliceStoreDescriptor object to disk.
    * 
    * @param vaultIdentifier
    * @param descriptor
    * @throws SliceStoreExistsException
    * @throws SliceStoreIOException
    */
   public void saveDescriptor(UUID vaultIdentifier, SliceStoreDescriptor descriptor)
         throws SliceStoreExistsException, SliceStoreIOException
   {
      File descriptorFile = getDescriptorFile(vaultIdentifier);

      if (descriptorFile.exists() == true)
      {
         _logger.warn("Attempted to over-write an existant slice store descriptor file "
               + descriptorFile);
         throw new SliceStoreExistsException("SliceStore already exists");
      }

      try
      {
         descriptorFile.getParentFile().mkdirs();
         FileOutputStream outFile = new FileOutputStream(descriptorFile);
         byte[] encodedBuffer = ASN1.DERConverter.encode(descriptor);
         outFile.write(encodedBuffer);
         outFile.flush();
         outFile.close();
         _logger.trace("Wrote slice store descriptor file " + descriptorFile + " to disk");
      }
      catch (IOException ex)
      {
         throw new SliceStoreIOException("Unable to write Slice Store Descriptor");
      }
      catch (InvalidClassException ex)
      {
         throw new SliceStoreIOException("Unable to serialize Slice Store Descriptor");
      }
   }

   /**
    * Deletes the corresponding SliceStoreDescriptor file, called after a slice store is deleted
    * 
    * @param vaultIdentifier
    * @throws SliceStoreNotFoundException
    */
   public void deleteDescriptor(UUID vaultIdentifier) throws SliceStoreNotFoundException
   {
      File descriptorFile = getDescriptorFile(vaultIdentifier);

      if (descriptorFile.exists() == false)
      {
         _logger.warn("Attempted to delete a non-existant slice store descriptor file");
         throw new SliceStoreNotFoundException("Slice Store Descriptor not found");
      }

      if (descriptorFile.delete() == false)
      {
         _logger.warn("Attempt to delete slice store descriptor file " + descriptorFile + " failed");
      }
      descriptorFile.getParentFile().delete();

   }

   protected File getDescriptorFile(UUID vaultIdentfier)
   {
      _logger.trace("Creating descriptor file name from base path and vault id");
      return new File(basePath + File.separator + vaultIdentfier.toString() + File.separator
            + fileName);
   }

}
