//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Aug 9, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.codec.Codec;
import org.cleversafe.codec.Decoder;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.layer.slicestore.exceptions.WriteTransactionDeadlockException;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.VaultLoader;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * This class implements common functionality reused throughout the SliceStore implementation
 */
public abstract class SliceStoreBase implements SliceStore
{
   private static Logger _logger = Logger.getLogger(SliceStoreBase.class);

   public static final long SLICE_STORE_SIZE_UNLIMITED = -1;

   public static final String ACL_FILE_NAME = "ACL.der";

   public static final String VAULT_DESCRIPTOR_EXTENSION = ".xml";

   private VaultDescriptor vaultDescriptor = null;

   private final ConcurrentMap<SliceName, Long> uncommittedSlices =
         new ConcurrentHashMap<SliceName, Long>();

   /**
    * Creates SliceStore without requiring that an options map be provided
    */
   public void createStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreExistsException, SliceStoreIOException
   {
      createStore(vaultType, maxSliceSize, sliceStoreSize, accessControlList, vaultDescriptorBytes,
            new HashMap<String, String>());
   }

   /**
    * Default read implementation is implemented as a loop to read each source
    */
   public final List<DataSlice> read(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      return readImpl(names);
   }

   public final DataSlice read(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      return readImpl(name);
   }

   protected abstract DataSlice readImpl(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException;

   protected List<DataSlice> readImpl(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      if (_logger.isTraceEnabled())
      {
         _logger.trace("Reading sources: " + names);
      }

      List<DataSlice> buffers = new ArrayList<DataSlice>(names.size());
      for (SliceName name : names)
      {
         buffers.add(readImpl(name));
      }
      return buffers;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.slicestore.SliceStore#write(org.cleversafe.layer.grid.DataSlice)
    */
   public final void write(DataSlice data) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException
   {
      writeImpl(data, true);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.layer.slicestore.SliceStore#write(org.cleversafe.layer.grid.DataSlice)
    */
   public final void write(DataSlice data, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      writeImpl(data, allowOverwriteNewer);
   }

   /**
    * Default write implementation is implemented as a loop to write each source
    */
   public final void write(List<DataSlice> dataSlices) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException
   {
      writeImpl(dataSlices, true);
   }

   /**
    * Default write implementation is implemented as a loop to write each source
    */
   public final void write(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      writeImpl(dataSlices, allowOverwriteNewer);
   }

   protected abstract void writeImpl(DataSlice data, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException;

   protected void writeImpl(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException
   {
      for (DataSlice dataSlice : dataSlices)
      {
         if (_logger.isTraceEnabled())
            _logger.trace("Writing source: " + dataSlice);
         writeImpl(dataSlice, allowOverwriteNewer);
      }
   }

   /**
    * Default remove implementation is implemented as a loop to remove each source
    */
   public boolean[] remove(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      boolean[] removed = new boolean[names.size()];
      for (int i = 0; i < names.size(); ++i)
      {
         if (_logger.isTraceEnabled())
         {
            _logger.trace("Removing source: " + names.get(i).toString());
         }
         removed[i] = remove(names.get(i));
      }
      return removed;
   }

   public SliceInfo getSliceInfo(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      DataSlice slice = read(name);
      if (slice.getData() != null)
      {
         return new SliceInfo(name.getSourceName(), slice.getTransactionId());
      }
      else
      {
         return null;
      }
   }

   public List<SliceInfo> verifyIntegrity() throws VaultDescriptorException, SliceStoreIOException,
         IllegalSourceNameException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException
   {
      List<SliceInfo> corruptedSlices = new ArrayList<SliceInfo>();

      listBegin();

      while (listInProgress())
      {
         List<SliceInfo> slices = listContinue();

         corruptedSlices.addAll(verifyIntegrity(slices));
      }

      return corruptedSlices;
   }

   public List<SliceInfo> verifyIntegrity(List<SliceInfo> slices) throws VaultDescriptorException,
         SliceStoreIOException, IllegalSourceNameException, VaultIOException,
         VaultSecurityException, SliceStoreNotFoundException
   {
      List<SliceInfo> corruptedSlices = new ArrayList<SliceInfo>();

      for (SliceInfo sliceInfo : slices)
      {
         // TODO: Determine slice index
         SliceName sliceName = new SliceName(sliceInfo.getSourceName(), 0);
         DataSlice slice = read(sliceName);

         if (isCorrupted(slice))
         {
            // The slice is corrupted
            corruptedSlices.add(sliceInfo);
         }
      }

      return corruptedSlices;
   }

   protected boolean isCorrupted(DataSlice slice) throws VaultDescriptorException,
         SliceStoreIOException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException
   {
      List<Codec> sliceCodecs;
      try
      {
         synchronized (this)
         {
            if (this.vaultDescriptor == null)
            {
               this.vaultDescriptor = getVaultDescriptor();
            }
         }

         sliceCodecs = this.vaultDescriptor.getSliceCodecs();
      }
      catch (VaultACLException e)
      {
         throw new VaultSecurityException("Error accessing vault ACL", e);
      }

      if (slice != null && slice.getData() != null)
      {
         // Attempt to decode the slice if this fails then the slice is corrupted
         // TODO: Deal with case where non-integrity codecs are used
         // TODO: Deal with case where decoding fails due to a security issue
         Collections.reverse(sliceCodecs);

         try
         {
            for (Codec sliceCodec : sliceCodecs)
            {
               Decoder decoder = sliceCodec.getDecoder();
               decoder.reset(slice.getSliceName().getSourceName(), slice.getTransactionId());
               slice.setData(decoder.finish(slice.getData()));
            }
         }
         catch (Exception ex)
         {
            return true;
         }
      }

      return false;
   }

   /*
    * (non-Javadoc)
    * 
    * @see java.lang.Object#toString()
    */
   @Override
   public String toString()
   {
      return getIdentification();
   }

   public abstract VaultDescriptor getVaultDescriptor() throws SliceStoreIOException,
         VaultIOException, VaultSecurityException, VaultDescriptorException,
         SliceStoreNotFoundException;

   protected VaultDescriptor getVaultDescriptor(
         String dirPath,
         PrivateKey privateKey,
         Credentials credentials) throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultACLException, SliceStoreNotFoundException
   {
      FileInputStream vaultACLInputStream = null;
      FileInputStream vaultDescriptorInputStream = null;
      VaultDescriptor descriptor = null;
      try
      {
         // Get the ACL
         vaultACLInputStream = new FileInputStream(dirPath + File.separator + ACL_FILE_NAME);
         VaultACL vaultACL = new FileVaultACL(vaultACLInputStream);

         // Get the vault descriptor
         vaultDescriptorInputStream =
               new FileInputStream(dirPath + File.separator
                     + vaultACL.getVaultIdentifier().toString() + VAULT_DESCRIPTOR_EXTENSION);

         VaultLoader vaultLoader =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     VaultLoader.class);

         ExecutionContext ctx = new ExecutionContext();
         ctx.add(VaultDescriptor.ACL_CTX_STRING, vaultACL);
         ctx.add(VaultDescriptor.ACL_ENTRY_CTX_STRING, vaultACL.getEntry(vaultACL.getOwner(),
               privateKey));
         if (credentials != null)
            ctx.add(VaultDescriptor.CREDENTIALS_CTX_STRING, credentials);

         descriptor = vaultLoader.loadVaultDescriptor(vaultDescriptorInputStream, ctx);
      }
      catch (FileNotFoundException e)
      {
         throw new SliceStoreIOException("Error opening vault acl or descriptor file", e);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Error opening vault acl or descriptor file", e);
      }
      catch (ConfigurationException e)
      {
         throw new SliceStoreIOException("Error loading vault loader from configuration", e);
      }
      catch (VaultDescriptorException e)
      {
         throw new SliceStoreIOException("Error loading vault descriptor", e);
      }
      finally
      {
         if (vaultACLInputStream != null)
         {
            try
            {
               vaultACLInputStream.close();
            }
            catch (IOException e)
            {
               _logger.warn("Could not close vault acl file", e);
            }
         }
         if (vaultDescriptorInputStream != null)
         {
            try
            {
               vaultDescriptorInputStream.close();
            }
            catch (IOException e)
            {
               _logger.warn("Could not close vault descriptor file", e);
            }
         }
      }

      return descriptor;
   }

   protected void writeVaultDescriptor(String path, byte[] vaultDescriptorBytes)
         throws SliceStoreIOException
   {
      // Write the vault descriptor
      FileOutputStream descriptorFile = null;
      try
      {
         descriptorFile = new FileOutputStream(path);
         if (vaultDescriptorBytes != null)
         {
            descriptorFile.write(vaultDescriptorBytes);
         }
      }
      catch (FileNotFoundException e)
      {
         throw new SliceStoreIOException("Could not write vault descriptor", e);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Could not write vault descriptor", e);
      }
      finally
      {
         try
         {
            if (descriptorFile != null)
            {
               descriptorFile.close();
            }
         }
         catch (IOException e)
         {
            throw new SliceStoreIOException("Could not close vault descriptor file", e);
         }
      }
   }

   protected void writeVaultACL(String path, VaultACL vaultACL) throws SliceStoreIOException
   {
      // Serialize ACL and write it
      File outFile = new File(path);

      FileOutputStream fileOutputStream = null;
      try
      {
         fileOutputStream = new FileOutputStream(outFile);
         vaultACL.flush(fileOutputStream);
      }
      catch (final IOException ex)
      {
         throw new SliceStoreIOException("Unable to serialize the ACL", ex);
      }
      catch (final VaultIOException ex)
      {
         throw new SliceStoreIOException("Unable to serialize the ACL", ex);
      }
      finally
      {
         try
         {
            if (fileOutputStream != null)
            {
               fileOutputStream.close();
            }
         }
         catch (IOException e)
         {
            throw new SliceStoreIOException("Could not close ACL file", e);
         }
      }
   }

   /**
    * Can't start transaction on an datasource that is currently part of another
    * 
    * @param dataSlice
    * @throws WriteTransactionDeadlockException
    */
   protected void addUncommittedSlice(DataSlice dataSlice) throws WriteTransactionDeadlockException
   {
      Long transactionId =
            this.uncommittedSlices.putIfAbsent(dataSlice.getSliceName(),
                  dataSlice.getTransactionId());
      if (transactionId != null && transactionId != dataSlice.getTransactionId())
      {
         throw new WriteTransactionDeadlockException("Unable to add slice "
               + dataSlice.getSliceName() + " to transaction " + dataSlice.getTransactionId()
               + ", already part of open transaction " + transactionId);
      }
   }

   protected void removeUncommittedSlices(SliceName sliceName)
   {
      this.uncommittedSlices.remove(sliceName);
   }

   protected void checkOverwriteNewer(DataSlice dataSlice) throws SliceStoreIOException,
         SliceStoreNotFoundException, SliceStoreTransactionException, IllegalSourceNameException
   {
      SliceInfo previousSliceInfo = getSliceInfo(dataSlice.getSliceName());

      if (previousSliceInfo != null
            && previousSliceInfo.getTransactionId() > dataSlice.getTransactionId())
      {
         try
         {
            if (!isCorrupted(dataSlice))
            {
               throw new SliceStoreTransactionException("Attempted to overwrite slice "
                     + previousSliceInfo.getSourceName() + " txid="
                     + previousSliceInfo.getTransactionId() + " with txid="
                     + dataSlice.getTransactionId());
            }
         }
         catch (VaultException ex)
         {
            // TODO: Determine which type of exception to throw
            throw new RuntimeException(ex);
         }
      }
   }

}
