//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Apr 25, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.util.List;
import java.util.Map;

import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;

/**
 * Interface defines mechanism of the GridAPI layer to store slices on the grid
 * 
 * Different implementations are provided: - Single File - Berkeley DB - Remote - Unreliable
 */
public interface SliceStore
{
   enum RevisionComparison
   {
      EQUAL, LESS, LESS_OR_EQUAL, GREATER, GREATER_OR_EQUAL
   }

   /**
    * Initializes a new slice store. This will erase all data on an already existing vault.
    * 
    * <pre>
    * Precondition: isOperational() == false
    * </pre>
    * 
    * @param vaultType
    * @param maxSliceSize
    * @param sliceStoreSize
    * @param accessControlList
    * @param vaultDescriptorBytes
    * @param options
    * @throws SliceStoreExistsException
    * @throws SliceStoreIOException
    */
   void createStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes,
         Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException;

   /**
    * Initializes a new slice store. This will erase all data on an already existing vault.
    * 
    * <pre>
    * Precondition: isOperational() == false
    * </pre>
    * 
    * @param vaultType
    * @param maxSliceSize
    * @param sliceStoreSize
    * @param accessControlList
    * @param vaultDescriptorBytes
    * @throws SliceStoreExistsException
    * @throws SliceStoreIOException
    */
   void createStore(
         String vaultType,
         long maxSliceSize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreExistsException, SliceStoreIOException;

   void updateStore(
         String vaultType,
         long maxSliceStize,
         long sliceStoreSize,
         VaultACL accessControlList,
         byte[] vaultDescriptorBytes) throws SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * 
    * Deletes a slice store from the system. All data will be deleted.
    * 
    * <pre>
    * Precondition: isOperational() == false
    * </pre>
    * 
    * @throws SliceStoreIOException
    */
   void deleteStore() throws SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Checks whether a slice store is operational.
    * 
    * @return true if slice store is ready to be used for data operations
    */
   boolean isOperational();

   /**
    * Starts a session to a Slice store
    * 
    * <pre>
    * Precondition: isOperational() == false
    * 
    * Postcondition: isOperational() == true
    * </pre>
    * 
    * @throws SliceStoreIOException
    */
   void startSession() throws SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Ends a session with a Slice Store. Any transactions that were active during the session are
    * rolled back.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * 
    * Postcondition: isOperational() == false
    * </pre>
    * 
    * @throws SliceStoreIOException
    */
   void endSession() throws SliceStoreIOException;

   /**
    * Regsiters with Slice Store to be notified of changes written to the slice store
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @throws SliceStoreIOException
    */
   void registerForNorNotification(NotificationHandler handler) throws SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Creates a transaction object which will remain inactive until it is begun
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: getTransaction(id) throws exception
    * 
    * Postcondition: returned transaction isActiveTransaction() == false 
    * Postcondition: transaction == getTransaction(transaction.getID())
    * </pre>
    * 
    * @return SliceStoreTransaction object, not operational until beginTransaction is used
    * @throws SliceStoreTransactionException
    * @throws SliceStoreIOException
    */
   SliceStoreTransaction createTransaction(long transactionId)
         throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Starts a transaction so that it can be used to perform write operations
    * 
    * <pre>
    * Precondition: isOperational() == true
    * Precondition: isActiveTransaction(transaction) == false
    * Precondition: transaction = getTransaction(transaction.getID())
    * 
    * Postcondition: isActiveTransaction(transaction) == true
    * </pre>
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreIOException
    */
   void beginTransaction(SliceStoreTransaction transaction) throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Commits transaction.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * Precondition: transaction = getTransaction(transaction.getID()) 
    * Precondition: isActiveTransaction(transaction) == true
    * 
    * Postcondition: isActiveTransaction(transaction) == false 
    * Postcondition: getTransaction(id) throws exception
    * </pre>
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreIOException
    */
   void commitTransaction(SliceStoreTransaction transaction) throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Rollback transaction.
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: transaction = getTransaction(transaction.getID()) 
    * Precondition: isActiveTransaction(transaction) == true
    * 
    * Postcondition: isActiveTransaction(transaction) == false 
    * Postcondition: getTransaction(id) throws exception
    * </pre>
    * 
    * @param transaction
    * @throws SliceStoreTransactionException
    * @throws SliceStoreIOException
    */
   void rollbackTransaction(SliceStoreTransaction transaction)
         throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Returns an existing transaction object with the associated transactionID if found. The
    * transaction object must have already been created using createTransaction() and not yet
    * committed or rolled back.
    * 
    * This is needed for referencing transactions over the network in which only the ID of the
    * transaction is sent.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param transactionId
    * @return transaction with the matching transaction id
    * @throws SliceStoreTransactionException
    * @throws SliceStoreIOException
    * @deprecated // TODO: REMOVE
    */
   SliceStoreTransaction getTransaction(long transactionId) throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Checks whether a slice store has created the given transaction, and that the transaction has
    * been begun but not yet committed or rolled back.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param transaction
    * @return true if the given transaction is active within the slice store
    */
   boolean isActiveTransaction(SliceStoreTransaction transaction);

   /**
    * Lists all slices in the store
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: listInProgress() == false
    * </pre>
    * 
    * @return List of sources, although not necessarily complete. listContinue() should be used to
    *         continue the list operation
    * @throws SliceStoreIOException
    */
   void listBegin() throws SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * List all slices that have a given name. Named extension are interpreted as:values could be set
    * to null to request all in selection Name and any extension value can be given as a regular
    * expression??
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: listInProgress() == false
    * </pre>
    * 
    * @param name
    * @return List of matching sources, although not necessarily complete. listContinue() should be
    *         used to continue the list operation
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   void listBegin(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException;

   /**
    * Instructs the slice store to provide an additional listing of the items matching the pattern
    * provided by listBegin
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: listInProgress() == true
    * </pre>
    * 
    * @return a list of matching source names, empty if no matches or listing has completed
    * @throws SliceStoreIOException
    */
   List<SliceInfo> listContinue() throws SliceStoreIOException,
         SliceStoreNotFoundException;

   /**
    * Cancels an active list operation
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: listInProgress() == true
    * </pre>
    * 
    * @throws SliceStoreIOException
    */
   void listStop() throws SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Determines if a list operation is currently in progress, will remain true after a call to
    * listBegin() until all matching items have been returned or until listStop() is called.
    * 
    * <pre>
    * Precondition: isOperational == true
    * </pre>
    * 
    * @return true is list operation is in progress
    * @throws SliceStoreIOException
    */
   boolean listInProgress() throws SliceStoreIOException, SliceStoreNotFoundException;

   /**
    * Checks whether a slice with a given name exist
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param name
    * @return true if the given source exists
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   boolean exists(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException;

   /**
    * Removes a slice from a store.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param name
    *           source name to be checked
    * @return true if the source existed and was removed
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    * @return True if operation was success, false otherwise.
    */
   boolean remove(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException;

   /**
    * Removes multiple slices from a store.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param names
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    * @throws SliceNotFoundException
    * @return Array of booleans that indicate whether each slice was removed or not.
    */
   boolean[] remove(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException;

   // TODO: Add remove with transaction

   /**
    * Writes a slice with a given name in the slice store. The slice will be available for reading
    * after the transaction is committed
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: isActiveTransaction(transaction) == true
    * </pre>
    * 
    * @param name
    *           source name to be written
    * @param transaction
    * @param data
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   void write(DataSlice data) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException;

   /**
    * Writes a slice with a given name in the slice store. The slice will be available for reading
    * after the transaction is committed
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: isActiveTransaction(transaction) == true
    * </pre>
    * 
    * @param name
    *           source name to be written
    * @param transaction
    * @param data
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   void write(DataSlice data, boolean allowOverwriteNewer) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException;

   /**
    * More generic form, permits to write multiple sources at once. A previous is a special case
    * with a single named data. Some implementations may take advantage and do it in more efficient
    * way than reading a one at a time
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: isActiveTransaction(transaction) == true
    * </pre>
    * 
    * @param transaction
    * @param namedData
    *           Array pair of sources and associated data
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   void write(List<DataSlice> dataSlices) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreTransactionException, SliceStoreQuotaException,
         SliceStoreNotFoundException;

   /**
    * More generic form, permits to write multiple sources at once. A previous is a special case
    * with a single named data. Some implementations may take advantage and do it in more efficient
    * way than reading a one at a time
    * 
    * <pre>
    * Precondition: isOperational() == true 
    * Precondition: isActiveTransaction(transaction) == true
    * </pre>
    * 
    * @param transaction
    * @param namedData
    *           Array pair of sources and associated data
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   void write(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
         throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException,
         SliceStoreQuotaException, SliceStoreNotFoundException;

   /**
    * Reads a slice of a given name and returns a DataSlice which is a class containing the
    * transactionId and data. If the slice is not found, a null DataSlice is returned.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param name
    * @return a DataSlice containing transaction id and data
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   DataSlice read(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException;

   /**
    * More generic form of a previous. Reads several sources at once. Some implementations may take
    * advantage and do it in more efficient way then reading a one a time. For the slices that are
    * not found, if any, the DataSlice array contains null DataSlices.
    * 
    * <pre>
    * Precondition: isOperational() == true
    * </pre>
    * 
    * @param names -
    *           sources to be read
    * @return array of DataSlices in the order of SliceNames
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   List<DataSlice> read(List<SliceName> names) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException;

   /**
    * FIXME: Write me
    * 
    * @param name
    * @return
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    */
   SliceInfo getSliceInfo(SliceName name) throws SliceStoreIOException, IllegalSourceNameException,
         SliceStoreNotFoundException;

   /**
    * Verifies the integrity of the store.
    * 
    * @return A list of corrupted slice names
    * @throws VaultDescriptorException
    * @throws SliceStoreIOException
    * @throws IllegalSourceNameException
    * @throws VaultSecurityException
    * @throws VaultIOException
    */
   List<SliceInfo> verifyIntegrity() throws VaultDescriptorException, SliceStoreIOException,
         IllegalSourceNameException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException;

   List<SliceInfo> verifyIntegrity(List<SliceInfo> slices) throws VaultDescriptorException,
         SliceStoreIOException, IllegalSourceNameException, VaultIOException,
         VaultSecurityException, SliceStoreNotFoundException;

   /**
    * @return Size of slice store in bytes
    */
   // FIXME: public long getSize();
   /**
    * For debugging and logging purposes. May not be as exhaustive as toString
    */
   String getIdentification();

   VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException, SliceStoreNotFoundException;
}
