//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
// -----------------------
// Author: Manish Motwani/Ilya Volvoski
//
// Date: Apr 23, 2007
// ---------------------

package org.cleversafe.layer.protocol;

import java.util.*;

import org.cleversafe.layer.protocol.exceptions.OperationNotDefinedException;

/**
 * Enumeration for various types of protocol operations.
 * 
 * @author Manish Motwani
 * @author Ilya Volvovski
 */
public enum GridProtocolOperation implements ProtocolOperation
{
   // Noop
   NOOP_REQUEST(0),
   NOOP_RESPONSE(1),

   // Sessions
   BEGIN_SESSION_REQUEST(2),
   BEGIN_SESSION_RESPONSE(3),
   END_SESSION_REQUEST(4),
   END_SESSION_RESPONSE(5),

   // Notifications
   REGISTER_FOR_NOTIFICATIONS_REQUEST(6),
   REGISTER_FOR_NOTIFICATIONS_RESPONSE(7),
   WRITE_NOTIFY_UNSOLICITED(8),

   // Errors
   ERROR_UNSOLICITED(9), // UNSOLICITED_ERROR AT SERVER
   ERROR_RESPONSE(10),

   // Transactions
   BEGIN_TRANSACTION_REQUEST(13),
   BEGIN_TRANSACTION_RESPONSE(14),
   COMMIT_TRANSACTION_REQUEST(15),
   COMMIT_TRANSACTION_RESPONSE(16),
   ROLLBACK_TRANSACTION_REQUEST(17),
   ROLLBACK_TRANSACTION_RESPONSE(18),

   // Slice operations
   MULTIPLE_READ_REQUEST(19),
   MULTIPLE_READ_RESPONSE(20),
   MULTIPLE_WRITE_REQUEST(21),
   MULTIPLE_WRITE_RESPONSE(22),
   MULTIPLE_REMOVE_REQUEST(23),
   MULTIPLE_REMOVE_RESPONSE(24),
   EXISTS_REQUEST(25),
   EXISTS_RESPONSE(26),

   // Authentication operations
   PASSWORD_AUTHENTICATION_REQUEST(27),
   PASSWORD_AUTHENTICATION_RESPONSE(28),

   // Authorization operations
   VAULT_BIND_REQUEST(29),
   VAULT_BIND_RESPONSE(30),

   // Store creation and removal operations
   CREATE_STORE_REQUEST(31),
   CREATE_STORE_RESPONSE(32),
   REMOVE_STORE_REQUEST(33),
   REMOVE_STORE_RESPONSE(34),

   // Listing operations
   LIST_BEGIN_REQUEST(41),
   LIST_BEGIN_RESPONSE(42),
   LIST_CONTINUE_REQUEST(43),
   LIST_CONTINUE_RESPONSE(44),
   LIST_INPROGRESS_REQUEST(45),
   LIST_INPROGRESS_RESPONSE(46),
   LIST_STOP_REQUEST(47),
   LIST_STOP_RESPONSE(48),

   // Version operations
   GET_PROTOCOLS_REQUEST(49),
   GET_PROTOCOLS_RESPONSE(50),
   GET_PROTOCOL_VERSIONS_REQUEST(51),
   GET_PROTOCOL_VERSIONS_RESPONSE(52),

   INTEGRITY_VERIFICATION_REQUEST(53),
   INTEGRITY_VERIFICATION_RESPONSE(54),

   ;

   private final int operationCode;

   /**
    * 
    * @param opCode
    *           associated with this operation
    */
   private GridProtocolOperation(int opCode)
   {
      this.operationCode = opCode;
   }

   /**
    * 
    * @return associated code
    */
   public int getOperationCode()
   {
      return this.operationCode;
   }

   /**
    * Returns the Operation enumeration from an operation code.
    * 
    * @param operationCode
    *           The operation code.
    * @return The Operation enumeration.
    * @throws OperationNotDefinedException
    */
   public static ProtocolOperation fromOperationCode(int operationCode)
         throws OperationNotDefinedException
   {
      for (GridProtocolOperation po : GridProtocolOperation.values())
      {
         if (po.getOperationCode() == operationCode)
         {
            return po;
         }
      }

      throw new OperationNotDefinedException(
            "Could not create ProtocolOperation from operation code", operationCode);
   }

   /**
    * Get a list of all grid protocol operations
    * 
    * @return A list of all grid protocol operations
    */
   public static List<ProtocolOperation> getOperationList()
   {
      List<ProtocolOperation> operations = new ArrayList<ProtocolOperation>();

      for (GridProtocolOperation operation : GridProtocolOperation.values())
      {
         operations.add(operation);
      }

      return operations;
   }

   public static void main(String[] args)
   {
      for (GridProtocolOperation po : GridProtocolOperation.values())
      {
         System.out.println(po.ordinal() + ". " + po + " Op=" + po.getOperationCode());
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see java.lang.Enum#toString()
    */
   @Override
   public String toString()
   {
      return Integer.toString(operationCode);
   }
}
