//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 14, 2007
//---------------------

package org.cleversafe.layer.grid.smartcontroller;

import org.apache.log4j.Logger;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.StoreSliceSet;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.layer.grid.exceptions.GridLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;

import com.evelopers.unimod.runtime.context.StateMachineContext;

public class WriteStateMachine extends GridControllerStateMachine
{
   private static Logger _logger = Logger.getLogger(WriteStateMachine.class);

   /************************************************************************************************
    * World state
    ***********************************************************************************************/

   // Transaction for this write
   protected GridTransaction gridTransaction;

   /**
    * Called by Unimod
    */
   public WriteStateMachine()
   {
      super();

      // Initial world state
      this.gridTransaction = null;
   }

   /**
    * Set this operation's grid transaction (mandatory)
    * 
    * @param gridTransaction
    */
   public void setGridTransaction(GridTransaction gridTransaction)
   {
      this.gridTransaction = gridTransaction;
   }

   /************************************************************************************************
    * Permissible world operations
    ***********************************************************************************************/

   /**
    * Job to start transactions asynchronously
    */
   private class StartTransactionTask extends ConcurrentFunction
   {
      private SliceStore store;

      public StartTransactionTask(SliceStore store)
      {
         this.store = store;
      }

      public void run()
      {
         WriteStateMachine world = WriteStateMachine.this;

         try
         {
            long txid = world.gridTransaction.getId();

            // Attempt to start SliceStoreTransaction
            SliceStoreTransaction tx;
            try
            {
               tx = store.createTransaction(txid);
               store.beginTransaction(tx);
            }
            catch (final SliceStoreTransactionException ex)
            {
               // Transaction already exists
               tx = store.getTransaction(txid);
               // FIXME: Has this transaction been started for sure?
            }
            assert tx != null;

            // Add to the grid transaction. This may be done by more than one
            // thread in a race, but that's OK
            world.gridTransaction.addSliceStoreTransaction(tx);

            // This store has a successfully started transaction
            this.fireEvent(new SliceStoreTaskSuccessEvent(this.store));
         }
         catch (final Exception ex)
         {
            if (_logger.isDebugEnabled())
               _logger.debug("Begin transaction error '" + ex + "' from store: " + store);

            // Fire exception event
            this.fireEvent(new SliceStoreTaskExceptionEvent(this.store, ex));
         }
      }
   }

   /**
    * @unimod.action.descr Attempts to start transactions on necessary SliceStores and add to the
    *                      GridTransaction
    */
   public void beginTransactions(StateMachineContext context)
   {
      // Run start transaction jobs on necessary stores
      for (SliceStore store : this.matrix.getStores())
      {
         // See if a SliceStoreTransaction for this store exists
         this.matrix.setStoreStatus(store, OperationScorecard.Status.PENDING);
         if (this.gridTransaction.find(store) == null)
         {
            this.runTask(new StartTransactionTask(store));
         }
         else
         {
            // This store has a successfully started transaction
            this.fireEvent(new SliceStoreTaskSuccessEvent(store));
         }
      }
   }

   /**
    * @unimod.action.descr Called when transaction has been successfully started on a SliceStore
    */
   public void transactionStarted(StateMachineContext context)
   {
      UnimodStateMachineContext unimodContext = (UnimodStateMachineContext) context;
      SliceStoreTaskSuccessEvent event =
            (SliceStoreTaskSuccessEvent) unimodContext.getCurrentEvent();

      // Set existing status for all of this store's entries in the matrix
      this.matrix.setStoreStatus(event.getSliceStore(), OperationScorecard.Status.EXISTING);
   }

   /**
    * Job to write asynchronously
    */
   private class WriteTask extends ConcurrentFunction
   {
      private StoreSliceSet sliceData;

      public WriteTask(StoreSliceSet sliceData)
      {
         this.sliceData = sliceData;
      }

      public void run()
      {
         SliceStore store = this.sliceData.getSliceStore();

         try
         {
            store.write(sliceData.getSlices());

            // Write was successful
            this.fireEvent(new SliceStoreTaskSuccessEvent(store));
         }
         catch (final Exception ex)
         {
            if (_logger.isDebugEnabled())
               _logger.debug("Write error '" + ex + "' from store: " + store);

            // Fire exception event
            this.fireEvent(new SliceStoreTaskExceptionEvent(store, ex));
         }
      }
   }

   /**
    * @unimod.action.descr Attempts to write to all SliceStores concurrently
    */
   public void beginWrite(StateMachineContext context)
   {
      // Run all write jobs
      for (SliceStore store : this.matrix.getStores())
      {
         StoreSliceSet set = this.matrix.getStoreSliceSet(store);

         if (set.size() > 0)
         {
            this.matrix.setStoreStatus(store, OperationScorecard.Status.PENDING);
            this.runTask(new WriteTask(set));
         }
      }
   }

   /**
    * @unimod.action.descr Called when a write was successful on a SliceStore
    */
   public void writeComplete(StateMachineContext context)
   {
      UnimodStateMachineContext unimodContext = (UnimodStateMachineContext) context;
      SliceStoreTaskSuccessEvent event =
            (SliceStoreTaskSuccessEvent) unimodContext.getCurrentEvent();

      // Set ignored status for all of this store's entries in the matrix
      this.matrix.setStoreStatus(event.getSliceStore(), OperationScorecard.Status.IGNORED);
   }

   /**
    * @unimod.action.descr Is the current status acceptable by policy?
    */
   public void checkPolicyStatusOK(StateMachineContext context)
   {
      InformationDispersalCodec ida = this.vault.getInformationDispersalCodec();
      int maxErrors =
            ida.getNumSlices() - Math.max(this.vault.getWriteThreshold(), ida.getThreshold());

      for (SourceName source : this.matrix.getSources())
      {
         int numErrors = this.matrix.getSourceStatusCount(source, OperationScorecard.Status.ERROR);
         if (numErrors > maxErrors)
         {
            // This source does not meet policy
            this.fireEvent(new UnimodEvent(EV_NO), true);
            return;
         }
      }

      this.fireEvent(new UnimodEvent(EV_YES), true);
   }

   /**
    * @unimod.action.descr Determine what exception to throw based on the grid state and error threshold
    */
   public void determineException(StateMachineContext context) throws GridLayerException
   {
      InformationDispersalCodec ida = this.vault.getInformationDispersalCodec();
      int maxErrors =
            ida.getNumSlices() - Math.max(this.vault.getWriteThreshold(), ida.getThreshold());
      super.determineExceptionHelper("Write", maxErrors);
   }

   /************************************************************************************************
    * External interface
    ***********************************************************************************************/

   /**
    * Multi-datasource write
    * @throws ControllerStoresNotFoundException 
    */
   public void run(OperationScorecard data) throws ControllerTransactionException,
         ControllerIOException, ControllerIllegalSourceNameException,
         ControllerGridStateUnknownException, ControllerStoresNotFoundException
   {
      try
      {
         this.setMatrix(data);
         super.runStateMachine(new UnimodStateMachineContext());
      }
      catch (final GridLayerException ex)
      {
         // Marshal to known exceptions
         if (ex instanceof ControllerTransactionException)
         {
            throw (ControllerTransactionException) ex;
         }
         else if (ex instanceof ControllerIOException)
         {
            throw (ControllerIOException) ex;
         }
         else if (ex instanceof ControllerIllegalSourceNameException)
         {
            throw (ControllerIllegalSourceNameException) ex;
         }
         else if (ex instanceof ControllerGridStateUnknownException)
         {
            throw (ControllerGridStateUnknownException) ex;
         }
         else if (ex instanceof ControllerStoresNotFoundException)
         {
            throw (ControllerStoresNotFoundException) ex;
         }
         else
         {
            throw new RuntimeException("Unexpected exception type", ex);
         }
      }
   }
}
