//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.layer.grid.simplecontroller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.commons.lang.NotImplementedException;
import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.OperationControllerBase;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.SourceSliceSet;
import org.cleversafe.layer.grid.StoreSliceSet;
import org.cleversafe.layer.grid.Transaction;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.util.TransactionIdGenerator;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

// TODO: Describe class or interface
public class SimpleWriteController extends OperationControllerBase implements WriteController
{
   private static final Logger _logger = Logger.getLogger(SimpleWriteController.class);

   /**
    * Executor class for begin transaction
    */
   private static class SliceStoreTransactionBeginExecutor
         implements
            Callable<SliceStoreTransaction>
   {
      private long transactionId;
      private SliceStore sliceStore;

      public SliceStoreTransactionBeginExecutor(long transactionId, SliceStore sliceStore)
      {
         this.transactionId = transactionId;
         this.sliceStore = sliceStore;
      }

      public SliceStoreTransaction call() throws SliceStoreIOException,
            SliceStoreTransactionException, SliceStoreNotFoundException
      {
         SliceStoreTransaction sliceStoreTransaction = sliceStore.createTransaction(transactionId);

         sliceStore.beginTransaction(sliceStoreTransaction);

         return sliceStoreTransaction;
      }
   }
   /**
    * Executor for transaction (commit/rollback)
    */
   private static class SliceStoreTransactionOperationExecutor implements Callable<Void>
   {
      public enum TransactionOperation
      {
         COMMIT, ROLLBACK
      };

      private SliceStore sliceStore;
      private SliceStoreTransaction sliceStoreTransaction;
      private TransactionOperation operation;

      public SliceStoreTransactionOperationExecutor(
            SliceStore sliceStore,
            SliceStoreTransaction sliceStoreTransaction,
            TransactionOperation operation)
      {
         this.sliceStore = sliceStore;
         this.sliceStoreTransaction = sliceStoreTransaction;
         this.operation = operation;
      }

      public Void call() throws SliceStoreIOException, SliceStoreTransactionException,
            SliceStoreNotFoundException
      {
         switch (operation)
         {
            case COMMIT :
               sliceStore.commitTransaction(sliceStoreTransaction);
               break;

            case ROLLBACK :
               sliceStore.rollbackTransaction(sliceStoreTransaction);
               break;
         }

         return null;
      }
   }

   private static class SliceStoreWriter implements Callable<Void>
   {
      private List<DataSlice> slices;
      private SliceStore store;

      public SliceStoreWriter(SliceStore store, List<DataSlice> slices)
      {
         this.slices = slices;
         this.store = store;
      }

      public Void call() throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException,
            SliceStoreNotFoundException
      {
         store.write(this.slices);
         return null;
      }
   }

   /**
    * Class variables
    */
   // private boolean currentTransactionSuccess = true;
   private Transaction transaction = null;
   private int writeTolerance = 0;
   private ExecutorService executor;

   /**
    * This is for testing
    * 
    * @param vault
    * @param writeTolerance
    * @throws ControllerTransactionBeginException
    */
   public SimpleWriteController(Vault vault, int writeTolerance, ExecutorService executor)
         throws ControllerTransactionException
   {
      super(vault);
      this.executor = executor;
      // new BoundedThreadPoolExecutor("WriteController-" + this.,2 * vault.getWidth());
      this.writeTolerance = writeTolerance;
      assert writeTolerance <= this.getVault().getThreshold() : "Write tolerance is higher then threshold";
      beginTransaction(TransactionIdGenerator.generate());
   }

   /**
    * Begins a transaction with a specified id
    * 
    * @param transactionId
    * @throws ControllerTransactionBeginException
    */
   private void beginTransaction(long transactionId) throws ControllerTransactionException
   {
      assert isOperational() : "Attempt to invoke beginTransaction() on non-operational write controller";

      List<SliceStore> sliceStores = getVault().getSliceStores();

      this.transaction =
            new Transaction(transactionId, new SliceStoreTransaction[sliceStores.size()]);

      List<Future<SliceStoreTransaction>> futureSliceStoreTransactions =
            new ArrayList<Future<SliceStoreTransaction>>();

      // Begin the transaction on each slice store
      for (int idx = 0; idx < sliceStores.size(); idx++)
      {
         SliceStore sliceStore = sliceStores.get(idx);

         assert sliceStore != null;

         SliceStoreTransactionBeginExecutor transactionBeginController;
         transactionBeginController =
               new SliceStoreTransactionBeginExecutor(this.transaction.getId(), sliceStore);

         futureSliceStoreTransactions.add(executor.submit(transactionBeginController));
      }

      // Wait for transaction creation to complete on each slice store

      int successCount = 0;
      for (int idx = 0; idx < futureSliceStoreTransactions.size(); idx++)
      {
         Future<SliceStoreTransaction> futureSliceStoreTransaction =
               futureSliceStoreTransactions.get(idx);

         assert futureSliceStoreTransaction != null;

         SliceStoreTransaction sliceStoreTransaction = null;
         try
         {
            sliceStoreTransaction = futureSliceStoreTransaction.get();
         }
         catch (Exception ex)
         {
            _logger.debug("failed to begin transaction on " + idx + " slice", ex);
         }

         if (sliceStoreTransaction != null)
         {
            this.transaction.setSliceStoreTransaction(idx, sliceStoreTransaction);
            successCount++;
         }
      }
      // require all transactions to start
      if (successCount < this.getVault().getWidth() - this.writeTolerance)
      {
         try
         {
            rollback();
         }
         catch (Exception ex)
         {
            _logger.error("Unable to rollback transaction", ex);
            this.transaction = null;
         }

         throw new ControllerTransactionException("Unable to begin transaction " + transactionId
               + " on the threshold number of slice stores");
      }
      if (_logger.isTraceEnabled())
         _logger.trace("Began transaction " + getTransactionId());
   }

   public long getTransactionId()
   {
      assert transaction != null;
      return transaction.getId();
   }

   private void performTransactionOperation(
         SliceStoreTransactionOperationExecutor.TransactionOperation operation)
         throws ControllerTransactionException
   {
      List<SliceStore> sliceStores = getVault().getSliceStores();
      List<Future<Void>> futureSliceStoreOperations = new ArrayList<Future<Void>>();

      // Begin transaction operation on slice stores
      for (int idx = 0; idx < sliceStores.size(); idx++)
      {
         SliceStore sliceStore = sliceStores.get(idx);

         SliceStoreTransaction sliceStoreTransaction =
               this.transaction.getSliceStoreTransaction(idx);

         assert sliceStore != null;
         if (sliceStoreTransaction != null)
         {
            SliceStoreTransactionOperationExecutor operationController;
            operationController =
                  new SliceStoreTransactionOperationExecutor(sliceStore, sliceStoreTransaction,
                        operation);

            futureSliceStoreOperations.add(executor.submit(operationController));
         }
      }

      // No matter what we are done with this transaction
      long transactionId = getTransactionId();
      this.transaction = null;

      // Wait for transaction operation to complete on each slice store
      int successCount = 0;
      for (Future<Void> futureSliceStoreOperation : futureSliceStoreOperations)
      {
         try
         {
            futureSliceStoreOperation.get();
            successCount++;
         }
         catch (Exception ex)
         {
            _logger.debug("Operation " + operation + " failed", ex);
         }
      }

      if (successCount < getVault().getWidth() - this.writeTolerance)
      {
         if (operation == SliceStoreTransactionOperationExecutor.TransactionOperation.COMMIT)
         {
            throw new ControllerTransactionException("Unable to commit transaction on transaction "
                  + transactionId + " on the threshold number of slice stores");
         }
         else if (operation == SliceStoreTransactionOperationExecutor.TransactionOperation.ROLLBACK)
         {
            throw new ControllerTransactionException("Unable to rollback transaction "
                  + transactionId + " on the threshold number of slice stores");
         }
      }
   }

   public void commit() throws ControllerTransactionException, ControllerIOException
   {
      try
      {
         performTransactionOperation(SliceStoreTransactionOperationExecutor.TransactionOperation.COMMIT);
      }
      catch (ControllerException e)
      {
         throw new ControllerTransactionException("Failed to commit", e);
      }
      finally
      {
         try
         {
            doShutdown();
         }
         catch (ControllerException e)
         {
            throw new ControllerTransactionException("Failed to shutdown after failed commit", e);
         }
      }
   }

   public void rollback() throws ControllerTransactionException, ControllerIOException
   {
      try
      {
         performTransactionOperation(SliceStoreTransactionOperationExecutor.TransactionOperation.ROLLBACK);
      }
      catch (ControllerException e)
      {
         throw new ControllerTransactionException("Failed to rollabck", e);
      }
      finally
      {
         try
         {
            doShutdown();
         }
         catch (ControllerException e)
         {
            throw new ControllerTransactionException("Failed to shutdown after failed rollback", e);
         }
      }
   }

   public void write(DataSource source) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException
   {
      this.writeImpl(Arrays.asList(new DataSource[]{
         source
      }));
   }

   public void write(List<DataSource> sources) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException
   {
      this.writeImpl(sources);
   }

   private void writeImpl(List<DataSource> sources) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerIllegalSourceNameException
   {
      // Create a disperse matrix
      List<SourceName> sourceNames = new ArrayList<SourceName>(sources.size());
      for (DataSource source : sources)
      {
         sourceNames.add(source.getName());
      }
      OperationScorecard dmMartix =
            new OperationScorecard(sourceNames, this.getVault().getSliceStores());

      for (DataSource source : sources)
      {
         try
         {
            SourceSliceSet dispersedSource = this.getVault().applyDispersalStack(source);
            PlacedSourceSliceSet placedSlices = this.getVault().mapSliceSetToStore(dispersedSource);
            dmMartix.addSourceSliceSet(placedSlices, OperationScorecard.Status.EXISTING);
         }
         catch (InformationDispersalException dispersalExc)
         {
            throw new ControllerInformationDispersalException(
                  "Error applying dispersal during write operation", dispersalExc);
         }
         catch (DataTransformationException transfExc)
         {
            throw new ControllerDataTransformationException(
                  "Error applying transformaton during write operation", transfExc);
         }
      }

      // Start writing
      List<Future<Void>> futureSliceWrites = new ArrayList<Future<Void>>();

      for (SliceStore store : this.getVault().getSliceStores())
      {
         // PlacedSourceSliceSet placedSlices = this.getVault().mapSliceSetToStore(dispersedSource);
         StoreSliceSet slices = dmMartix.getStoreSliceSet(store);
         SliceStoreWriter writer = new SliceStoreWriter(store, slices.getSlices());
         futureSliceWrites.add(executor.submit(writer));
      }
      // Wait for all async write tasks to complete
      int successCount = 0;
      if (_logger.isTraceEnabled())
         _logger.trace("Waiting for write request to complete...");
      for (Future<Void> futureSliceWrite : futureSliceWrites)
      {
         try
         {
            futureSliceWrite.get();
            if (_logger.isTraceEnabled())
               _logger.trace("Successfully complete slice write");
            successCount++;
         }
         catch (InterruptedException e)
         {
            throw new RuntimeException("Slice store interrupted", e);
         }
         catch (ExecutionException e)
         {
            try
            {
               throw e.getCause();
            }
            catch (IllegalSourceNameException ex)
            {
               throw new ControllerIllegalSourceNameException("Illegal source name", ex);
            }
            catch (Throwable ex)
            {
               _logger.warn("Error while perfroming write:" + ex.getMessage());
               throw new RuntimeException("Unknown write exception", ex);
            }
         }
      }

      if (successCount < getVault().getWidth() - this.writeTolerance)
      {
         throw new ControllerIOException("Unable to write to the required number of slice stores");
      }
   }

   /**
    * @return the writeTolerance
    */
   public int getWriteTolerance()
   {
      return this.writeTolerance;
   }

   /**
    * @param writeTolerance
    *           the writeTolerance to set
    */
   public void setWriteTolerance(int writeTolerance)
   {
      this.writeTolerance = writeTolerance;
   }

   public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
   {
      throw new NotImplementedException("addCorruptedSliceObserver is not implemented");
   }

}
