//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Aug 21, 2007
//---------------------

package org.cleversafe.layer.grid.simplecontroller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.commons.lang.NotImplementedException;
import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.OperationControllerBase;
import org.cleversafe.layer.grid.PlacedSliceName;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.StoreSliceSet;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;

public class SimpleReadController extends OperationControllerBase implements ReadController
{
   private static Logger _logger = Logger.getLogger(SimpleReadController.class);

   /**
    * Just to demonstrate different strategies
    */
   enum Order
   {
      NORMAL, REVERSE, RANDOM
   }
   public static class FutureExceptionWrapper extends Exception
   {
      /**
       * 
       */
      private static final long serialVersionUID = 1L;
      private SliceStoreLayerException cause;
      private SliceStore originator;

      /**
       * 
       * @param cause -
       *           the original exception
       * @param originator
       *           store that originated the exception
       */
      public FutureExceptionWrapper(SliceStoreLayerException cause, SliceStore originator)
      {
         super();
         this.cause = cause;
         this.originator = originator;
      }

      /**
       * @return the cause
       */
      public SliceStoreLayerException getCause()
      {
         return cause;
      }

      /**
       * @return the store originator
       */
      public SliceStore getOriginator()
      {
         return originator;
      }
   }

   private static class SliceStoreReader implements Callable<List<DataSlice>>
   {
      private List<SliceName> sliceNames;
      private SliceStore sliceStore;

      SliceStoreReader(List<SliceName> sliceNames, SliceStore sliceStore)
      {
         this.sliceNames = sliceNames;
         this.sliceStore = sliceStore;
      }

      public List<DataSlice> call() throws FutureExceptionWrapper
      {
         try
         {
            return this.sliceStore.read(this.sliceNames);
         }
         catch (SliceStoreLayerException ex)
         {
            throw new FutureExceptionWrapper(ex, this.sliceStore);
         }
      }
   }

   // Member variables
   private Order order;
   private ExecutorService executor;
   private StringBuffer cumulativeFailureReason = null;

   /**
    * By default normal order of slices which is performance optimal
    */
   public SimpleReadController(Vault vault, ExecutorService executor)
   {
      this(vault, Order.NORMAL, executor);
   }

   public SimpleReadController(Vault vault, Order order, ExecutorService executor)
   {
      super(vault);
      this.order = order;
      this.executor = executor;
   }

   public DataSource read(SourceName name) throws ControllerDataTransformationException,
         ControllerInformationDispersalException, ControllerIOException,
         ControllerGridStateUnknownException, ControllerIllegalSourceNameException
   {
      return readImpl(Arrays.asList(new SourceName[]{
         name
      })).get(0);
   }

   public List<DataSource> read(List<SourceName> names)
         throws ControllerDataTransformationException, ControllerInformationDispersalException,
         ControllerIOException, ControllerGridStateUnknownException,
         ControllerIllegalSourceNameException
   {
      return readImpl(names);
   }

   private List<DataSource> readImpl(List<SourceName> names)
         throws ControllerDataTransformationException, ControllerInformationDispersalException,
         ControllerIOException, ControllerGridStateUnknownException,
         ControllerIllegalSourceNameException
   {
      this.cumulativeFailureReason = null;

      // Try to read at least threshold number of slices on each store.
      // Transaction count is not supported, as soon as 2 different transactions are
      // discovered an error is generated
      int maxSliceIndex = this.getVault().getSliceStores().size();
      int[] readOrder = orderSlices(maxSliceIndex);
      List<SliceStore> orderedSlices = new ArrayList<SliceStore>(maxSliceIndex);
      // Arrange slice stores according to the order
      for (int i = 0; i < maxSliceIndex; i++)
      {
         orderedSlices.add(this.getVault().getSliceStores().get(readOrder[i]));
      }

      // Read in specified order until all source are assembled or there is no stores left
      // If any read error occurs the whole operation fails

      // It is empty now, we'll fill it an as we go
      OperationScorecard dmMatrix = new OperationScorecard(names, orderedSlices);

      int nextSliceStoreIndex = 0; // index of a next store in readOrder to be read

      // Iterate over list of slices until either all sources are restores or stores are exhausted
      while (nextSliceStoreIndex < maxSliceIndex)
      {
         List<Tuple2<SourceName, Integer>> sourcesToRead =
               this.getVault().getIncompleteSources(dmMatrix);
         if (sourcesToRead.size() == 0)
         {
            break;
         }

         // Create a parallel operation to read required number of slices to have
         // the required number of slices
         Tuple2<Integer, List<SourceName>> missed = getMaxMissingSlices(sourcesToRead);
         int tryNumber = missed.getFirst();
         if (_logger.isTraceEnabled())
            _logger.trace("Trying to read " + tryNumber + " slices");
         List<SliceStore> readGroup = new ArrayList<SliceStore>(tryNumber);
         for (; nextSliceStoreIndex < this.getVault().getWidth() && tryNumber > 0; nextSliceStoreIndex++)
         {
            if (orderedSlices.get(nextSliceStoreIndex).isOperational())
            {
               readGroup.add(orderedSlices.get(nextSliceStoreIndex));
               tryNumber--;
            }
         }
         if (_logger.isTraceEnabled())
            _logger.trace("Trying to read " + tryNumber + " slices for " + missed.getSecond());
         readIncompleteSources(missed.getSecond(), readGroup, dmMatrix);
      }

      try
      {
         return analyzeResult(this.getVault().rebuildSources(dmMatrix), names.size());
      }
      catch (InformationDispersalException e)
      {
         throw new ControllerInformationDispersalException("IDA error while rebuilding sources", e);
      }
      catch (DataTransformationException e)
      {
         throw new ControllerDataTransformationException(
               "Transformation error while rebuilding sources", e);
      }
   }

   /**
    * Makes sure that all requested data sources are read If not throws an the best guess exception
    * collected during reading
    * 
    * @param rebuildSources -
    *           resulting data sources
    * @return the original list if sufficient
    */
   private List<DataSource> analyzeResult(List<DataSource> rebuildSources, int expectedSources)
         throws ControllerIOException
   {
      if (rebuildSources.size() == expectedSources)
      {
         return rebuildSources;
      }

      if (this.cumulativeFailureReason != null)
      {
         throw new ControllerIOException(this.cumulativeFailureReason.toString());
      }
      else
      {
         throw new ControllerIOException("Inconsistent state from at least one of the sources");
      }
   }

   /**
    * Returns source requiring more slices and minimal additional # of slices to read for success
    * 
    * @param missingSlices
    * @return
    */
   private Tuple2<Integer, List<SourceName>> getMaxMissingSlices(
         List<Tuple2<SourceName, Integer>> missingSlices)
   {
      int max = 0;
      List<SourceName> missingSources = new ArrayList<SourceName>(missingSlices.size());
      for (Tuple2<SourceName, Integer> missing : missingSlices)
      {
         missingSources.add(missing.getFirst());
         if (missing.getSecond() > max)
         {
            max = missing.getSecond();
         }
      }
      assert max > 0 : "Missing slices don't really miss anthing";
      return new Tuple2<Integer, List<SourceName>>(max, missingSources);
   }

   /**
    * Read specified sources from specified stores
    * 
    * @param namesToRead
    * @param readGroup
    * @param dmMatrix
    * @throws ControllerIllegalSourceNameException
    */
   private void readIncompleteSources(
         List<SourceName> sourcesToRead,
         List<SliceStore> readGroup,
         OperationScorecard dmMatrix) throws ControllerIllegalSourceNameException
   {
      List<Future<List<DataSlice>>> futureSliceBuffers = new ArrayList<Future<List<DataSlice>>>();

      // Create a list of required SliceName to read
      for (SliceStore sliceStore : readGroup)
      {
         List<SliceName> slicesToRead = new ArrayList<SliceName>(sourcesToRead.size());
         for (SourceName name : sourcesToRead)
         {
            // FIXME: This is a known performance hotspot. Upcoming improvements to the matrix will
            // resolve this
            // All slice names with corresponding stores
            List<PlacedSliceName> placedSliceNames =
                  this.getVault().mapSourceNameToSliceNames(name);
            for (PlacedSliceName psn : placedSliceNames)
            {
               // Choose only those that we are interested during this sub-read
               // This operation should be encapsulated in matrix
               if (psn.getStore() == sliceStore)
               {
                  slicesToRead.add(new SliceName(name, psn.getSliceIndex()));
                  break;
               }
            }
            if (_logger.isTraceEnabled())
               _logger.trace("Initiating reading from slice store " + sliceStore);
         }
         futureSliceBuffers.add(executor.submit(new SliceStoreReader(slicesToRead, sliceStore)));
      }

      int storeIndex = 0;
      for (Future<List<DataSlice>> futureSliceBuffer : futureSliceBuffers)
      {
         try
         {
            // Wait for the slice read request to complete
            List<DataSlice> dataSlices = futureSliceBuffer.get(); // size of this.sources.length
            // assert dataSlices.size() == sourcesToRead.size();

            // Record new readings into matrix
            StoreSliceSet sliceSet = new StoreSliceSet(readGroup.get(storeIndex), dataSlices);
            dmMatrix.addStoreSliceSet(sliceSet, OperationScorecard.Status.EXISTING);
         }

         catch (InterruptedException e)
         {
            throw new RuntimeException("Slice store controller was interrupted", e);
         }
         catch (ExecutionException e)
         {
            assert (e.getCause() instanceof FutureExceptionWrapper);
            FutureExceptionWrapper fromPast = (FutureExceptionWrapper) e.getCause();
            SliceStore store = fromPast.getOriginator();
            _logger.error(fromPast.getCause().getMessage() + " from " + store.getIdentification());

            if (fromPast.getCause() instanceof IllegalSourceNameException)
            {
               throw new ControllerIllegalSourceNameException("Invalid source name",
                     fromPast.getCause());
            }
            // report the first exception as ultimate exception
            if (this.cumulativeFailureReason == null)
            {
               this.cumulativeFailureReason = new StringBuffer();
            }
            this.cumulativeFailureReason.append("{" + store.getIdentification() + ":"
                  + fromPast.getCause().getMessage() + "}");
         }
         storeIndex++;
      }
   }

   /**
    * Order read according to predefined algorithm
    * 
    * @param numberOfSlices
    * @return
    */
   private int[] orderSlices(int numberOfSlices)
   {
      int[] orderArray = new int[numberOfSlices];
      if (this.order == Order.NORMAL)
      {
         for (int i = 0; i < numberOfSlices; i++)
         {
            orderArray[i] = i;
         }
      }
      else if (this.order == Order.REVERSE)
      {
         for (int i = 0; i < numberOfSlices; i++)
         {
            orderArray[i] = numberOfSlices - i - 1;
         }
      }
      else
      {
         Random r = new Random();
         boolean[] setFlag = new boolean[numberOfSlices];
         Arrays.fill(setFlag, false);
         Arrays.fill(orderArray, -1);

         // I suppose a better standard algorithm that creates a random array of integers
         // without repetition could be used
         for (int i = 0; i < numberOfSlices; i++)
         {
            int next = r.nextInt(numberOfSlices);
            for (int j = next; j < numberOfSlices; j++)
            {
               if (setFlag[j] == false)
               {
                  orderArray[i] = next;
                  setFlag[j] = true;
                  break;
               }
            }
            if (orderArray[i] != -1)
            {
               for (int j = 0; j < next; j++)
               {
                  if (setFlag[j] == false)
                  {
                     orderArray[i] = next;
                     setFlag[j] = true;
                     break;
                  }
               }
            }
            assert orderArray[i] != -1;
         }
      }
      return orderArray;
   }

   public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
   {
      // TODO: Implement this method
      throw new NotImplementedException("addCorruptedSliceObserver is not implemented");
   }

}
