//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 12, 2007
//---------------------

package org.cleversafe.layer.grid;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.util.ThreadlessExecutor;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.TypeAggregator;

/**
 * Grid transaction composed of multiple SliceStoreTransactions
 */
public class GridTransaction
{
   private static Logger _logger = Logger.getLogger(GridTransaction.class);

   // Fake transaction id, should be less than any real transaction id
   static public final long NOT_FOUND_TRANSACTION_ID = Long.MIN_VALUE;

   // Transaction id
   private long transactionId;

   // SliceStoreTransactions in this grid transaction. Any transaction in this list should have
   // already been started
   private ConcurrentMap<SliceStore, SliceStoreTransaction> sliceStoreTransactions;

   // Executor to use when performing commits and rollbacks
   private ExecutorService executor;

   /**
    * Verify the success of a transaction operation given any errors that have occurred
    */
   public static interface OperationVerifier
   {
      /**
       * Determine success or failure for a transaction operation
       * @param errors Collection of errors that occur during the operation
       * @return Return true iff the transaction operation should be considered a success in light of the 
       *         provided errors
       */
      public boolean verifyOperation(Collection<Tuple2<SliceStore, Exception>> errors);
   }

   /**
    * Rejects operation if any errors occur
    */
   public static class StrictOperationVerifier implements OperationVerifier
   {
      public boolean verifyOperation(Collection<Tuple2<SliceStore, Exception>> errors)
      {
         return (errors.size() == 0);
      }
   }

   /**
    * Create a new, empty, grid transaction. SliceStoreTransactions should be added later.  
    * All operations will be performed in the calling thread
    * 
    * @param transactionId
    */
   public GridTransaction(long transactionId)
   {
      this(transactionId, new ThreadlessExecutor());
   }

   /**
    * Create a new, empty, grid transaction. SliceStoreTransactions should be added later.  
    * The provided executor will be used for all operations.
    * 
    * @param transactionId
    */
   public GridTransaction(long transactionId, ExecutorService executor)
   {
      this.transactionId = transactionId;
      this.sliceStoreTransactions = new ConcurrentHashMap<SliceStore, SliceStoreTransaction>();
      this.executor = executor;
   }

   /**
    * Add a new SliceStoreTransaction to this grid transaction
    * 
    * @param tx
    */
   public void addSliceStoreTransaction(SliceStoreTransaction tx)
   {
      this.sliceStoreTransactions.put(tx.getSliceStore(), tx);
   }

   /**
    * Get all SliceStoreTransactions in this grid transaction
    * 
    * @return
    */
   public Collection<SliceStoreTransaction> getSliceStoreTransactions()
   {
      return this.sliceStoreTransactions.values();
   }

   /**
    * Get this transaction's id
    * 
    * @return
    */
   public long getId()
   {
      return this.transactionId;
   }

   /**
    * Find a SliceStoreTransaction in this grid transaction by SliceStore
    * 
    * @param store
    * @return
    */
   public SliceStoreTransaction find(SliceStore store)
   {
      return this.sliceStoreTransactions.get(store);
   }

   /**
    * Commit grid transaction with strict success verification
    * @throws ControllerStoresNotFoundException 
    * @throws ControllerIOException 
    * @throws ControllerTransactionException 
    */
   public void commit() throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      this.commit(new StrictOperationVerifier());
   }

   /**
    * Commit grid transaction
    * 
    * @throws ControllerTransactionException
    * @throws ControllerInvalidTransactionStateException
    * @throws ControllerIOException
    * @throws ControllerStoresNotFoundException
    */
   public void commit(OperationVerifier verifier) throws ControllerTransactionException,
         ControllerIOException, ControllerStoresNotFoundException
   {
      // Commit functor
      SliceStoreTransactionFunctor functor = new SliceStoreTransactionFunctor()
      {
         public void call(SliceStore store, SliceStoreTransaction transaction)
               throws SliceStoreLayerException
         {
            store.commitTransaction(transaction);
         }

         public String toString()
         {
            return "Unable to commit grid transaction";
         }
      };

      performSliceStoreTransactionOperation(functor, verifier, "commit");
   }

   /**
    * Rollback grid transaction with strict success verification
    * @throws ControllerStoresNotFoundException 
    * @throws ControllerIOException 
    * @throws ControllerTransactionException 
    */
   public void rollback() throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      this.rollback(new StrictOperationVerifier());
   }

   /**
    * Rollback grid transaction
    * 
    * @throws ControllerTransactionException
    * @throws ControllerInvalidTransactionStateException
    * @throws ControllerIOException
    * @throws ControllerStoresNotFoundException
    */
   public void rollback(OperationVerifier verifier) throws ControllerTransactionException,
         ControllerIOException, ControllerStoresNotFoundException
   {
      // Rollback functor
      SliceStoreTransactionFunctor functor = new SliceStoreTransactionFunctor()
      {
         public void call(SliceStore store, SliceStoreTransaction transaction)
               throws SliceStoreLayerException
         {
            store.rollbackTransaction(transaction);
         }

         public String toString()
         {
            return "Unable to rollback grid transaction";
         }
      };

      performSliceStoreTransactionOperation(functor, verifier, "rollback");
   }

   /**
    * Helper operation for SliceStore transaction operations
    */
   private static interface SliceStoreTransactionFunctor
   {
      public void call(SliceStore store, SliceStoreTransaction transaction)
            throws SliceStoreLayerException;

      public String toString();
   }

   /**
    * Helper function for marshaling of exceptions in SliceStore transaction operations
    * 
    * @param functor
    * @param verifier
    * @throws ControllerTransactionException
    * @throws ControllerInvalidTransactionStateException
    * @throws ControllerIOException
    * @throws ControllerStoresNotFoundException
    */
   private void performSliceStoreTransactionOperation(
         final SliceStoreTransactionFunctor functor,
         OperationVerifier verifier,
         String operationName) throws ControllerTransactionException, ControllerIOException,
         ControllerStoresNotFoundException
   {
      TypeAggregator<SliceStoreLayerException> errorHistogram =
            new TypeAggregator<SliceStoreLayerException>();
      List<Tuple2<SliceStore, Exception>> allErrors =
            new LinkedList<Tuple2<SliceStore, Exception>>();

      long beginTime = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace(String.format("GridTransaction.%s(tx=%d): started", operationName,
               this.transactionId));

      // Begin tasks
      List<Tuple2<Future<Void>, SliceStore>> tasks =
            new ArrayList<Tuple2<Future<Void>, SliceStore>>(this.sliceStoreTransactions.size());
      for (Map.Entry<SliceStore, SliceStoreTransaction> entry : this.sliceStoreTransactions.entrySet())
      {
         final SliceStore store = entry.getKey();
         final SliceStoreTransaction transaction = entry.getValue();

         Callable<Void> task = new Callable<Void>()
         {
            public Void call() throws SliceStoreLayerException
            {
               functor.call(store, transaction);
               return null;
            }
         };

         Future<Void> taskFuture = this.executor.submit(task);
         tasks.add(new Tuple2<Future<Void>, SliceStore>(taskFuture, store));
      }

      // Await task completion, collect errors
      for (Tuple2<Future<Void>, SliceStore> task : tasks)
      {
         Future<Void> future = task.getFirst();
         SliceStore store = task.getSecond();

         try
         {
            future.get();
         }
         catch (InterruptedException ex)
         {
            _logger.warn("GridTransaction operation was interrupted", ex);
            throw new ControllerTransactionException("GridTransaction operation was interrupted",
                  ex);
         }
         catch (ExecutionException execEx)
         {
            SliceStoreLayerException ssEx = (SliceStoreLayerException) execEx.getCause();

            _logger.debug("Transaction " + operationName + " error '" + ssEx + "' for store: "
                  + store);
            errorHistogram.add(ssEx);
            allErrors.add(new Tuple2<SliceStore, Exception>(store, ssEx));
         }
      }

      if (_logger.isTraceEnabled())
         _logger.trace(String.format("GridTransaction.%s(tx=%d): successful - %dms", operationName,
               this.transactionId, System.currentTimeMillis() - beginTime));

      // Check for successful completion
      if ((errorHistogram.size() > 0) && !verifier.verifyOperation(allErrors))
      {
         Map.Entry<Class<? extends SliceStoreLayerException>, Tuple2<Integer, SliceStoreLayerException>> entry =
               errorHistogram.getMaximum();
         Class<? extends SliceStoreLayerException> c = entry.getKey();
         String msg =
               String.format("GridTransaction %s error for txid=%d: ", operationName,
                     this.transactionId);

         // Map SliceStore transaction exceptions to Grid transaction exceptions
         if (c == SliceStoreTransactionException.class)
         {
            ControllerTransactionException ex =
                  new ControllerTransactionException(functor.toString(),
                        entry.getValue().getSecond());

            msg += "Transaction error";
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);

            throw ex;
         }
         else if (c == SliceStoreIOException.class)
         {
            ControllerIOException ex =
                  new ControllerIOException(functor.toString(), entry.getValue().getSecond());

            msg += "IO error";
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);

            throw ex;
         }
         else if (c == SliceStoreNotFoundException.class)
         {
            ControllerStoresNotFoundException ex =
                  new ControllerStoresNotFoundException(functor.toString(),
                        entry.getValue().getSecond());

            msg += "SliceStores not found";
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);

            throw ex;
         }
         else
         {
            RuntimeException ex =
                  new RuntimeException("Unknown SliceStoreLayerException: " + c,
                        entry.getValue().getSecond());

            msg += "Unknown error - " + c.toString();
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);

            throw ex;
         }
      }
   }
}
