//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.layer.grid;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.simplecontroller.SimpleDeleteControllerFactory;
import org.cleversafe.vault.Vault;

/**
 * Grid controller is a container for pluggable operational controllers, 
 * such as: 
 * - write 
 * - read -
 * - remove 
 * - list 
 * - exist 
 * Objects responsible for those operations would be normally be instantiated 
 * from a configuration.
 * Different combinations of operations controllers could be used but 
 * they should be be compatible. For   example, if exist() returns true, 
 * then next read operations should succeed granted that a source was not removed
 */
public final class GridController
{
   @SuppressWarnings("unused")
   private static Logger _logger = Logger.getLogger(GridController.class);
   private Vault vault = null;

   private WriteControllerFactory writeFactory = null;
   private ReadControllerFactory readFactory = null;
   private DeleteControllerFactory deleteFactory = null;

   /**
    * Setter to be used by configuration framework
    * 
    * @param vault
    */
   public GridController(Vault vault, WriteControllerFactory wcf, ReadControllerFactory rcf)
   {
      this.vault = vault;
      this.writeFactory = wcf;
      this.readFactory = rcf;
      this.deleteFactory = new SimpleDeleteControllerFactory();
   }

   public GridController(
         Vault vault,
         WriteControllerFactory wcf,
         ReadControllerFactory rcf,
         DeleteControllerFactory dcf)
   {
      this.vault = vault;
      this.writeFactory = wcf;
      this.readFactory = rcf;
      this.deleteFactory = dcf;
   }

   /**
    * 
    * @return
    */
   public Vault getVault()
   {
      return this.vault;
   }

   /**
    * Associated write controller, creates an object capable of writing data to the grid
    * 
    * @return
    * @throws ControllerException 
    */
   public WriteController getWriteController() throws ControllerException
   {
      return this.writeFactory.getInstance(this.vault);
   }

   /**
    * Associated read controller, creates an object capable of reading data to the grid
    * 
    * @return
    */
   public ReadController getReadController() throws ControllerException
   {
      return this.readFactory.getInstance(this.vault);
   }

   public DeleteController getDeleteController() throws ControllerException
   {
      // TODO Need to convert to Factory based instantiation.

      return this.deleteFactory.getInstance(this.vault);
   }

   /**
    * Release a WriteController
    * @param wc
    */
   public void release(WriteController wc)
   {
      this.writeFactory.release(wc);
   }

   /**
    * Release a ReadController
    * @param rc
    */
   public void release(ReadController rc)
   {
      this.readFactory.release(rc);
   }

   /**
    * Release a DeleteController
    * @param dc
    */
   public void release(DeleteController dc)
   {
      this.deleteFactory.release(dc);
   }
}
