//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 23, 2007
//---------------------

package org.cleversafe.layer.grid;

import java.util.Arrays;

import org.apache.log4j.Logger;
import org.cleversafe.serialization.GridSerializable;

/**
 * A named and located slice of data
 */
public class DataSlice implements GridSerializable
{
   private static Logger _logger = Logger.getLogger(DataSlice.class);

   // Datasource or slice name
   protected SliceName sliceName = null;

   // Grid transaction id
   protected long transactionId = GridTransaction.NOT_FOUND_TRANSACTION_ID;

   // Datasource or slice data
   protected byte[] data = null;

   // Used only for ASN.1 deserialization
   public DataSlice()
   {
   }

   /**
    * Construct a named and located slice from its data
    * 
    * @param name
    * @param store
    * @param transactionId
    * @param data
    */
   public DataSlice(SliceName sliceName, long transactionId, byte[] data)
   {
      this.sliceName = sliceName;
      this.transactionId = transactionId;
      this.data = data;
   }

   /**
    * Construct a named dataslice with null data.
    * 
    * @param sliceName
    */
   public DataSlice(SliceName sliceName)
   {
      this(sliceName, GridTransaction.NOT_FOUND_TRANSACTION_ID, null);
   }

   public SliceName getSliceName()
   {
      return this.sliceName;
   }

   public long getTransactionId()
   {
      return this.transactionId;
   }

   public byte[] getData()
   {
      return this.data;
   }

   public void setSliceName(SliceName sliceName)
   {
      this.sliceName = sliceName;
   }

   public void setTransactionId(long transactionId)
   {
      this.transactionId = transactionId;
   }

   public void setData(byte[] data)
   {
      this.data = data;
   }

   @Override
   public boolean equals(Object object)
   {
      if (object instanceof DataSlice)
      {
         DataSlice that = (DataSlice) object;

         if ((this.sliceName == null && that.sliceName != null)
               || (that.sliceName == null && this.sliceName != null))
         {
            return false;
         }

         if ((this.data == null && that.data != null) || (that.data == null && this.data != null))
         {
            return false;
         }

         if ((this.sliceName == null || this.sliceName.equals(that.sliceName))
               && this.transactionId == that.transactionId
               && (this.data == null || Arrays.equals(this.data, that.data)))
         {
            return true;
         }
         else
         {
            return false;
         }
      }
      else
      {
         return false;
      }
   }

   @Override
   public String toString()
   {
      StringBuffer buffer = new StringBuffer();
      buffer.append(this.sliceName).append("(").append(this.transactionId).append(") '");

      // Only in trace mode we would be so verbose
      // It is probably a security hole and we should get rid of it all together
      if (_logger.isTraceEnabled())
      {
         int dataLength;
         if (data == null)
         {
            dataLength = 0;
         }
         else
         {
            dataLength = data.length;
         }

         int part = Math.min(10, dataLength);

         // Be nice build hex and ascii forms for debugging
         StringBuffer bc = new StringBuffer();
         StringBuffer bh = new StringBuffer();

         for (int i = 0; i < part; i++)
         {
            // All printable in the original form
            if (this.data[i] >= ' ' && this.data[i] < '~')
            {
               bc.append((char) this.data[i]);
            }
            else
            {
               bc.append('.');
            }
            bh.append(" ").append(Integer.toHexString(0xff & this.data[i]));
         }
         buffer.append(bc.toString()).append('|').append(bh.toString());
         if (dataLength > 10)
         {
            buffer.append("...");
         }
         buffer.append('\'');
      }
      return buffer.toString();
   }
}
