//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: abaptist
//
// Date: Sep 27, 2007
//---------------------

package org.cleversafe.layer.communication.policy;

import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.log4j.Logger;
import org.cleversafe.util.NamedThreadFactory;

/**
 * A TimeQueue is a generic class that accepts events to fire at a fixed interval in the future.
 * Events are added to the end of the queue and if they make it to the front of the queue they fire.
 * The only method that is regularly called is the "touch" method. This method either adds an event
 * if it is not already on the queue, or if it is already on the queue, the event is moved to the
 * end of the queue.
 * 
 * Events that are added to the queue need to implement the TimeQueueEvent interface which has the
 * single method "event()".
 * 
 * @see TimeQueueEvent
 * 
 * @author Andrew Baptist <abaptist@cleversafe.com>
 * @version $Id$
 */
public class TimeQueue extends Thread
{
   private class TimeQueueEventWrapper implements Runnable
   {
      private TimeQueueEvent event;
      private long eventExpirationTime;

      private TimeQueueEventWrapper(TimeQueueEvent event)
      {
         this.event = event;
         eventExpirationTime = System.currentTimeMillis() + _expirationInterval;
      }

      public boolean equals(Object other)
      {
         return event == ((TimeQueueEventWrapper) other).event;
      }

      public int hashCode()
      {
         return event.hashCode();
      }

      public void run()
      {
         if (_logger.isTraceEnabled())
         {
            _logger.trace("Running an event of type " + event.getClass().toString());
         }
         event.event();
      }
   }

   private static Logger _logger = Logger.getLogger(TimeQueue.class);
   protected static ExecutorService eventExecutor =
         Executors.newCachedThreadPool(new NamedThreadFactory("Time Queue Event", true));

   private Set<TimeQueueEventWrapper> jobList;
   private volatile boolean _shutdown = false;
   private int _expirationInterval = -1;
   private int _initialSize = 20;

   /** Create the time queue with a specific expiration time */
   public TimeQueue(int expirationInterval, String name)
   {
      _logger.trace("Starting a time queue with interval " + expirationInterval);
      this._expirationInterval = expirationInterval;
      jobList = new LinkedHashSet<TimeQueueEventWrapper>(_initialSize);
      this.setDaemon(true);
      this.setName(name);
      this.start();
   }

   public void initialize()
   {
      jobList = new LinkedHashSet<TimeQueueEventWrapper>(_initialSize);
      this.start();
      // Executors.defaultThreadFactory().newThread(this).start();
   }

   /**
    * Wait in a loop for timeouts to occur, when they do, notify the listening process and wait for
    * the next event. Loop continues until _shutdown is called or an InterruptedException occurs
    */
   public void run()
   {
      _logger.debug("Starting time queue");
      while (!_shutdown)
      {
         TimeQueueEventWrapper nextTask = null;

         synchronized (this)
         {
            try
            {
               if (jobList.isEmpty())
               {
                  _logger.trace("wait until something is added to the queue");
                  wait();
               }
               TimeQueueEventWrapper potentialTask = jobList.iterator().next();

               // find out diff between when first task is supposed to run and current time
               long timeDiff = potentialTask.eventExpirationTime - System.currentTimeMillis();
               if (timeDiff <= 0)
               {
                  jobList.remove(potentialTask);
                  nextTask = potentialTask;
               }
               else
               {
                  _logger.trace("wait until this event is going to time out to check again "
                        + timeDiff);
                  wait(timeDiff);
               }
            }
            catch (InterruptedException e)
            {
               _logger.warn("Caught interrupted exception - shutting down");
               _shutdown = true;
            }
         }
         // run this outside the synchronized block
         if (nextTask != null)
         {
            if (_logger.isTraceEnabled())
               _logger.trace("Running event " + nextTask.event.getClass());

            eventExecutor.execute(nextTask);
         }
      }
      _logger.debug("Shut down");
   }

   /**
    * Shut down this time queue. Implemented by interrupting this thread which will then notice that
    * _Shutdown is set and stop
    */
   public synchronized void shutdown()
   {
      _logger.debug("Shutting down");
      _shutdown = true;
      this.interrupt();
   }

   /**
    * 
    * @param element -
    *           element to add to the queue (or just reset time on if it already exists)
    */
   public synchronized void touch(TimeQueueEvent element)
   {
      if (_logger.isTraceEnabled())
         _logger.trace("Touch " + element.getClass());
      boolean needNotify = false;
      if (jobList.isEmpty())
      {
         needNotify = true;
      }

      TimeQueueEventWrapper wrapper = new TimeQueueEventWrapper(element);
      // remove and re-add so that it moves to the front of the queue.
      jobList.remove(wrapper);
      jobList.add(wrapper);
      if (needNotify)
      {
         this.notify();
      }
   }

   /**
    * Set the initial size of the map, if not specified defaults to 20
    * 
    * @param initialSize
    */
   public void setInitialSize(int initialSize)
   {
      this._initialSize = initialSize;
   }
}
