//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: abaptist
//
// Date: Sep 27, 2007
//---------------------

package org.cleversafe.layer.communication.policy;

import org.apache.log4j.Logger;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.ConnectorManager;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;

/**
 * A SmartConnectorManager is responsible for managing the underlying Connector that creates it The
 * manager has the following primary responsibilities
 * <ul>
 * <li> When a connection has been idle for too long it</li>
 * disconnects it
 * <li> When a connection has not been used, send a ping message to keep it alive </li>
 * <li> When a connection has been reporting network errors, delay using the connection for a while
 * to allow the errors to possibly clear</li>
 * </ul>
 * 
 * 
 * @see Connector
 * 
 * @author Andrew Baptist <abaptist@cleversafe.com>
 * @version $Id$
 */
public class SmartConnectorManager implements ConnectorManager
{

   private static final Logger _logger = Logger.getLogger(SmartConnectorManager.class);

   private enum DISCONNECTION_STATE
   {
      NOT_ERROR, ERROR, ERROR_CLEARED
   }

   // whether this connection is currently in an error state
   private DISCONNECTION_STATE disconnectState = DISCONNECTION_STATE.NOT_ERROR;

   // connection we are monitoring
   private Connector _connection;

   // whether to do a lazy initialization (don't open connection until first needed)
   private boolean _isLazy = true;

   /**
    * A IdleNotify event disconnects the connection when it fires.
    */
   private class IdleNotify implements TimeQueueEvent
   {
      public void event()
      {
         synchronized (SmartConnectorManager.this)
         {
            if (SmartConnectorManager.this.disconnectState != DISCONNECTION_STATE.ERROR)
            {
               _logger.trace("closing idle connection: " + _connection);
               try
               {
                  // NOT_ERROR state if going idle 
                  SmartConnectorManager.this.disconnectState = DISCONNECTION_STATE.NOT_ERROR;
                  if (_connection.isConnected())
                  {
                     _connection.disconnect();
                  }
               }
               catch (final CommunicationException e)
               {
                  _logger.warn("problem while disconnecting: " + _connection);
               }
            }
         }
      }
   }

   private IdleNotify idleNotify = new IdleNotify();

   /**
    * A PingNotify event instructs the Connector to send a ping when the event fires.
    */
   private class SuccessNotify implements TimeQueueEvent
   {
      private static final int UNRESPONSIVE_DISCONNECT_TIME = 30 * 1000; // 30 seconds
      private long waitingForExchangeSince = 0;

      public synchronized void event()
      {
         SmartConnectorManagerQueue.getSuccessQueue().touch(this);

         if (!_connection.isConnected())
         {
            waitingForExchangeSince = 0;
            return;
         }

         if (waitingForExchangeSince != 0
               && System.currentTimeMillis() - waitingForExchangeSince > UNRESPONSIVE_DISCONNECT_TIME)
         {
            _logger.info("Disconnecting due to unresponsiveness " + _connection);
            waitingForExchangeSince = 0;
            onError();
         }
         else if ((waitingForExchangeSince == 0) && (_connection.getNumOutstandingExchanges() > 0))
         {
            waitingForExchangeSince = System.currentTimeMillis();
         }
      }

      public synchronized void onSuccess()
      {
         waitingForExchangeSince = 0;
      }
   }

   private SuccessNotify successNotify = new SuccessNotify();

   /**
    * A PingNotify event instructs the Connector to send a ping when the event fires.
    */
   private class PingNotify implements TimeQueueEvent
   {
      public void event()
      {
         if (_logger.isTraceEnabled())
            _logger.trace("sending ping: " + _connection);
         if (SmartConnectorManager.this._connection.isConnected())
         {
            try
            {
               _connection.sendPing();
               // re-add to the queue so that it fires again
               SmartConnectorManagerQueue.getPingQueue().touch(pingNotify);
            }
            catch (CommunicationException e)
            {
               _logger.error("communication error while sending ping: " + _connection, e);
               onError();
            }
         }
      }

   }
   private PingNotify pingNotify = new PingNotify();

   /** 
    * An ErrorNotify event occurs when a connection has been in "error" state for long enough and we are ready to retry it.
    */
   private class ErrorNotify implements TimeQueueEvent
   {
      public void event()
      {
         _logger.info("attempt to connect: " + SmartConnectorManager.this._connection);

         try
         {
            SmartConnectorManager.this._connection.connect();
         }
         catch (final CommunicationException ex)
         {
            String msg =
                  "i/o exception encountered while asynchronously reconnecting: "
                        + SmartConnectorManager.this._connection;
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);
         }

         // Check if connection was re-established - it may be possible that connect returned without establishing the connection
         synchronized (SmartConnectorManager.this)
         {
            if (_connection.isConnected() == true)
            {
               SmartConnectorManager.this.disconnectState = DISCONNECTION_STATE.ERROR_CLEARED;
               // Start idle an ping queues to close this connection in the future if necessary
               onSuccess();
            }
            else
            {
               // restart in order to reconnect in the future
               SmartConnectorManagerQueue.getErrorQueue().touch(errorNotify);
            }
         }
      }
   }

   private ErrorNotify errorNotify = new ErrorNotify();

   public void initialize()
   {
      _logger.trace("init called");
   }

   public synchronized boolean ensureConnected() throws CommunicationIOException,
         CommunicationConnectionException
   {
      if (_logger.isTraceEnabled())
         _logger.trace("ensure connection to " + this._connection);

      // we have DISCONNECTION_STATE.ERROR from a previous error - don't try to reconnect, let the TimeQueue do it later
      if (this.disconnectState == DISCONNECTION_STATE.ERROR)
      {
         _logger.trace("Not attempting to connect to  - in an error state");
         throw new CommunicationConnectionException("connection is in an error state: "
               + this._connection);
      }
      // Freshly reconnected connection, we are first to discover, let caller know by returning true
      else if (this.disconnectState == DISCONNECTION_STATE.ERROR_CLEARED)
      {
         this.disconnectState = DISCONNECTION_STATE.NOT_ERROR;
         _logger.debug("Connection was asynchronously restored on " + this._connection);
         return true;
      }
      // Check with transport whether we really are connected 
      else if (_connection.isConnected())
      {
         if (_logger.isTraceEnabled())
            _logger.trace("already connected: " + this._connection);
         return false;
      }

      // State is NOT_ERROR but we are not connected
      else
      {
         _logger.info("attempt to connect: " + this._connection);

         try
         {
            this._connection.connect();
         }
         catch (final CommunicationIOException ex)
         {
            String msg = "i/o exception encountered while reconnecting: " + this._connection;
            if (_logger.isDebugEnabled())
               _logger.warn(msg, ex);
            else
               _logger.warn(msg);
         }
         catch (final CommunicationConnectionException ex)
         {
            _logger.warn("Connection exception encountered while reconnecting: " + this._connection);
         }

         // Check if connection was re-established - it may be possible that connect returned without establishing the connection
         if (_connection.isConnected() == true)
         {
            onSuccess();
            _logger.info("re-established connection: " + this._connection);
            return true;
         }
         else
         {
            onError();
            throw new CommunicationConnectionException("unable to connect to remote host: "
                  + this._connection);
         }
      }
   }

   /** This is only called once from the initialization of the connector */
   public void onConnectorInit()
   {
      assert _connection != null;

      // Start continuous liveness monitoring
      SmartConnectorManagerQueue.getSuccessQueue().touch(successNotify);

      if (!_isLazy)
      {
         _logger.trace("non-Lazy initialization");
         try
         {
            ensureConnected();
         }
         catch (final CommunicationException e)
         {
            // nothing to do - connection is not initialized, and error already logged
            _logger.debug("Caught exception while trying to initialize");
         }
      }
   }

   /**
    * Mark this as being in an error state so that it will be retried and disconnect the connection
    */
   public synchronized void onError()
   {
      _logger.info("marking connection as error: " + this._connection);
      SmartConnectorManagerQueue.getErrorQueue().touch(errorNotify);
      this.disconnectState = DISCONNECTION_STATE.ERROR;

      try
      {
         if (_connection.isConnected())
         {
            _connection.disconnect();
         }
      }
      catch (CommunicationIOException e)
      {
         _logger.warn("i/o error encountered while disconnecting connection: " + this._connection,
               e);
      }
      catch (CommunicationInterruptedException e)
      {
         _logger.warn("communication interrupted while disconnecting: " + this._connection, e);
      }
   }

   /**
    * On a successful message, mark the idle and ping queue to prevent pings from going out and also
    * to prevent the connection from being disconnected. 
    * Also let the succes queue know that we have a success so it will restart the timer if necessary
    */
   public void onSuccess()
   {
      SmartConnectorManagerQueue.getIdleQueue().touch(idleNotify);
      SmartConnectorManagerQueue.getPingQueue().touch(pingNotify);
      successNotify.onSuccess();
   }

   /** This should only be called before initialization is done */
   public void setConnection(Connector connection)
   {
      this._connection = connection;
   }

   public void setLazy(boolean _isLazy)
   {
      _logger.trace("Call set Lazy" + _isLazy);
      this._isLazy = _isLazy;
   }

   public boolean getLazy()
   {
      return _isLazy;
   }

   /** Notify that this connection is shutting down */
   public void shutdown()
   {
      SmartConnectorManagerQueue.shutdown();
   }

   public boolean isErrorState()
   {
      return this.disconnectState == DISCONNECTION_STATE.ERROR;
   }
}
