//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: jquigley
//
// Date: Jun 7, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Set;

import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.IoSession;
import org.apache.mina.filter.codec.ProtocolEncoderOutput;
import org.apache.mina.filter.codec.demux.MessageEncoder;
import org.cleversafe.layer.protocol.GridProtocolHeaderCodec;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.protocol.VersionedProtocolMessage;
import org.cleversafe.serialization.GridProtocolMessageFactory;
import org.cleversafe.serialization.ProtocolMessageFactory;

public abstract class ProtocolMessageEncoder implements MessageEncoder
{
   //private static Logger _logger = Logger.getLogger(ProtocolMessageEncoder.class);

   private ProtocolMessageFactory protocolMessageFactory = getProtocolMessageFactory();

   /**
    * Wrapper that allows us to write into a Mina pool-allocated buffer
    * using standard stream methods. Does not grow beyond the initial size.
    */
   private static class MinaByteBufferOutputStream extends OutputStream
   {
      private int size;
      private ByteBuffer minaBuffer;

      public MinaByteBufferOutputStream(int size)
      {
         super();

         this.size = size;
         this.minaBuffer = ByteBuffer.allocate(this.size);
         this.minaBuffer.clear();
      }

      /**
       * Flush and return the underlying ByteBuffer.  Can only be called once.
       */
      public ByteBuffer getByteBuffer()
      {
         return this.minaBuffer.flip();
      }

      public void position(int pos)
      {
         this.minaBuffer.position(pos);
      }

      public int position()
      {
         return this.minaBuffer.position();
      }

      @Override
      public void write(int b) throws IOException
      {
         this.minaBuffer.put((byte) b);
      }

      @Override
      public void write(byte[] b, int off, int len) throws IOException
      {
         this.minaBuffer.put(b, off, len);
      }

      @Override
      public void write(byte[] b) throws IOException
      {
         this.minaBuffer.put(b);
      }
   }

   public void encode(IoSession session, Object message, ProtocolEncoderOutput out)
         throws Exception
   {
      SequencedProtocolMessage pm = (SequencedProtocolMessage) message;
      int messageLength = protocolMessageFactory.getMaxSerializedSize(pm);

      // Allocate single buffer for serializing the header and message
      MinaByteBufferOutputStream bos =
            new MinaByteBufferOutputStream(GridProtocolHeaderCodec.LENGTH + messageLength);
      DataOutputStream outputStream = new DataOutputStream(bos);

      // Get protocol version number
      short protocolVersion = -1;
      if (pm instanceof VersionedProtocolMessage)
      {
         ((VersionedProtocolMessage) pm).setProtocolVersion(GridProtocolMessageFactory.getGridProtocolVersion());
         protocolVersion = ((VersionedProtocolMessage) pm).getProtocolVersion();
      }
      else
      {
         protocolVersion = GridProtocolHeaderCodec.PROTOCOL_VERSION;
      }

      // Serialize message first to get its true length
      bos.position(GridProtocolHeaderCodec.LENGTH);
      protocolMessageFactory.serialize(outputStream, pm);
      int endPosition = bos.position();
      
      // Serialize the Protocol Header
      bos.position(0);
      GridProtocolHeaderCodec header =
            new GridProtocolHeaderCodec(protocolVersion, pm.getSequenceNumber(),
                  pm.getOperation().getOperationCode(), endPosition
                        - GridProtocolHeaderCodec.LENGTH);
      header.serialize(outputStream);
      bos.position(endPosition);

      out.write(bos.getByteBuffer());
   }

   public void dispose(IoSession arg0) throws Exception
   {
   }

   // suppressing warnings caused by mina's MessageEncoder interface
   @SuppressWarnings("unchecked")
   public Set<Class<?>> getMessageTypes()
   {
      Set<Class<?>> msgTypes = new HashSet<Class<?>>();
      msgTypes.addAll(protocolMessageFactory.getMessageTypes());
      return msgTypes;
   }

   public abstract ProtocolMessageFactory getProtocolMessageFactory();

}
