//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: jquigley
//
// Date: Jun 7, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import java.io.DataInputStream;

import org.apache.log4j.Logger;
import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.IoSession;
import org.apache.mina.filter.codec.ProtocolDecoderOutput;
import org.apache.mina.filter.codec.demux.MessageDecoder;
import org.apache.mina.filter.codec.demux.MessageDecoderResult;
import org.cleversafe.layer.protocol.GridProtocolHeaderCodec;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.protocol.VersionedProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.HeaderFormatException;
import org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException;
import org.cleversafe.serialization.ProtocolMessageFactory;

public abstract class ProtocolMessageDecoder implements MessageDecoder
{
   private static final Logger _logger = Logger.getLogger(ProtocolMessageDecoder.class);

   private GridProtocolHeaderCodec header = null;

   private boolean decodeHeader = true;

   private ProtocolMessageFactory protocolMessageFactory = getProtocolMessageFactory();

   private byte headerBuffer[] = new byte[GridProtocolHeaderCodec.LENGTH];

   // TODO: How do we handle NOT_OK case?

   public MessageDecoderResult decodable(IoSession session, ByteBuffer in)
   {
      MessageDecoderResult retVal = MessageDecoderResult.NEED_DATA;

      if (in.remaining() >= GridProtocolHeaderCodec.LENGTH)
      {
         try
         {
            if (decodeHeader)
            {
               in.get(headerBuffer);
               header = new GridProtocolHeaderCodec(headerBuffer);
               decodeHeader = false;
            }
            else
            {
               in.skip(GridProtocolHeaderCodec.LENGTH);
            }

            if (in.remaining() >= header.getPayloadLength())
            {
               decodeHeader = true;
               retVal = MessageDecoderResult.OK;
            }
         }
         catch (HeaderFormatException e)
         {
            decodeHeader = true;
            retVal = MessageDecoderResult.NOT_OK;
         }
      }

      return retVal;
   }

   public MessageDecoderResult decode(IoSession session, ByteBuffer in, ProtocolDecoderOutput out)
         throws Exception
   {
      assert header != null : "The header has not been read";

      in.skip(GridProtocolHeaderCodec.LENGTH);

      // Create a data input stream from the ByteBuffer
      ByteBuffer dupe = in.duplicate().asReadOnlyBuffer();
      dupe.limit(dupe.position() + header.getPayloadLength());
      assert dupe.remaining() == header.getPayloadLength();
      DataInputStream payload = new DataInputStream(dupe.asInputStream());

      in.skip(header.getPayloadLength());

      try
      {
         ProtocolMessage message =
               protocolMessageFactory.deserialize(header.getOperationCode(), payload);

         if (message instanceof SequencedProtocolMessage)
         {
            ((SequencedProtocolMessage) message).setSequenceNumber(header.getMessageID());
         }
         if (message instanceof VersionedProtocolMessage)
         {
            ((VersionedProtocolMessage) message).setProtocolVersion(header.getProtocolVersion());
            ((VersionedProtocolMessage) message).setProtocolIdentifier(header.getProtocolID());
         }
         out.write(message);
      }
      catch (final ProtocolDeserializationException ex)
      {
         // TODO: Deal with this exception
         _logger.error("Error deserializing message", ex);
         throw ex;
      }

      // TODO: Is this necessary
      header = null;
      return MessageDecoderResult.OK;
   }

   public void finishDecode(IoSession session, ProtocolDecoderOutput out) throws Exception
   {
   }

   public abstract ProtocolMessageFactory getProtocolMessageFactory();
}
