//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Sep 13, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.DefaultIoFilterChainBuilder;
import org.apache.mina.common.IoAcceptor;
import org.apache.mina.common.PooledByteBufferAllocator;
import org.apache.mina.common.ThreadModel;
import org.apache.mina.filter.LoggingFilter;
import org.apache.mina.filter.SSLFilter;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.transport.socket.nio.SocketAcceptor;
import org.apache.mina.transport.socket.nio.SocketAcceptorConfig;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.PropertiesProvider;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.layer.communication.Acceptor;
import org.cleversafe.layer.communication.network.mina.ssl.SSLContextFactory;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.ServerApplication;
import org.cleversafe.server.exceptions.ServerIOException;
import org.cleversafe.server.exceptions.ServerSecurityException;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.util.NamedThreadFactory;
import org.cleversafe.util.ThreadlessExecutor;

public class MinaAcceptor implements Acceptor
{
   private static Logger _logger = Logger.getLogger(MinaAcceptor.class);

   public static final String THREAD_POOL_NAME = "Request Handler";

   // Thread pool for all network tasks
   private static ExecutorService executor;

   // Suggested TCP window size in bytes.  To fully utilize, the complimentary client socket 
   // should set a window at least as large as this
   static private int TCP_BUFFER_SIZE = 512 * 1024;
   static private boolean NAGEL_PARAMETER = true;

   private int listenPort;
   private String listenHost;
   private boolean enableSSL;
   private boolean enableLogging;
   private boolean doIdleDisconnect;
   private int connectionTimeout;
   private int maxConnections;
   private int listenThreads;
   private String sslCertificate;
   private String sslCertificatePassword;
   private ServerApplication serverApplication;
   private ProtocolMessageFactory protocolMessageFactory;

   private IoAcceptor acceptor;

   // Executor responsible for creating up to MAX_CONNECTIONS + 1 threads to handle incoming
   // connections
   // NOTE: It is safest to make this a boundless cached executor. Mina regulates the number of
   // threads created and this avoids any potential deadlocks
   protected ExecutorService connectionExecutor = Executors.newCachedThreadPool();

   protected volatile boolean isRunning;

   // Initialize thread pool from configuration properties
   // NOTE: This a temporary measure -- eventually concurrency will be handled
   //       by the ServerApplication
   static
   {
      ByteBuffer.setUseDirectBuffers(false);
      ByteBuffer.setAllocator(new PooledByteBufferAllocator());

      try
      {
         PropertiesProvider propProvider =
               ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

         String[] numThreadsElement = {
               "org", "cleversafe", "layer", "communication", "network", "acceptor-threads"
         };

         int numThreads = 0;
         try
         {
            numThreads = propProvider.getIntValue(numThreadsElement);
         }
         catch (ConfigurationItemNotDefinedException ex)
         {
            _logger.warn("Initializing MinaAcceptor: No controller-threads property, defaulting to single-threaded");
            numThreads = 0;
         }
         if (numThreads == 0)
         {
            _logger.info("Initializing MinaAcceptor: single-threaded");
            executor = new ThreadlessExecutor();
         }
         else
         {
            _logger.info("Initializing MinaAcceptor: " + numThreads + " threads");
            executor = new BoundedThreadPoolExecutor(THREAD_POOL_NAME, numThreads);
         }

         String[] tcpBufferSizeElement = {
               "org", "cleversafe", "layer", "communication", "network", "tcp-buffer-size"
         };
         try
         {
            TCP_BUFFER_SIZE = propProvider.getIntValue(tcpBufferSizeElement) * 1024;
            _logger.info("Initializing MinaAcceptor: TCP BUFFER SIZE is " + TCP_BUFFER_SIZE / 1024
                  + "KB");

         }
         catch (ConfigurationItemNotDefinedException ex)
         {
            _logger.warn("Initializing MinaAcceptor: TCP BUFFER SIZE is not defined, use default"
                  + TCP_BUFFER_SIZE / 1024 + "KB");
         }
         String[] nagelElement = {
               "org", "cleversafe", "layer", "communication", "network", "nagel"
         };
         try
         {
            NAGEL_PARAMETER = propProvider.getBooleanValue(nagelElement);
            _logger.info("Initializing MinaAccepter: Nagel parameter is " + NAGEL_PARAMETER);
         }
         catch (ConfigurationItemNotDefinedException ex)
         {
            _logger.warn("Initializing MinaAcceptor: Nagel parameter is not defined, use default "
                  + NAGEL_PARAMETER);
         }
      }
      catch (ConfigurationException ex)
      {
         _logger.error("Failed to parse properties file");
         _logger.warn("Initializing MinaAcceptor: No controller-threads property, defaulting to single-threaded");
         executor = new ThreadlessExecutor();
      }
   }

   /**
    * Returns the configured TCP buffer size
    * @return
    */
   public static int getTcpBufferSize()
   {
      return TCP_BUFFER_SIZE;
   }

   /**
    * Returns true if nagel's algorithm is disabled
    * @return
    */
   public static boolean isTcpNoDelay()
   {
      return NAGEL_PARAMETER;
   }

   /**
    * Construct a new unbound acceptor
    */
   public MinaAcceptor()
   {
      this.listenPort = 0;
      this.listenHost = "0.0.0.0";
      this.enableSSL = false;
      this.enableLogging = false;
      this.doIdleDisconnect = false;
      this.connectionTimeout = 3600;
      this.sslCertificate = "/etc/dsgrid/ssl.cert";
      this.sslCertificatePassword = "captainbackup";
      this.listenThreads = 100;
      this.maxConnections = 500;

      this.serverApplication = null;
      this.protocolMessageFactory = null;

      this.acceptor = null;
   }

   public String getHost()
   {
      return this.listenHost;
   }

   public void setHost(String host)
   {
      this.listenHost = host;
   }

   public int getPort()
   {
      return this.listenPort;
   }

   public void setPort(int port)
   {
      this.listenPort = port;
   }

   public boolean getEnableSSL()
   {
      return this.enableSSL;
   }

   public void setEnableSSL(boolean enableSSL)
   {
      this.enableSSL = enableSSL;
   }

   public String getSSLCertificate()
   {
      return this.sslCertificate;
   }

   public void setSSLCertificate(String cert)
   {
      this.sslCertificate = cert;
   }

   public String getSSLCertificatePassword()
   {
      return this.sslCertificatePassword;
   }

   public void setSSLCertificatePassword(String password)
   {
      this.sslCertificatePassword = password;
   }

   public boolean getEnableLogging()
   {
      return this.enableLogging;
   }

   public void setEnableLogging(boolean enableLogging)
   {
      this.enableLogging = enableLogging;
   }

   public boolean getIdleDisconnect()
   {
      return this.doIdleDisconnect;
   }

   public void setIdleDisconnect(boolean doIdleDisconnect)
   {
      this.doIdleDisconnect = doIdleDisconnect;
   }

   public int getConnectionTimeout()
   {
      return this.connectionTimeout;
   }

   public void setConnectionTimeout(int connectionTimeout)
   {
      this.connectionTimeout = connectionTimeout;
   }

   public int getMaxConnections()
   {
      return this.maxConnections;
   }

   public void setMaxConnections(int maxConnections)
   {
      this.maxConnections = maxConnections;
   }

   public int getListenThreads()
   {
      return this.listenThreads;
   }

   public void setListenThreads(int listenThreads)
   {
      this.listenThreads = listenThreads;
   }

   public ServerApplication getServerApplication()
   {
      return this.serverApplication;
   }

   public void setServerApplication(ServerApplication serverApplication)
   {
      this.serverApplication = serverApplication;
   }

   public ProtocolMessageFactory getProtocolMessageFactory()
   {
      return this.protocolMessageFactory;
   }

   public void setProtocolMessageFactory(ProtocolMessageFactory protocolMessageFactory)
   {
      this.protocolMessageFactory = protocolMessageFactory;
   }

   /**
    * Begins listening for connections and will process any received requests
    * 
    * @throws ServerSecurityException
    * @throws ServerIOException
    */
   public synchronized void start() throws ServerIOException
   {
      if (this.isRunning)
      {
         throw new RuntimeException("This SliceServer is already running");
      }

      SocketAcceptorConfig cfg = new SocketAcceptorConfig();
      cfg.getSessionConfig().setReuseAddress(true);
      cfg.getSessionConfig().setTcpNoDelay(NAGEL_PARAMETER);
      cfg.setThreadModel(ThreadModel.MANUAL);

      // This setting is critical for throughput over high latency connections
      cfg.getSessionConfig().setSendBufferSize(MinaAcceptor.TCP_BUFFER_SIZE);
      cfg.getSessionConfig().setReceiveBufferSize(MinaAcceptor.TCP_BUFFER_SIZE);

      DefaultIoFilterChainBuilder filterChain = cfg.getFilterChain();

      ((ThreadPoolExecutor) this.connectionExecutor).setThreadFactory(new NamedThreadFactory(
            "SliceServer Connection <" + this.listenHost + ":" + this.listenPort + ">"));

      // Enable SSL
      if (this.enableSSL)
      {
         boolean serverMode = true;

         try
         {
            SSLFilter sslFilter =
                  new SSLFilter(SSLContextFactory.getInstance(serverMode, this.sslCertificate,
                        this.sslCertificatePassword));
            filterChain.addLast("sslFilter", sslFilter);
         }
         catch (GeneralSecurityException e)
         {
            throw new ServerIOException("Unable to construct SSL context", e);
         }

         _logger.info("SSL connections enabled");
      }

      // Enable logging
      if (this.enableLogging)
      {
         filterChain.addLast("logger", new LoggingFilter());
      }

      // protocol
      ProtocolMessageFactory messageFactory = this.protocolMessageFactory;

      ProtocolCodecFilter codecFilter =
            new ProtocolCodecFilter(new ProtocolMessageCodecFactory(
                  messageFactory.getEncoderClass(), messageFactory.getDecoderClass()));

      filterChain.addLast("codec", codecFilter);

      filterChain.addLast("threadPool", new ExecutorFilter(executor));

      this.acceptor = new SocketAcceptor(this.listenThreads, this.connectionExecutor);

      MinaAcceptorHandler handler = new MinaAcceptorHandler();
      handler.setServerApplication(this.serverApplication);
      handler.setConnectionTimeout(this.connectionTimeout);
      handler.setIdleDisconnect(this.doIdleDisconnect);
      handler.setMaxConnections(this.maxConnections);

      try
      {
         this.acceptor.bind(new InetSocketAddress(this.listenHost, this.listenPort), handler, cfg);
      }
      catch (IOException e)
      {
         throw new ServerIOException("Unable to bind server to address (" + this.listenHost + ":"
               + this.listenPort + ")", e);
      }

      this.isRunning = true;

      _logger.info("Started Slice Server [" + this.listenPort + "]");
   }

   /**
    * Shuts down executor and stops listening for new connections
    * 
    */
   public synchronized void stop()
   {
      _logger.info("Shutdown request for Slice Server [" + this.listenPort + "]");

      this.acceptor.unbindAll();
      _logger.debug("Acceptor has stopped listening and is unbound");

      try
      {
         this.connectionExecutor.shutdown();
         this.connectionExecutor.awaitTermination(5, TimeUnit.SECONDS);
      }
      catch (InterruptedException e)
      {
         _logger.warn("Problem while waiting for executor termination. Details: " + e.getMessage());
      }

      this.isRunning = false;
      _logger.info("Shutdown complete for Slice Server [" + this.listenPort + "]");
   }

   // TODO: Do we need this?
   public Response receive(ClientSession session, Request request)
   {
      return this.serverApplication.service(request, session);

   }

   public synchronized boolean isRunning()
   {
      return this.isRunning;
   }
}
