//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Aug 8, 2007
//---------------------

package org.cleversafe.layer.communication.network;

import org.apache.log4j.Logger;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.ConnectorManager;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.NoopResponse;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.Response;

/**
 * An IPNetworkConnector is a connector that operates over an IP network. This is an abstract class,
 * that implements the base functionality independent of any network library that is used.
 * 
 */
public abstract class IPNetworkConnector implements Connector
{
   private static Logger _logger = Logger.getLogger(IPNetworkConnector.class);

   private int _port = -1;
   private String _host = null;
   private int _connectTimeout = 10 * 1000; // 10 seconds
   private int _sessionTimeout = 10 * 60 * 1000; // 10 minutes
   private ConnectorManager _manager = null;

   protected IPNetworkConnector()
   {
      try
      {
         this._manager =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     ConnectorManager.class);
         this._manager.setConnection(this);
      }
      catch (ConfigurationException e)
      {
         throw new InitializationException(
               "Could not instantiate ProtocolMessageFactory from configuration", e);
      }
   }

   protected IPNetworkConnector(final String host, final int port, final ConnectorManager manager)
   {
      this._host = host;
      this._port = port;
      this._manager = manager;
      this._manager.setConnection(this);
   }

   protected IPNetworkConnector(final String host, final int port, final int connectionTimeout)
   {
      this();
      this._host = host;
      this._port = port;
      this._connectTimeout = connectionTimeout;
   }

   public void initialize()
   {
      if (this._port < 0 || this._port > 65535)
      {
         throw new InitializationException("Connection port has not been properly specified.");
      }

      if (this._host == null)
      {
         throw new InitializationException("Connection host has not been properly specified.");
      }

      if (this._manager == null)
      {
         throw new InitializationException("Connection manager is not set");
      }

      _logger.debug("Successfully initialized");
   }

   public void setConnectTimeout(final int timeout)
   {
      this._connectTimeout = timeout;
   }

   public int getConnectionTimeout()
   {
      return this._connectTimeout;
   }

   public void setSessionTimeout(final int timeout)
   {
      this._sessionTimeout = timeout;
   }

   public int getSessionTimeout()
   {
      return this._sessionTimeout;
   }

   @RequiredForInitialization
   public void setPort(final int port)
   {
      this._port = port;
   }

   @RequiredForInitialization
   public void setHost(final String host)
   {
      this._host = host;
   }

   public String getRemoteAddress()
   {
      return this._host.toString() + ":" + this._port;
   }

   public String getHost()
   {
      return this._host;
   }

   public int getPort()
   {
      return this._port;
   }

   @Override
   public String toString()
   {
      return getRemoteAddress();
   }

   public String getIdentification()
   {
      return getRemoteAddress();
   }

   /**
    * The manager that controls connect/disconnects from the remote device
    * 
    * @param _manager
    */
   public void setManager(ConnectorManager _manager)
   {
      this._manager = _manager;
      _manager.setConnection(this);
   }

   public ConnectorManager getManager()
   {
      return this._manager;
   }

   /**
    * Implement this by passing on the request to the connection manager
    */
   public boolean ensureConnected() throws CommunicationIOException,
         CommunicationConnectionException
   {
      return this._manager.ensureConnected();
   }

   public abstract ProtocolMessage getNotification() throws CommunicationInterruptedException;

   public abstract ProtocolMessage getNotification(int timeout)
         throws CommunicationInterruptedException;

   /**
    * Open connection with a remote end point.
    * 
    * Note that initialize() must be called before open().
    * 
    * @throws CommunicationConnectionException
    * @throws CommunicationIOException
    */
   public abstract void connect() throws CommunicationIOException, CommunicationConnectionException;

   /**
    * Is this connection currently open, typically users should not call this, instead just call
    * connect and check the returned value
    */
   public abstract boolean isConnected();

   /** Default behavior is to send a noop message and wait for a response */
   public void sendPing() throws CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, CommunicationIOException, NotConnectedException
   {
      Response response = exchange(new NoopRequest());
      assert response instanceof NoopResponse;
   }

}
