//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 21, 2007
//---------------------

package org.cleversafe.jmx;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.rmi.NoSuchObjectException;
import java.rmi.registry.LocateRegistry;
import java.util.HashMap;
import java.util.Map;

import javax.management.InstanceNotFoundException;
import javax.management.JMException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServerInvocationHandler;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;
import javax.management.remote.rmi.RMIConnectorServer;
import javax.rmi.ssl.SslRMIClientSocketFactory;
import javax.rmi.ssl.SslRMIServerSocketFactory;

import mx4j.tools.naming.NamingServiceMBean;

import org.apache.log4j.Logger;
import org.cleversafe.util.Tuple2;

public class JMXService
{
   private static Logger _logger = Logger.getLogger(JMXService.class);
   
   private static Map<Tuple2<InetAddress,Integer>, JMXService> services = 
      new HashMap<Tuple2<InetAddress,Integer>, JMXService>();
   private static JMXService platformService = null;
   
   // This property is set if the administrator wants the registry to listen on SSL. Since we
   // don't use the built-in registry and connector server (via 1.5 "com.sun.management.jmxremote"
   // or via 1.6 "Attach API", we interpret this parameter ourselves. 
   public static String SSL_REGISTRY_PROPERTY = "com.sun.management.jmxremote.registry.ssl";
   
   
   private static boolean greaterThan5()
   {
      final String version = System.getProperty("java.version");
      if (version == null || version.split("\\.").length != 3)
         throw new RuntimeException("invalid version property format: " + version);
      if (Integer.valueOf(version.split("\\.")[1]) > 5)
         return true;
      else
         return false;
   }
   
   /**
    * Creates new JMXService or returns the JMXService servicing the indicated host and port.
    * If a new JMXService is created it will not use the platform MBeanServer.
    * @param host The RMI Registry host.
    * @param port The RMI Registry port.
    * @return The requested JMXService.
    */
   public static synchronized JMXService 
   getInstance(String host, int port) throws JMException, IOException
   {
      return JMXService.getInstance(host, port, false);
   }
   
   /**
    * Creates a new JMXService or returns the JMXService servicing the indicated host and port.
    * If a new JMXService is created it will use the platform MBeanServer if it is available.
    * If the platform server is not available but was requested an IOException will be thrown.
    * @param host The RMI Registry host.
    * @param port The RMI Registry port.
    * @param platform True is the platform server is required for this service.
    * @return The requested JMXService.
    */
   public static synchronized JMXService
   getInstance(String host, int port, boolean platform) throws JMException, IOException
   {
      final Tuple2<InetAddress,Integer> tuple = 
         new Tuple2<InetAddress,Integer>(InetAddress.getByName(host), port);
      if (services.containsKey(tuple))
      {
         JMXService service = services.get(tuple);
         if (platform && service != platformService)
            throw new IOException("Indicated service does not use platform server");
         return service;
      }
      else
      {
         if (platformService != null)
            throw new IOException("Service using platform server already exists");
         JMXService service = new JMXService(host, port, platform);
         services.put(tuple, service);
         if (platform)
            platformService = service;
         service.start();
         return service;
      }
   }
   
   /**
    * Resets global JMX servers. For use with unit test only.
    */
   protected static void reset() throws IOException
   {
      for (JMXService service : services.values())
      {
         service.stop();
         try
         {
            service.clear();
         }
         catch (Exception e)
         {
            _logger.debug("Clearing service failed");
         }
      }
      services.clear();
      platformService = null;
   }
   
   
   private final JMXConnectorServer server;
   private final SslRMIClientSocketFactory csf;
   private final SslRMIServerSocketFactory ssf;
   
   
   
   
   protected JMXService(String host, int port, boolean platform) throws JMException, IOException
   {
      _logger.debug("Instantiating JMXService");
      
      /*
       * See <http://blogs.sun.com/jmxetc/entry/jmx_connecting_through_firewalls_using> for an
       * explaination of what we are doing here. Basically, we are ensuring that all RMI
       * connections use the same SSL RMI socket factories so that the RMI registry and
       * connector server can be exported on the same port.
       * 
       * "If we didn't use the same factories everywhere, we would have to use at least two
       * ports, because two different RMI socket factories cannot share the same port.
       */
      
      this.csf = new SslRMIClientSocketFactory();
      this.ssf = new SslRMIServerSocketFactory();
      
      final Boolean isSsl = Boolean.valueOf(System.getProperty(SSL_REGISTRY_PROPERTY));
      
      if (isSsl && !greaterThan5())
         throw new JMException("JMX RMI registry configured to use SSL\n" +
               "This feature is not supported with JVM versions earlier than 1.6.0");
         
      
      _logger.debug((isSsl ? "Enabling" : "Disabling") + " SSL for RMI registry");
      if (isSsl)
         LocateRegistry.createRegistry(port, csf, ssf);
      else
         LocateRegistry.createRegistry(port);
      
      
      MBeanServer mbs;
      if (platform)
      {
         mbs = ManagementFactory.getPlatformMBeanServer();
      }
      else
      {
         mbs = MBeanServerFactory.createMBeanServer(); 
      }
      
      // Initialize the environment map
      
      HashMap<String,Object> env = new HashMap<String,Object>();
      
      env.put(RMIConnectorServer.JNDI_REBIND_ATTRIBUTE, "true");
      if (isSsl)
      {
         env.put(RMIConnectorServer.RMI_CLIENT_SOCKET_FACTORY_ATTRIBUTE, csf);
         env.put(RMIConnectorServer.RMI_SERVER_SOCKET_FACTORY_ATTRIBUTE, ssf);
         env.put("com.sun.jndi.rmi.factory.socket", csf);
      }
      
      
      
      // We are now able to use a single port for both the registry and connector server
      JMXServiceURL url =
            new JMXServiceURL("service:jmx:rmi://" + host + ":" + port + "/jndi/rmi://" + host
                  + ":" + port + "/jmxrmi");
      
      this.server = JMXConnectorServerFactory.newJMXConnectorServer(url, env, mbs);
      
      ObjectName serverName = ObjectName.getInstance("connectors:protocol=rmi");
      mbs.registerMBean(this.server, serverName);
      this.server.start();
   }
   
   /**
    * Clears object registrations for testing purposes.
    */
   private void clear() throws MalformedObjectNameException, NullPointerException,
         NoSuchObjectException, InstanceNotFoundException, MBeanRegistrationException
   {
      MBeanServer mbs = this.server.getMBeanServer();
      ObjectName namingName = ObjectName.getInstance("Adaptor:type=rmiregistry");
      NamingServiceMBean naming = 
         (NamingServiceMBean) MBeanServerInvocationHandler.newProxyInstance(
               mbs,
               namingName,
               NamingServiceMBean.class,
               false);
      naming.stop();
      ObjectName serverName = ObjectName.getInstance("connectors:protocol=rmi");
      mbs.unregisterMBean(namingName);
      mbs.unregisterMBean(serverName);
   }
   
   public void start() throws IOException
   {
      this.server.start();
   }
   
   public void stop() throws IOException
   {
      this.server.stop();
      try
      {
         MBeanServer mbs = this.server.getMBeanServer();
         ObjectName serverName = ObjectName.getInstance("connectors:protocol=rmi");
         if (mbs.isRegistered(serverName))
         {
            mbs.unregisterMBean(serverName);   
         }
      }
      catch (MalformedObjectNameException e)
      {
         throw new IOException("could not unregister connector from mbean server");
      }
      catch (InstanceNotFoundException e)
      {
         throw new IOException("could not unregister connector from mbean server");
      }
      catch (MBeanRegistrationException e)
      {
         throw new IOException("could not unregister connector from mbean server");
      }
      catch (NullPointerException e)
      {
         throw new IOException("could not unregister connector from mbean server");
      }
   }
   
   public JMXConnectorServer getJMXConnectorServer()
   {
      return this.server;
   }
   


}


