//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Vance Thornton
//
//---------------------

package org.cleversafe.ida.cauchy;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;


public class CauchyInformationDispersalEncoder
   implements InformationDispersalEncoder
{
   /** Cauchy IDA parameters */
   private Parameters params = null;

   private int messageSize;
   private int fragmentSize;

   private boolean initialized = false;
  
   /** Buffers for unencoded data */
   // private int message[];

   /** Buffer for encoded data */
   // private int fragments[];

   // private List<byte[]> outputBuffers;
   
   /** The total number of slices */
   private int numSlices;
   
   /** The number of slices required to restore */
   private int threshold;

   private boolean useNative = false;
   
   private static Logger logger = Logger.getLogger(CauchyInformationDispersalEncoder.class);
   
   public CauchyInformationDispersalEncoder()
   {
   }
   
   public CauchyInformationDispersalEncoder(int numSlices, int threshold)
   {
      this.numSlices = numSlices;
      this.threshold = threshold;
   }

   public void initialize() throws IDAInvalidParametersException
   {
      this.initialized = false;
      
      if (this.numSlices < 1) 
      {
         throw new IDAInvalidParametersException("Number of slices must be positive");
      }
      if (this.threshold <= 0)
      {
         throw new IDAInvalidParametersException("Threshold must be greater than zero");
      }
      if (this.threshold > this.numSlices) 
      {
         throw new IDAInvalidParametersException("Threshold must be less than or equal to number of slices");
      }
      
      this.params = new Parameters(this.threshold, this.numSlices - this.threshold);
      this.messageSize = params.Plen() * params.Mfragments();;
      this.fragmentSize = params.Plentot();
      useNative = NativeIF.available;
      
      this.initialized = true;
      
      logger.debug( "Slice count = " + numSlices );
      logger.debug( "threshold = " + threshold );
      logger.debug( "Message size: " + this.getMessageSize() );
      logger.debug( "Fragment size: " + this.getFragmentSize() );

      logger.debug( "Blowup = " + ( (float)getFragmentSize() *
         (float)numSlices / (float)getMessageSize()));
      
      logger.debug( "Ideal = " + ((float) numSlices / (float) threshold) );
   }

   public List<byte[]> update(byte buffer[]) 
      throws IDAEncodeException, IDANotInitializedException
   {
      throw new NotImplementedException( "Update not implemented" );
   }

   public List<byte[]> finish(byte buffer[])
      throws IDAEncodeException, IDANotInitializedException
   {
      if (!this.initialized)
      {
         throw new IDANotInitializedException(
               "IDA is not initialized, Call initialize() first");
      }
      
      int message[] = new int[getMessageSize()];

      int fragments[] = new int[getFragmentSize() * numSlices];

      List<byte[]> outputBuffers = new ArrayList<byte[]>();
      for( int idx = 0; idx < numSlices; idx++ )
      {
         outputBuffers.add( new byte[ getFragmentSize() ] );
      }

      int inputPosition = 0;
      int outputPosition = 0;

      // Calculate the size of each output buffer
      int outputSize = 
         ( (buffer.length + 1) / getMessageSize()) * getFragmentSize();

      if ( (buffer.length + 1) % getMessageSize() != 0 )
      {
         outputSize += getFragmentSize();
      }

      // If the output size is greater than the pre-allocated size
      if ( outputSize > getFragmentSize() )
      {
         // Allocate new buffers for output        
         outputBuffers = new ArrayList<byte[]>();   
   
         // Allocate the output buffers
         for (int fragmentIdx = 0; fragmentIdx < getNumSlices(); fragmentIdx++)
         {
            outputBuffers.add(new byte[outputSize]);
         }
      }

      while (outputPosition < outputSize)
      {
         byte fillerByte = 1;
         
         // Copy data from the input buffer into the data buffer

         for( int dataPosition = 0 ; dataPosition < message.length; dataPosition++ )
         {
            if ( inputPosition < buffer.length )
            {
               message[dataPosition] = buffer[inputPosition++];
            } else 
            {
               message[dataPosition] = fillerByte;               
               fillerByte = 0;
            }
         }
         
         

         // Encode the data buffer into the fragments array
         if ( useNative && false)
         {
            byte[] data = new byte[message.length];
            for (int i=0; i<data.length; i++)
            {
               data[i] = (byte) message[i];
            }
            
            byte[] frags = new byte[fragments.length];
            
            NativeIF.cauchy_encode_using_c_helper( 
               data, 
               params.Mfragments(), 
               params.Rfragments(), 
               params.Nsegs(), 
               params.Lfield(), 
               frags );
            
            for (int i=0; i<frags.length; i++)
            {
               fragments[i] = frags[i];
            }
             
         } else 
         {         
            CauchyEncode.encode(fragments, message, params);               
         }

         // For each fragment
         for (int fragmentIdx = 0; fragmentIdx < numSlices; fragmentIdx++)
         {
            byte fragment[] = outputBuffers.get(fragmentIdx);

            int fragmentOffset = fragmentIdx * fragmentSize;

            
            for (int idx = 0; idx < fragmentSize; idx++)
            {              
               fragment[outputPosition + idx] = 
                  (byte) fragments[fragmentOffset + idx];
            }
         }

         outputPosition += fragmentSize;
      }

      //this.initialized = false;
      
      return outputBuffers;
   }

   public int getNumSlices()
   {
      return numSlices;
   }

   public int getThreshold()
   {
      return threshold;
   }
      
   private int getMessageSize()
   {
      return messageSize;
   }

   private int getFragmentSize()
   {
      return fragmentSize;
   }

   public void setNumSlices(int numSlices)
   {
      this.numSlices = numSlices;
   }
   
   public void setThreshold(int threshold)
   {
      this.threshold = threshold;
   }

   public int getChunkSize()
   {
      // TODO Auto-generated method stub
      return 0;
   }

   public void setChunkSize(int ignored)
   {
      // TODO Auto-generated method stub
   }
}
